;; urdf-interface.l
;;
;; this file provides a converter from
;; model-iform (generated by eusmodel-description) to urdf xml
(unless (fboundp 'stl2eus) (load "read-stl.l"))
(load "package://euscollada/src/xml.l")

(defun eusmodel-description->urdf (name root-path description
                                   &key (scale 0.001))
  "convert eusmode-description to urdf sxml"
  (warn "start to converting to urdf sxml~%")
  (let ((links-spec (car description))
        (joints-spec (cdr description)))
    (let ((link-sxml (links-spec->urdf links-spec name root-path scale))
          (joint-sxml (joints-spec->urdf joints-spec links-spec name scale)))
    `(robot
      (@ (name ,name))
      ,@link-sxml
      ,@joint-sxml))))

(defun eus2urdf (obj root-dir           ;you can use package:// convention
                 &key
                 (scale 0.001)
                 (file-suffix ".urdf"))
  "convert an irteus robot model to urdf file."
  (let ((name (send obj :name))
        (full-root-dir (ros::resolve-ros-path root-dir)))
    (unless name
      (error "You need to specify name"))
    (let ((descrpition (eusmodel-description obj)))
      (let ((sxml (eusmodel-description->urdf
                   name root-dir
                   descrpition
                   :scale scale)))
        ;; here we setup filesystem
        (setup-urdf-filesystem name full-root-dir)
        (warn "writing stl...~%")
        ;; write mesh files...
        (let ((mesh-dir (format nil "~A/meshes/~A/"
                                full-root-dir name)))
          (let ((meshes (removen-if #'null (mapcar #'(lambda (x)
                                                      (cdr (assoc :mesh x)))
                                                  (car descrpition))))) ;links
            (dolist (mesh meshes)
              (let ((fname (format nil "~A~A.stl" mesh-dir (send mesh :name))))
                (eus2stl fname mesh :scale scale)))))
        ;; write sxmlirt
        
        (let ((sxml-fname (format nil "~A/urdf/~A.urdf" full-root-dir name)))
          (warn "making xml...~")
          (let ((xml (sxml->xml sxml)))
            (warn "writing sxml...~%")
            (let ((ss (make-string-output-stream)))
              ;;(xml-output-to-string-stream ss xml)
              (with-open-file (f sxml-fname :direction :output)
                (xml-output-to-string-stream f xml)))
                ;;(format f (get-output-stream-string ss))))
            sxml))))))

(defun setup-urdf-filesystem (obj-name base-dir)
  (if (not (eq (probe-file base-dir) :directory))
      (error "~A must be a directory" base-dir)
    (progn
      (unix::mkdir (format nil "~A/meshes" base-dir))
      (unix::mkdir (format nil "~A/meshes/~A" base-dir obj-name))
      (unix::mkdir (format nil "~A/urdf" base-dir))
      (unix::mkdir (format nil "~A/launch" base-dir))
      (unix::mkdir (format nil "~A/config" base-dir))
      (unix::mkdir (format nil "~A/rviz" base-dir)))))

(defun urdf-origin-sxml (origin scale)
  `(origin
    (@
     (xyz
      ,(format nil "~A ~A ~A"
               (* scale (elt (send origin :worldpos) 0))
               (* scale (elt (send origin :worldpos) 1))
               (* scale (elt (send origin :worldpos) 2))))
     (rpy
      ,(format nil "~A ~A ~A"
               (elt (car (send origin :rpy-angle)) 2)
               (elt (car (send origin :rpy-angle)) 1)
               (elt (car (send origin :rpy-angle)) 0))))))

(defun links-spec->urdf (links-spec robot-name root-path scale)
  (mapcar #'(lambda (x) (link-spec->urdf x robot-name root-path scale)) links-spec))

(defun link-spec->urdf (link-spec robot-name root-path scale)
  (let ((name (cdr (assoc :name link-spec)))
        (origin (cdr (assoc :origin link-spec)))
        (mesh (cdr (assoc :mesh link-spec))))
    (let ((visual-sxml
           (if mesh
               `(visual
                 ,(urdf-origin-sxml (make-coords) scale) ;joint has the offset
                 (geometry
                  (mesh
                   (@ (filename ,(urdf-mesh-name mesh root-path robot-name)))))
                 ,(urdf-material-sxml mesh))))
          (collision-sxml
           (if mesh
               `(collision
                 ,(urdf-origin-sxml (make-coords) scale) ;joint has the offset
                 (geometry
                  (mesh
                   (@ (filename ,(urdf-mesh-name mesh root-path robot-name)))))))))
    `(link
      (@ (name ,(format nil "~A/~A" robot-name name)))
      ,visual-sxml
      ,collision-sxml))))

(defun joints-spec->urdf (joints-spec links-spec robot-name scale)
  (mapcar #'(lambda (x) (joint-spec->urdf x links-spec robot-name scale)) joints-spec))

(defun urdf-joint-type (joint-type)
  (case joint-type
    (:rotational-joint
      "revolute")
    (:linear-joint
      "prismatic")))

(defun joint-spec->urdf (joint-spec links-spec robot-name scale)
  (let* ((child-link-name (cdr (assoc :child joint-spec)))
         (child-link (find child-link-name links-spec :test #'string=
                           :key #'(lambda (x) (cdr (assoc :name x))))))
    `(joint
      (@
       (name ,(cdr (assoc :name joint-spec)))
       (type ,(urdf-joint-type (cdr (assoc :type joint-spec)))))
      (axis
       (@
        (xyz ,(format nil "~A ~A ~A"
                      (elt (cdr (assoc :axis joint-spec)) 0)
                      (elt (cdr (assoc :axis joint-spec)) 1)
                      (elt (cdr (assoc :axis joint-spec)) 2)))))
      ;; origin
      ,(if child-link
           (urdf-origin-sxml (cdr (assoc :origin child-link)) scale))
      ,(if (cdr (assoc :parent joint-spec))
           `(parent
             (@ (link ,(format nil "~A/~A" robot-name (cdr (assoc :parent joint-spec)))))))
      ,(if (cdr (assoc :child joint-spec))
           `(child
             (@ (link ,(format nil "~A/~A" robot-name (cdr (assoc :child joint-spec)))))))
      (limit
       (@ (effort 0.0)
          (velocity 0.0)
          (upper ,(cdr (assoc :upper (cdr (assoc :limit joint-spec)))))
          (lower ,(cdr (assoc :lower (cdr (assoc :limit joint-spec))))))))))

(defun urdf-mesh-name (mesh-obj root-path robot-name)
  ;; check mesh-obj has name or not
  ;; if mesh-obj has no name, name it using `robot-name'
  (unless (send mesh-obj :name)
    (send mesh-obj :name (string (gensym robot-name))))
  (format nil "~A/meshes/~A/~A.stl" root-path robot-name (string (send mesh-obj :name))))

(defun urdf-material-sxml (mesh-obj)
  (let ((name (string (gensym "mesh"))))
    (if (find-method mesh-obj :bodies)
        (let ((non-sorted-bodies
               (flatten (mapcar #'(lambda (x)
                                    (if (derivedp x bodyset)
                                        (send x :bodies)
                                      x))
                                (send mesh-obj :bodies)))))
          ;; sort by volume
          (let ((bodies (sort (copy-list non-sorted-bodies)
                              #'>
                              #'(lambda (x)
                                  (send x :volume)))))
            (let ((material
                   (find-if
                    #'(lambda (x)
                        (derivedp x colormaterial))
                    (mapcar #'(lambda (x)
                                (if (derivedp x colormaterial) x
                                  (gl::find-color x))) ;if x is symbol
                            (mapcar #'(lambda (x) (get x :face-color)) bodies)))))
              (if material
                  `(material
                    (@ (name ,name))
                    (color
                     (@ (rgba ,(format nil "~A ~A ~A 1.0"
                                       (elt (send material :diffuse) 0)
                                       (elt (send material :diffuse) 1)
                                       (elt (send material :diffuse) 2))))))
                `(material
                  (@ (name ,name))
                  (color
                   (@ (rgba "0.8 0.8 0.8 1.0"))))))))
      `(material
        (@ (name ,name))
        (color
         (@ (rgba "0.8 0.8 0.8 1.0")))))))

;; (load "../../irteus_models/eusmodels/aibo-robot.l")
