;; eusscene_to_world.l
;; Author: Yuki Furuta <furushchev@jsk.imi.i.u-tokyo.ac.jp>

;; assumed executed at PROJECT_SOURCE_DIR
(defvar *eusurdf-package-dir* (unix:getwd))
(require :conversion-utils (format nil "~A/euslisp/conversion-utils.l" *eusurdf-package-dir*))

(defun write-world-file-header (s)
  (format s "<?xml version=\"1.0\" ?>~%")
  (format s "<!-- This is a source file of world. -->~%")
  (format s "<!-- Do not edit this file. -->~%")
  (format s "<sdf version=\"1.4\">~%")
  (format s "  <world name=\"default\">~%")
  (format s "    <physics type=\"ode\">~%")
  (format s "      <gravity>0 0 -9.81</gravity>~%")
  (format s "      <ode>~%")
  (format s "        <solver>~%")
  (format s "          <type>quick</type>~%")
  (format s "          <iters>50</iters>~%")
  (format s "          <sor>1.4</sor>~%")
  (format s "        </solver>~%")
  (format s "        <constraints>~%")
  (format s "          <cfm>0.0</cfm>~%")
  (format s "          <erp>0.2</erp>~%")
  (format s "          <contact_max_correcting_vel>100.0</contact_max_correcting_vel>~%")
  (format s "          <contact_surface_layer>0.0</contact_surface_layer>~%")
  (format s "        </constraints>~%")
  (format s "      </ode>~%")
  (format s "      <real_time_update_rate>1000</real_time_update_rate>~%")
  (format s "      <max_step_size>0.001</max_step_size>~%")
  (format s "    </physics>~%")
  (format s "    <!-- A global light source -->~%")
  (format s "    <include>~%")
  (format s "      <uri>model://sun</uri>~%")
  (format s "    </include>~%")
  (format s "    <!-- A ground plane -->~%")
  (format s "    <include>~%")
  (format s "      <uri>model://ground_plane</uri>~%")
  (format s "    </include>~%")
  (format s "    <!-- A robot -->~%")
  (format s "    <!-- Objects -->~%"))

(defun write-world-file-footer (s)
  (format s "  </world>~%")
  (format s "</sdf>~%"))

(defun write-empty-world-file (world-file-path)
  (with-open-file (f world-file-path :direction :output :if-exists :new-version)
    (write-world-file-header f)
    (write-world-file-footer f)))

(defun add-model-include (world-file-path model &key (name-suffix "") (uri-suffix "_static"))
  (let ((pos-string (string-join " " (coerce (scale 0.001 (send model :worldpos)) cons)))
        (rpy-string (string-join " " (reverse (coerce (car (rpy-angle (send model :worldrot))) cons)))))
    (unix:system (format nil "sed -i -e \"s@    <!-- Objects -->@    <!-- Objects -->\\n    <include>\\n      <name>~A~A</name>\\n      <uri>model://~A~A</uri>\\n      <pose>~A ~A</pose>\\n    </include>@g\" ~a"
                         (send model :name) name-suffix
                         (send model :name) uri-suffix
                         pos-string rpy-string
                         world-file-path))))

(defun write-model-include (s model &key model-name (suffix "_static") (indent 2))
  (let ((pos-string (string-join " " (coerce (scale 0.001 (send model :worldpos)) cons)))
        (rpy-string (string-join " " (reverse (coerce (car (rpy-angle (send model :worldrot))) cons))))
        (indent-string (make-indent-string indent)))
    (format s "~A<include>~%" indent-string)
    (when model-name
      (format s "~A  <name>~A</name>~%" indent-string model-name))
    (format s "~A  <uri>model://~A~A</uri>~%" indent-string (send model :name)
            (if suffix suffix ""))
    (format s "~A  <pose>~A ~A</pose>~%" indent-string pos-string rpy-string)
    (format s "~A</include>~%" indent-string)))

(defun eusscenefile2world (scene-file-name world-file-name)
  (unless (probe-file scene-file-name)
    (errorf "file ~A not exists" scene-file-name))
  (load scene-file-name)
  (let* ((scene-name (string-join "-"
                                  (butlast
                                   (string-split (send (pathname scene-file-name) :name) #\-))))
         (scene (funcall (read-from-string scene-name))))
    (eusscene2world scene world-file-name)))

(defun eusscene2world (scene world-file-name)
  (let ((model-count (make-hash-table)))
    (make-dirs (send (pathname world-file-name) :directory-string))
    (with-open-file (f world-file-name :direction :output :if-exists :new-version)
      (write-world-file-header f)
      (dolist (model (remove-if-not #'(lambda (x)
                                        (and (subclassp (class x) cascaded-link)
                                             (not (null (send x :links)))))
                                    (send scene :objects)))
        (cond ((null (send model :name)) (errorf "model name must not be empty: ~A" model))
              ((symbolp (send model :name)) (send model :name (string-downcase (send model :name)))))
        (let ((model-name-key (read-from-string (send model :name)))
              model-name)
          (if (gethash model-name-key model-count)
              (incf (gethash model-name-key model-count))
              (setf (gethash model-name-key model-count) 0))
          (setq model-name (string-join "_"
                                        (list
                                         (send model :name)
                                         (gethash model-name-key model-count))))
          (case (use-textured-model-p (send model :name))
            (:fixed
             (write-model-include f model :model-name model-name)) ;; :suffix "_fixed")))
            (:static
             (write-model-include f model :model-name model-name))
            (nil
             (write-model-include f model :model-name model-name))
            (t (write-model-include f model :model-name model-name :suffix nil)))))
      (write-world-file-footer f))))

(when (string= "eusscene_to_world"
               (send (pathname (cadr lisp::*eustop-argument*)) :name))
  ;; eval-when :execute
  (setq scene-file-path (car (last (butlast lisp::*eustop-argument*))))
  (setq world-file-path (car (last lisp::*eustop-argument*)))
  (format t "converting eus scene ~A -> ~A~%" scene-file-path world-file-path)
  (eusscenefile2world scene-file-path world-file-path)
  (exit))
(provide :eusscene_to_world)
