#pragma once

#include <vamp/vector.hh>
#include <vamp/vector/math.hh>
#include <vamp/collision/environment.hh>
#include <vamp/collision/validity.hh>

// NOLINTBEGIN(*-magic-numbers)
namespace vamp::robots
{
    struct UR5
    {
        static constexpr char *name = "ur5";
        static constexpr std::size_t dimension = 6;
        static constexpr std::size_t n_spheres = 40;
        static constexpr float min_radius = 0.014999999664723873;
        static constexpr float max_radius = 0.07999999821186066;
        static constexpr std::size_t resolution = 32;

        static constexpr std::array<std::string_view, dimension> joint_names = {
            "shoulder_pan_joint",
            "shoulder_lift_joint",
            "elbow_joint",
            "wrist_1_joint",
            "wrist_2_joint",
            "wrist_3_joint"};
        static constexpr char *end_effector = "robotiq_85_base_link";

        using Configuration = FloatVector<dimension>;
        using ConfigurationArray = std::array<FloatT, dimension>;

        struct alignas(FloatVectorAlignment) ConfigurationBuffer
          : std::array<float, Configuration::num_scalars_rounded>
        {
        };

        template <std::size_t rake>
        using ConfigurationBlock = FloatVector<rake, dimension>;

        template <std::size_t rake>
        struct Spheres
        {
            FloatVector<rake, n_spheres> x;
            FloatVector<rake, n_spheres> y;
            FloatVector<rake, n_spheres> z;
            FloatVector<rake, n_spheres> r;
        };

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> s_m{
            6.2831854820251465,
            6.2831854820251465,
            6.2831854820251465,
            6.2831854820251465,
            6.2831854820251465,
            6.2831854820251465};

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> s_a{
            -3.1415927410125732,
            -3.1415927410125732,
            -3.1415927410125732,
            -3.1415927410125732,
            -3.1415927410125732,
            -3.1415927410125732};

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> d_m{
            0.15915493667125702,
            0.15915493667125702,
            0.15915493667125702,
            0.15915493667125702,
            0.15915493667125702,
            0.15915493667125702};

        inline static void scale_configuration(Configuration &q) noexcept
        {
            q = q * Configuration(s_m) + Configuration(s_a);
        }

        inline static void descale_configuration(Configuration &q) noexcept
        {
            q = (q - Configuration(s_a)) * Configuration(d_m);
        }

        template <std::size_t rake>
        inline static void scale_configuration_block(ConfigurationBlock<rake> &q) noexcept
        {
            q[0] = -3.1415927410125732 + (q[0] * 6.2831854820251465);
            q[1] = -3.1415927410125732 + (q[1] * 6.2831854820251465);
            q[2] = -3.1415927410125732 + (q[2] * 6.2831854820251465);
            q[3] = -3.1415927410125732 + (q[3] * 6.2831854820251465);
            q[4] = -3.1415927410125732 + (q[4] * 6.2831854820251465);
            q[5] = -3.1415927410125732 + (q[5] * 6.2831854820251465);
        }

        template <std::size_t rake>
        inline static void descale_configuration_block(ConfigurationBlock<rake> &q) noexcept
        {
            q[0] = 0.15915493667125702 * (q[0] - -3.1415927410125732);
            q[1] = 0.15915493667125702 * (q[1] - -3.1415927410125732);
            q[2] = 0.15915493667125702 * (q[2] - -3.1415927410125732);
            q[3] = 0.15915493667125702 * (q[3] - -3.1415927410125732);
            q[4] = 0.15915493667125702 * (q[4] - -3.1415927410125732);
            q[5] = 0.15915493667125702 * (q[5] - -3.1415927410125732);
        }

        inline static auto space_measure() noexcept -> float
        {
            return 61528.90796697732;
        }

        template <std::size_t rake>
        inline static void sphere_fk(const ConfigurationBlock<rake> &x, Spheres<rake> &out) noexcept
        {
            std::array<FloatVector<rake, 1>, 19> v;
            std::array<FloatVector<rake, 1>, 160> y;

            v[0] = cos(x[0]);
            v[1] = sin(x[0]);
            v[2] = 0.000796326710733264 * v[0] + -0.999999682931835 * v[1];
            v[3] = sin(x[1]);
            v[4] = cos(x[1]);
            v[5] = 1.79489656471077e-09 * v[3] + v[4];
            v[6] = v[2] * v[5];
            v[7] = -v[1];
            v[8] = 0.000796326710733264 * v[7] + -0.999999682931835 * v[0];
            y[24] = 0.13585 * v[8];
            y[8] = 0.105 * v[6] + y[24];
            v[1] = 0.999999682931835 * v[0] + 0.000796326710733264 * v[1];
            v[5] = v[1] * v[5];
            v[7] = 0.999999682931835 * v[7] + 0.000796326710733264 * v[0];
            y[25] = 0.13585 * v[7];
            y[9] = 0.105 * v[5] + y[25];
            v[0] = -1. * v[3] + 1.79489656471077e-09 * v[4];
            y[10] = 1.003559 + 0.105 * v[0];
            y[12] = 0.21 * v[6] + y[24];
            y[13] = 0.21 * v[5] + y[25];
            y[14] = 1.003559 + 0.21 * v[0];
            y[16] = 0.315 * v[6] + y[24];
            y[17] = 0.315 * v[5] + y[25];
            y[18] = 1.003559 + 0.315 * v[0];
            y[20] = 0.42 * v[6] + y[24];
            y[21] = 0.42 * v[5] + y[25];
            y[22] = 1.003559 + 0.42 * v[0];
            y[28] = y[24] + -0.1197 * v[8] + 0.425 * v[6];
            y[29] = y[25] + -0.1197 * v[7] + 0.425 * v[5];
            y[30] = 1.003559 + 0.425 * v[0];
            v[3] = -v[3];
            v[9] = 1.79489656471077e-09 * v[4] + v[3];
            v[2] = v[2] * v[9];
            v[10] = sin(x[2]);
            v[11] = cos(x[2]);
            v[12] = v[2] * v[10] + v[6] * v[11];
            y[32] = 0.1 * v[12] + y[28];
            v[9] = v[1] * v[9];
            v[1] = v[9] * v[10] + v[5] * v[11];
            y[33] = 0.1 * v[1] + y[29];
            v[3] = -1. * v[4] + 1.79489656471077e-09 * v[3];
            v[4] = v[3] * v[10] + v[0] * v[11];
            y[34] = 0.1 * v[4] + y[30];
            y[36] = 0.14 * v[12] + y[28];
            y[37] = 0.14 * v[1] + y[29];
            y[38] = 0.14 * v[4] + y[30];
            y[40] = 0.18 * v[12] + y[28];
            y[41] = 0.18 * v[1] + y[29];
            y[42] = 0.18 * v[4] + y[30];
            y[44] = 0.22 * v[12] + y[28];
            y[45] = 0.22 * v[1] + y[29];
            y[46] = 0.22 * v[4] + y[30];
            y[48] = 0.26 * v[12] + y[28];
            y[49] = 0.26 * v[1] + y[29];
            y[50] = 0.26 * v[4] + y[30];
            y[52] = 0.3 * v[12] + y[28];
            y[53] = 0.3 * v[1] + y[29];
            y[54] = 0.3 * v[4] + y[30];
            y[56] = 0.34 * v[12] + y[28];
            y[57] = 0.34 * v[1] + y[29];
            y[58] = 0.34 * v[4] + y[30];
            y[60] = 0.38 * v[12] + y[28];
            y[61] = 0.38 * v[1] + y[29];
            y[62] = 0.38 * v[4] + y[30];
            v[10] = -v[10];
            v[2] = v[2] * v[11] + v[6] * v[10];
            v[6] = sin(x[3]);
            v[13] = cos(x[3]);
            v[14] = 1.79489656471077e-09 * v[6] + v[13];
            v[15] = -1. * v[6] + 1.79489656471077e-09 * v[13];
            v[16] = v[2] * v[14] + v[12] * v[15];
            v[17] = y[28] + 0.39225 * v[12];
            y[64] = 0.09 * v[8] + 0.03 * v[16] + v[17];
            v[9] = v[9] * v[11] + v[5] * v[10];
            v[5] = v[9] * v[14] + v[1] * v[15];
            v[18] = y[29] + 0.39225 * v[1];
            y[65] = 0.09 * v[7] + 0.03 * v[5] + v[18];
            v[10] = v[3] * v[11] + v[0] * v[10];
            v[15] = v[10] * v[14] + v[4] * v[15];
            y[74] = y[30] + 0.39225 * v[4];
            y[66] = 0.03 * v[15] + y[74];
            y[68] = 0.09 * v[8] + -0.03 * v[16] + v[17];
            y[69] = 0.09 * v[7] + -0.03 * v[5] + v[18];
            y[70] = -0.03 * v[15] + y[74];
            y[72] = 0.09 * v[8] + v[17];
            y[73] = 0.09 * v[7] + v[18];
            v[6] = -v[6];
            v[14] = 1.79489656471077e-09 * v[13] + v[6];
            v[6] = -1. * v[13] + 1.79489656471077e-09 * v[6];
            v[2] = v[2] * v[14] + v[12] * v[6];
            v[12] = sin(x[4]);
            v[13] = -v[12];
            v[3] = cos(x[4]);
            v[11] = v[2] * v[13] + v[8] * v[3];
            v[17] = v[17] + 0.093 * v[8];
            y[76] = 0.03 * v[11] + 0.09 * v[16] + v[17];
            v[9] = v[9] * v[14] + v[1] * v[6];
            v[1] = v[9] * v[13] + v[7] * v[3];
            v[18] = v[18] + 0.093 * v[7];
            y[77] = 0.03 * v[1] + 0.09 * v[5] + v[18];
            v[6] = v[10] * v[14] + v[4] * v[6];
            v[13] = v[6] * v[13];
            y[78] = 0.03 * v[13] + 0.09 * v[15] + y[74];
            y[80] = -0.03 * v[11] + 0.09 * v[16] + v[17];
            y[81] = -0.03 * v[1] + 0.09 * v[5] + v[18];
            y[82] = -0.03 * v[13] + 0.09 * v[15] + y[74];
            y[84] = 0.09 * v[16] + v[17];
            y[85] = 0.09 * v[5] + v[18];
            y[86] = 0.09 * v[15] + y[74];
            v[17] = v[17] + 0.09465 * v[16];
            y[88] = 0.06 * v[11] + v[17];
            v[18] = v[18] + 0.09465 * v[5];
            y[89] = 0.06 * v[1] + v[18];
            v[14] = y[74] + 0.09465 * v[15];
            y[90] = 0.06 * v[13] + v[14];
            v[2] = v[2] * v[3] + v[8] * v[12];
            v[8] = cos(x[5]);
            v[10] = sin(x[5]);
            v[4] = -v[10];
            v[0] = v[2] * v[8] + v[16] * v[4];
            y[92] = 1.59265611381251e-05 * v[0] + 0.0973000063413347 * v[11] + v[17];
            v[9] = v[9] * v[3] + v[7] * v[12];
            v[12] = v[9] * v[8] + v[5] * v[4];
            y[93] = 1.59265611381251e-05 * v[12] + 0.0973000063413347 * v[1] + v[18];
            v[6] = v[6] * v[3];
            v[4] = v[6] * v[8] + v[15] * v[4];
            y[94] = 1.59265611381251e-05 * v[4] + 0.0973000063413347 * v[13] + v[14];
            v[2] = v[2] * v[10] + v[16] * v[8];
            y[96] = -4.77794169794995e-05 * v[0] + 0.177299961951912 * v[11] + 0.000547779602643996 * v[2] +
                    v[17];
            v[9] = v[9] * v[10] + v[5] * v[8];
            y[97] = -4.77794169794995e-05 * v[12] + 0.177299961951912 * v[1] + 0.000547779602643996 * v[9] +
                    v[18];
            v[6] = v[6] * v[10] + v[15] * v[8];
            y[98] = -4.77794169794995e-05 * v[4] + 0.177299961951912 * v[13] + 0.000547779602643996 * v[6] +
                    v[14];
            y[100] =
                -1.592643044558e-05 * v[0] + 0.137299987317304 * v[11] + 0.000515926534214666 * v[2] + v[17];
            y[101] =
                -1.592643044558e-05 * v[12] + 0.137299987317304 * v[1] + 0.000515926534214666 * v[9] + v[18];
            y[102] =
                -1.592643044558e-05 * v[4] + 0.137299987317304 * v[13] + 0.000515926534214666 * v[6] + v[14];
            y[104] =
                0.0326288719300173 * v[0] + 0.206633483191718 * v[11] + 0.000571117947718783 * v[2] + v[17];
            y[105] =
                0.0326288719300173 * v[12] + 0.206633483191718 * v[1] + 0.000571117947718783 * v[9] + v[18];
            y[106] =
                0.0326288719300173 * v[4] + 0.206633483191718 * v[13] + 0.000571117947718783 * v[6] + v[14];
            y[108] =
                0.0471739661448351 * v[0] + 0.257142085198042 * v[11] + 0.000611330073064923 * v[2] + v[17];
            y[109] =
                0.0471739661448351 * v[12] + 0.257142085198042 * v[1] + 0.000611330073064923 * v[9] + v[18];
            y[110] =
                0.0471739661448351 * v[4] + 0.257142085198042 * v[13] + 0.000611330073064923 * v[6] + v[14];
            y[112] =
                0.0471938742614188 * v[0] + 0.232142101051412 * v[11] + 0.000591421905296592 * v[2] + v[17];
            y[113] =
                0.0471938742614188 * v[12] + 0.232142101051412 * v[1] + 0.000591421905296592 * v[9] + v[18];
            y[114] =
                0.0471938742614188 * v[4] + 0.232142101051412 * v[13] + 0.000591421905296592 * v[6] + v[14];
            y[116] =
                0.0305511319550535 * v[0] + 0.180116344904696 * v[11] + 0.000550002959776232 * v[2] + v[17];
            y[117] =
                0.0305511319550535 * v[12] + 0.180116344904696 * v[1] + 0.000550002959776232 * v[9] + v[18];
            y[118] =
                0.0305511319550535 * v[4] + 0.180116344904696 * v[13] + 0.000550002959776232 * v[6] + v[14];
            y[120] =
                0.0622118532100359 * v[0] + 0.218207593376579 * v[11] + 0.000580315961208078 * v[2] + v[17];
            y[121] =
                0.0622118532100359 * v[12] + 0.218207593376579 * v[1] + 0.000580315961208078 * v[9] + v[18];
            y[122] =
                0.0622118532100359 * v[4] + 0.218207593376579 * v[13] + 0.000580315961208078 * v[6] + v[14];
            y[124] =
                0.0622437061965698 * v[0] + 0.178207618741971 * v[11] + 0.000548462892778747 * v[2] + v[17];
            y[125] =
                0.0622437061965698 * v[12] + 0.178207618741971 * v[1] + 0.000548462892778747 * v[9] + v[18];
            y[126] =
                0.0622437061965698 * v[4] + 0.178207618741971 * v[13] + 0.000548462892778747 * v[6] + v[14];
            y[128] =
                0.0622277797033028 * v[0] + 0.198207606059275 * v[11] + 0.000564389426993413 * v[2] + v[17];
            y[129] =
                0.0622277797033028 * v[12] + 0.198207606059275 * v[1] + 0.000564389426993413 * v[9] + v[18];
            y[130] =
                0.0622277797033028 * v[4] + 0.198207606059275 * v[13] + 0.000564389426993413 * v[6] + v[14];
            y[132] =
                -0.0327711073338182 * v[0] + 0.206581403542295 * v[11] + 0.000571117947718776 * v[2] + v[17];
            y[133] =
                -0.0327711073338182 * v[12] + 0.206581403542295 * v[1] + 0.000571117947718776 * v[9] + v[18];
            y[134] =
                -0.0327711073338182 * v[4] + 0.206581403542295 * v[13] + 0.000571117947718776 * v[6] + v[14];
            y[136] =
                -0.0469176493992621 * v[0] + 0.257427526148899 * v[11] + 0.000611617044105126 * v[2] + v[17];
            y[137] =
                -0.0469176493992621 * v[12] + 0.257427526148899 * v[1] + 0.000611617044105126 * v[9] + v[18];
            y[138] =
                -0.0469176493992621 * v[4] + 0.257427526148899 * v[13] + 0.000611617044105126 * v[6] + v[14];
            y[140] =
                -0.0468977412826784 * v[0] + 0.232427542002269 * v[11] + 0.000591708876336795 * v[2] + v[17];
            y[141] =
                -0.0468977412826784 * v[12] + 0.232427542002269 * v[1] + 0.000591708876336795 * v[9] + v[18];
            y[142] =
                -0.0468977412826784 * v[4] + 0.232427542002269 * v[13] + 0.000591708876336795 * v[6] + v[14];
            y[144] =
                -0.0306511374916797 * v[0] + 0.180067607997177 * v[11] + 0.000550002959776232 * v[2] + v[17];
            y[145] =
                -0.0306511374916797 * v[12] + 0.180067607997177 * v[1] + 0.000550002959776232 * v[9] + v[18];
            y[146] =
                -0.0306511374916797 * v[4] + 0.180067607997177 * v[13] + 0.000550002959776232 * v[6] + v[14];
            y[148] =
                -0.0623912961612096 * v[0] + 0.218440976014459 * v[11] + 0.000580580825254713 * v[2] + v[17];
            y[149] =
                -0.0623912961612096 * v[12] + 0.218440976014459 * v[1] + 0.000580580825254713 * v[9] + v[18];
            y[150] =
                -0.0623912961612096 * v[4] + 0.218440976014459 * v[13] + 0.000580580825254713 * v[6] + v[14];
            y[152] =
                -0.0623594431746757 * v[0] + 0.178441001379851 * v[11] + 0.000548727756825382 * v[2] + v[17];
            y[153] =
                -0.0623594431746757 * v[12] + 0.178441001379851 * v[1] + 0.000548727756825382 * v[9] + v[18];
            y[154] =
                -0.0623594431746757 * v[4] + 0.178441001379851 * v[13] + 0.000548727756825382 * v[6] + v[14];
            y[156] =
                -0.0623753696679426 * v[0] + 0.198440988697155 * v[11] + 0.000564654291040047 * v[2] + v[17];
            y[157] =
                -0.0623753696679426 * v[12] + 0.198440988697155 * v[1] + 0.000564654291040047 * v[9] + v[18];
            y[158] =
                -0.0623753696679426 * v[4] + 0.198440988697155 * v[13] + 0.000564654291040047 * v[6] + v[14];
            // dependent variables without operations
            y[0] = 0.;
            y[1] = 0.;
            y[2] = 0.9144;
            y[3] = 0.0799999982118607;
            y[4] = 0.;
            y[5] = 0.;
            y[6] = 1.003559;
            y[7] = 0.0799999982118607;
            y[11] = 0.0799999982118607;
            y[15] = 0.0799999982118607;
            y[19] = 0.0799999982118607;
            y[23] = 0.0799999982118607;
            y[26] = 1.003559;
            y[27] = 0.0799999982118607;
            y[31] = 0.0799999982118607;
            y[35] = 0.0399999991059303;
            y[39] = 0.0399999991059303;
            y[43] = 0.0399999991059303;
            y[47] = 0.0399999991059303;
            y[51] = 0.0399999991059303;
            y[55] = 0.0399999991059303;
            y[59] = 0.0399999991059303;
            y[63] = 0.0399999991059303;
            y[67] = 0.0399999991059303;
            y[71] = 0.0399999991059303;
            y[75] = 0.0399999991059303;
            y[79] = 0.0399999991059303;
            y[83] = 0.0399999991059303;
            y[87] = 0.0399999991059303;
            y[91] = 0.0399999991059303;
            y[95] = 0.0399999991059303;
            y[99] = 0.0399999991059303;
            y[103] = 0.0399999991059303;
            y[107] = 0.0199999995529652;
            y[111] = 0.0149999996647239;
            y[115] = 0.0149999996647239;
            y[119] = 0.0199999995529652;
            y[123] = 0.0149999996647239;
            y[127] = 0.0149999996647239;
            y[131] = 0.0149999996647239;
            y[135] = 0.0199999995529652;
            y[139] = 0.0149999996647239;
            y[143] = 0.0149999996647239;
            y[147] = 0.0199999995529652;
            y[151] = 0.0149999996647239;
            y[155] = 0.0149999996647239;
            y[159] = 0.0149999996647239;

            for (auto i = 0U; i < 40; ++i)
            {
                out.x[i] = y[i * 4 + 0];
                out.y[i] = y[i * 4 + 1];
                out.z[i] = y[i * 4 + 2];
                out.r[i] = y[i * 4 + 3];
            }
        }

        using Debug = std::
            pair<std::vector<std::vector<std::string>>, std::vector<std::pair<std::size_t, std::size_t>>>;

        template <std::size_t rake>
        inline static auto fkcc_debug(
            const vamp::collision::Environment<FloatVector<rake>> &environment,
            const ConfigurationBlock<rake> &x) noexcept -> Debug
        {
            std::array<FloatVector<rake, 1>, 30> v;
            std::array<FloatVector<rake, 1>, 228> y;

            v[0] = cos(x[0]);
            v[1] = sin(x[0]);
            v[2] = 0.000796326710733264 * v[0] + -0.999999682931835 * v[1];
            v[3] = sin(x[1]);
            v[4] = cos(x[1]);
            v[5] = 1.79489656471077e-09 * v[3] + v[4];
            v[6] = v[2] * v[5];
            v[7] = -v[1];
            v[8] = 0.000796326710733264 * v[7] + -0.999999682931835 * v[0];
            y[24] = 0.13585 * v[8];
            y[8] = 0.105 * v[6] + y[24];
            v[1] = 0.999999682931835 * v[0] + 0.000796326710733264 * v[1];
            v[5] = v[1] * v[5];
            v[7] = 0.999999682931835 * v[7] + 0.000796326710733264 * v[0];
            y[25] = 0.13585 * v[7];
            y[9] = 0.105 * v[5] + y[25];
            v[0] = -1. * v[3] + 1.79489656471077e-09 * v[4];
            y[10] = 1.003559 + 0.105 * v[0];
            y[12] = 0.21 * v[6] + y[24];
            y[13] = 0.21 * v[5] + y[25];
            y[14] = 1.003559 + 0.21 * v[0];
            y[16] = 0.315 * v[6] + y[24];
            y[17] = 0.315 * v[5] + y[25];
            y[18] = 1.003559 + 0.315 * v[0];
            y[20] = 0.42 * v[6] + y[24];
            y[21] = 0.42 * v[5] + y[25];
            y[22] = 1.003559 + 0.42 * v[0];
            y[28] = y[24] + -0.1197 * v[8] + 0.425 * v[6];
            y[29] = y[25] + -0.1197 * v[7] + 0.425 * v[5];
            y[30] = 1.003559 + 0.425 * v[0];
            v[3] = -v[3];
            v[9] = 1.79489656471077e-09 * v[4] + v[3];
            v[2] = v[2] * v[9];
            v[10] = sin(x[2]);
            v[11] = cos(x[2]);
            v[12] = v[2] * v[10] + v[6] * v[11];
            y[32] = 0.1 * v[12] + y[28];
            v[9] = v[1] * v[9];
            v[1] = v[9] * v[10] + v[5] * v[11];
            y[33] = 0.1 * v[1] + y[29];
            v[3] = -1. * v[4] + 1.79489656471077e-09 * v[3];
            v[4] = v[3] * v[10] + v[0] * v[11];
            y[34] = 0.1 * v[4] + y[30];
            y[36] = 0.14 * v[12] + y[28];
            y[37] = 0.14 * v[1] + y[29];
            y[38] = 0.14 * v[4] + y[30];
            y[40] = 0.18 * v[12] + y[28];
            y[41] = 0.18 * v[1] + y[29];
            y[42] = 0.18 * v[4] + y[30];
            y[44] = 0.22 * v[12] + y[28];
            y[45] = 0.22 * v[1] + y[29];
            y[46] = 0.22 * v[4] + y[30];
            y[48] = 0.26 * v[12] + y[28];
            y[49] = 0.26 * v[1] + y[29];
            y[50] = 0.26 * v[4] + y[30];
            y[52] = 0.3 * v[12] + y[28];
            y[53] = 0.3 * v[1] + y[29];
            y[54] = 0.3 * v[4] + y[30];
            y[56] = 0.34 * v[12] + y[28];
            y[57] = 0.34 * v[1] + y[29];
            y[58] = 0.34 * v[4] + y[30];
            y[60] = 0.38 * v[12] + y[28];
            y[61] = 0.38 * v[1] + y[29];
            y[62] = 0.38 * v[4] + y[30];
            v[10] = -v[10];
            v[2] = v[2] * v[11] + v[6] * v[10];
            v[13] = sin(x[3]);
            v[14] = cos(x[3]);
            v[15] = 1.79489656471077e-09 * v[13] + v[14];
            v[16] = -1. * v[13] + 1.79489656471077e-09 * v[14];
            v[17] = v[2] * v[15] + v[12] * v[16];
            v[18] = y[28] + 0.39225 * v[12];
            y[64] = 0.09 * v[8] + 0.03 * v[17] + v[18];
            v[9] = v[9] * v[11] + v[5] * v[10];
            v[19] = v[9] * v[15] + v[1] * v[16];
            v[20] = y[29] + 0.39225 * v[1];
            y[65] = 0.09 * v[7] + 0.03 * v[19] + v[20];
            v[10] = v[3] * v[11] + v[0] * v[10];
            v[16] = v[10] * v[15] + v[4] * v[16];
            y[74] = y[30] + 0.39225 * v[4];
            y[66] = 0.03 * v[16] + y[74];
            y[68] = 0.09 * v[8] + -0.03 * v[17] + v[18];
            y[69] = 0.09 * v[7] + -0.03 * v[19] + v[20];
            y[70] = -0.03 * v[16] + y[74];
            y[72] = 0.09 * v[8] + v[18];
            y[73] = 0.09 * v[7] + v[20];
            v[13] = -v[13];
            v[15] = 1.79489656471077e-09 * v[14] + v[13];
            v[13] = -1. * v[14] + 1.79489656471077e-09 * v[13];
            v[2] = v[2] * v[15] + v[12] * v[13];
            v[14] = sin(x[4]);
            v[3] = -v[14];
            v[11] = cos(x[4]);
            v[21] = v[2] * v[3] + v[8] * v[11];
            v[22] = v[18] + 0.093 * v[8];
            y[76] = 0.03 * v[21] + 0.09 * v[17] + v[22];
            v[9] = v[9] * v[15] + v[1] * v[13];
            v[23] = v[9] * v[3] + v[7] * v[11];
            v[24] = v[20] + 0.093 * v[7];
            y[77] = 0.03 * v[23] + 0.09 * v[19] + v[24];
            v[13] = v[10] * v[15] + v[4] * v[13];
            v[3] = v[13] * v[3];
            y[78] = 0.03 * v[3] + 0.09 * v[16] + y[74];
            y[80] = -0.03 * v[21] + 0.09 * v[17] + v[22];
            y[81] = -0.03 * v[23] + 0.09 * v[19] + v[24];
            y[82] = -0.03 * v[3] + 0.09 * v[16] + y[74];
            y[84] = 0.09 * v[17] + v[22];
            y[85] = 0.09 * v[19] + v[24];
            y[86] = 0.09 * v[16] + y[74];
            v[15] = v[22] + 0.09465 * v[17];
            y[88] = 0.06 * v[21] + v[15];
            v[10] = v[24] + 0.09465 * v[19];
            y[89] = 0.06 * v[23] + v[10];
            v[25] = y[74] + 0.09465 * v[16];
            y[90] = 0.06 * v[3] + v[25];
            v[2] = v[2] * v[11] + v[8] * v[14];
            v[26] = cos(x[5]);
            v[27] = sin(x[5]);
            v[28] = -v[27];
            v[29] = v[2] * v[26] + v[17] * v[28];
            y[92] = 1.59265611381251e-05 * v[29] + 0.0973000063413347 * v[21] + v[15];
            v[9] = v[9] * v[11] + v[7] * v[14];
            v[14] = v[9] * v[26] + v[19] * v[28];
            y[93] = 1.59265611381251e-05 * v[14] + 0.0973000063413347 * v[23] + v[10];
            v[13] = v[13] * v[11];
            v[28] = v[13] * v[26] + v[16] * v[28];
            y[94] = 1.59265611381251e-05 * v[28] + 0.0973000063413347 * v[3] + v[25];
            v[2] = v[2] * v[27] + v[17] * v[26];
            y[96] = -4.77794169794995e-05 * v[29] + 0.177299961951912 * v[21] + 0.000547779602643996 * v[2] +
                    v[15];
            v[9] = v[9] * v[27] + v[19] * v[26];
            y[97] = -4.77794169794995e-05 * v[14] + 0.177299961951912 * v[23] + 0.000547779602643996 * v[9] +
                    v[10];
            v[13] = v[13] * v[27] + v[16] * v[26];
            y[98] = -4.77794169794995e-05 * v[28] + 0.177299961951912 * v[3] + 0.000547779602643996 * v[13] +
                    v[25];
            y[100] =
                -1.592643044558e-05 * v[29] + 0.137299987317304 * v[21] + 0.000515926534214666 * v[2] + v[15];
            y[101] =
                -1.592643044558e-05 * v[14] + 0.137299987317304 * v[23] + 0.000515926534214666 * v[9] + v[10];
            y[102] =
                -1.592643044558e-05 * v[28] + 0.137299987317304 * v[3] + 0.000515926534214666 * v[13] + v[25];
            y[104] =
                0.0326288719300173 * v[29] + 0.206633483191718 * v[21] + 0.000571117947718783 * v[2] + v[15];
            y[105] =
                0.0326288719300173 * v[14] + 0.206633483191718 * v[23] + 0.000571117947718783 * v[9] + v[10];
            y[106] =
                0.0326288719300173 * v[28] + 0.206633483191718 * v[3] + 0.000571117947718783 * v[13] + v[25];
            y[108] =
                0.0471739661448351 * v[29] + 0.257142085198042 * v[21] + 0.000611330073064923 * v[2] + v[15];
            y[109] =
                0.0471739661448351 * v[14] + 0.257142085198042 * v[23] + 0.000611330073064923 * v[9] + v[10];
            y[110] =
                0.0471739661448351 * v[28] + 0.257142085198042 * v[3] + 0.000611330073064923 * v[13] + v[25];
            y[112] =
                0.0471938742614188 * v[29] + 0.232142101051412 * v[21] + 0.000591421905296592 * v[2] + v[15];
            y[113] =
                0.0471938742614188 * v[14] + 0.232142101051412 * v[23] + 0.000591421905296592 * v[9] + v[10];
            y[114] =
                0.0471938742614188 * v[28] + 0.232142101051412 * v[3] + 0.000591421905296592 * v[13] + v[25];
            y[116] =
                0.0305511319550535 * v[29] + 0.180116344904696 * v[21] + 0.000550002959776232 * v[2] + v[15];
            y[117] =
                0.0305511319550535 * v[14] + 0.180116344904696 * v[23] + 0.000550002959776232 * v[9] + v[10];
            y[118] =
                0.0305511319550535 * v[28] + 0.180116344904696 * v[3] + 0.000550002959776232 * v[13] + v[25];
            y[120] =
                0.0622118532100359 * v[29] + 0.218207593376579 * v[21] + 0.000580315961208078 * v[2] + v[15];
            y[121] =
                0.0622118532100359 * v[14] + 0.218207593376579 * v[23] + 0.000580315961208078 * v[9] + v[10];
            y[122] =
                0.0622118532100359 * v[28] + 0.218207593376579 * v[3] + 0.000580315961208078 * v[13] + v[25];
            y[124] =
                0.0622437061965698 * v[29] + 0.178207618741971 * v[21] + 0.000548462892778747 * v[2] + v[15];
            y[125] =
                0.0622437061965698 * v[14] + 0.178207618741971 * v[23] + 0.000548462892778747 * v[9] + v[10];
            y[126] =
                0.0622437061965698 * v[28] + 0.178207618741971 * v[3] + 0.000548462892778747 * v[13] + v[25];
            y[128] =
                0.0622277797033028 * v[29] + 0.198207606059275 * v[21] + 0.000564389426993413 * v[2] + v[15];
            y[129] =
                0.0622277797033028 * v[14] + 0.198207606059275 * v[23] + 0.000564389426993413 * v[9] + v[10];
            y[130] =
                0.0622277797033028 * v[28] + 0.198207606059275 * v[3] + 0.000564389426993413 * v[13] + v[25];
            y[132] =
                -0.0327711073338182 * v[29] + 0.206581403542295 * v[21] + 0.000571117947718776 * v[2] + v[15];
            y[133] =
                -0.0327711073338182 * v[14] + 0.206581403542295 * v[23] + 0.000571117947718776 * v[9] + v[10];
            y[134] =
                -0.0327711073338182 * v[28] + 0.206581403542295 * v[3] + 0.000571117947718776 * v[13] + v[25];
            y[136] =
                -0.0469176493992621 * v[29] + 0.257427526148899 * v[21] + 0.000611617044105126 * v[2] + v[15];
            y[137] =
                -0.0469176493992621 * v[14] + 0.257427526148899 * v[23] + 0.000611617044105126 * v[9] + v[10];
            y[138] =
                -0.0469176493992621 * v[28] + 0.257427526148899 * v[3] + 0.000611617044105126 * v[13] + v[25];
            y[140] =
                -0.0468977412826784 * v[29] + 0.232427542002269 * v[21] + 0.000591708876336795 * v[2] + v[15];
            y[141] =
                -0.0468977412826784 * v[14] + 0.232427542002269 * v[23] + 0.000591708876336795 * v[9] + v[10];
            y[142] =
                -0.0468977412826784 * v[28] + 0.232427542002269 * v[3] + 0.000591708876336795 * v[13] + v[25];
            y[144] =
                -0.0306511374916797 * v[29] + 0.180067607997177 * v[21] + 0.000550002959776232 * v[2] + v[15];
            y[145] =
                -0.0306511374916797 * v[14] + 0.180067607997177 * v[23] + 0.000550002959776232 * v[9] + v[10];
            y[146] =
                -0.0306511374916797 * v[28] + 0.180067607997177 * v[3] + 0.000550002959776232 * v[13] + v[25];
            y[148] =
                -0.0623912961612096 * v[29] + 0.218440976014459 * v[21] + 0.000580580825254713 * v[2] + v[15];
            y[149] =
                -0.0623912961612096 * v[14] + 0.218440976014459 * v[23] + 0.000580580825254713 * v[9] + v[10];
            y[150] =
                -0.0623912961612096 * v[28] + 0.218440976014459 * v[3] + 0.000580580825254713 * v[13] + v[25];
            y[152] =
                -0.0623594431746757 * v[29] + 0.178441001379851 * v[21] + 0.000548727756825382 * v[2] + v[15];
            y[153] =
                -0.0623594431746757 * v[14] + 0.178441001379851 * v[23] + 0.000548727756825382 * v[9] + v[10];
            y[154] =
                -0.0623594431746757 * v[28] + 0.178441001379851 * v[3] + 0.000548727756825382 * v[13] + v[25];
            y[156] =
                -0.0623753696679426 * v[29] + 0.198440988697155 * v[21] + 0.000564654291040047 * v[2] + v[15];
            y[157] =
                -0.0623753696679426 * v[14] + 0.198440988697155 * v[23] + 0.000564654291040047 * v[9] + v[10];
            y[158] =
                -0.0623753696679426 * v[28] + 0.198440988697155 * v[3] + 0.000564654291040047 * v[13] + v[25];
            y[168] = 0.209999993443489 * v[6] + y[24];
            y[169] = 0.209999993443489 * v[5] + y[25];
            y[170] = 1.003559 + 0.209999993443489 * v[0];
            y[172] = 0.170000001788139 * v[12] + y[28];
            y[173] = 0.170000001788139 * v[1] + y[29];
            y[174] = 0.170000001788139 * v[4] + y[30];
            y[176] = 0.0900000035762787 * v[8] + v[18];
            y[177] = 0.0900000035762787 * v[7] + v[20];
            y[180] = 0.0900000035762787 * v[17] + v[22];
            y[181] = 0.0900000035762787 * v[19] + v[24];
            y[182] = 0.0900000035762787 * v[16] + y[74];
            y[184] = 0.0599999986588955 * v[21] + v[15];
            y[185] = 0.0599999986588955 * v[23] + v[10];
            y[186] = 0.0599999986588955 * v[3] + v[25];
            y[188] = 1.59265619004145e-05 * v[29] + 0.0973000079393387 * v[21] + v[15];
            y[189] = 1.59265619004145e-05 * v[14] + 0.0973000079393387 * v[23] + v[10];
            y[190] = 1.59265619004145e-05 * v[28] + 0.0973000079393387 * v[3] + v[25];
            y[192] = -3.18529237119947e-05 * v[29] + 0.157299980521202 * v[21] + 0.000531853060238063 * v[2] +
                     v[15];
            y[193] = -3.18529237119947e-05 * v[14] + 0.157299980521202 * v[23] + 0.000531853060238063 * v[9] +
                     v[10];
            y[194] = -3.18529237119947e-05 * v[28] + 0.157299980521202 * v[3] + 0.000531853060238063 * v[13] +
                     v[25];
            y[196] =
                0.0326288715004921 * v[29] + 0.206633478403091 * v[21] + 0.000571117969229817 * v[2] + v[15];
            y[197] =
                0.0326288715004921 * v[14] + 0.206633478403091 * v[23] + 0.000571117969229817 * v[9] + v[10];
            y[198] =
                0.0326288715004921 * v[28] + 0.206633478403091 * v[3] + 0.000571117969229817 * v[13] + v[25];
            y[200] = 0.047183919698 * v[29] + 0.244642093777657 * v[21] + 0.000601375999394804 * v[2] + v[15];
            y[201] = 0.047183919698 * v[14] + 0.244642093777657 * v[23] + 0.000601375999394804 * v[9] + v[10];
            y[202] = 0.047183919698 * v[28] + 0.244642093777657 * v[3] + 0.000601375999394804 * v[13] + v[25];
            y[204] =
                0.0305511318147182 * v[29] + 0.180116340517998 * v[21] + 0.000550002965610474 * v[2] + v[15];
            y[205] =
                0.0305511318147182 * v[14] + 0.180116340517998 * v[23] + 0.000550002965610474 * v[9] + v[10];
            y[206] =
                0.0305511318147182 * v[28] + 0.180116340517998 * v[3] + 0.000550002965610474 * v[13] + v[25];
            y[208] =
                0.0622277781367302 * v[29] + 0.198207601904869 * v[21] + 0.000564389454666525 * v[2] + v[15];
            y[209] =
                0.0622277781367302 * v[14] + 0.198207601904869 * v[23] + 0.000564389454666525 * v[9] + v[10];
            y[210] =
                0.0622277781367302 * v[28] + 0.198207601904869 * v[3] + 0.000564389454666525 * v[13] + v[25];
            y[212] =
                -0.0327711068093777 * v[29] + 0.206581398844719 * v[21] + 0.000571117969229817 * v[2] + v[15];
            y[213] =
                -0.0327711068093777 * v[14] + 0.206581398844719 * v[23] + 0.000571117969229817 * v[9] + v[10];
            y[214] =
                -0.0327711068093777 * v[28] + 0.206581398844719 * v[3] + 0.000571117969229817 * v[13] + v[25];
            y[216] =
                -0.0469076968729496 * v[29] + 0.244927540421486 * v[21] + 0.000601662963163108 * v[2] + v[15];
            y[217] =
                -0.0469076968729496 * v[14] + 0.244927540421486 * v[23] + 0.000601662963163108 * v[9] + v[10];
            y[218] =
                -0.0469076968729496 * v[28] + 0.244927540421486 * v[3] + 0.000601662963163108 * v[13] + v[25];
            y[220] =
                -0.0306511372327805 * v[29] + 0.180067613720894 * v[21] + 0.000550002965610474 * v[2] + v[15];
            y[221] =
                -0.0306511372327805 * v[14] + 0.180067613720894 * v[23] + 0.000550002965610474 * v[9] + v[10];
            y[222] =
                -0.0306511372327805 * v[28] + 0.180067613720894 * v[3] + 0.000550002965610474 * v[13] + v[25];
            y[224] =
                -0.062375370413065 * v[29] + 0.198440983891487 * v[21] + 0.000564654299523681 * v[2] + v[15];
            y[225] =
                -0.062375370413065 * v[14] + 0.198440983891487 * v[23] + 0.000564654299523681 * v[9] + v[10];
            y[226] =
                -0.062375370413065 * v[28] + 0.198440983891487 * v[3] + 0.000564654299523681 * v[13] + v[25];
            // variable duplicates: 1
            y[178] = y[74];
            // dependent variables without operations
            y[0] = 0.;
            y[1] = 0.;
            y[2] = 0.9144;
            y[3] = 0.0799999982118607;
            y[4] = 0.;
            y[5] = 0.;
            y[6] = 1.003559;
            y[7] = 0.0799999982118607;
            y[11] = 0.0799999982118607;
            y[15] = 0.0799999982118607;
            y[19] = 0.0799999982118607;
            y[23] = 0.0799999982118607;
            y[26] = 1.003559;
            y[27] = 0.0799999982118607;
            y[31] = 0.0799999982118607;
            y[35] = 0.0399999991059303;
            y[39] = 0.0399999991059303;
            y[43] = 0.0399999991059303;
            y[47] = 0.0399999991059303;
            y[51] = 0.0399999991059303;
            y[55] = 0.0399999991059303;
            y[59] = 0.0399999991059303;
            y[63] = 0.0399999991059303;
            y[67] = 0.0399999991059303;
            y[71] = 0.0399999991059303;
            y[75] = 0.0399999991059303;
            y[79] = 0.0399999991059303;
            y[83] = 0.0399999991059303;
            y[87] = 0.0399999991059303;
            y[91] = 0.0399999991059303;
            y[95] = 0.0399999991059303;
            y[99] = 0.0399999991059303;
            y[103] = 0.0399999991059303;
            y[107] = 0.0199999995529652;
            y[111] = 0.0149999996647239;
            y[115] = 0.0149999996647239;
            y[119] = 0.0199999995529652;
            y[123] = 0.0149999996647239;
            y[127] = 0.0149999996647239;
            y[131] = 0.0149999996647239;
            y[135] = 0.0199999995529652;
            y[139] = 0.0149999996647239;
            y[143] = 0.0149999996647239;
            y[147] = 0.0199999995529652;
            y[151] = 0.0149999996647239;
            y[155] = 0.0149999996647239;
            y[159] = 0.0149999996647239;
            y[160] = 0.;
            y[161] = 0.;
            y[162] = 0.914399981498718;
            y[163] = 0.0799999982118607;
            y[164] = 0.;
            y[165] = 0.;
            y[166] = 1.003559;
            y[167] = 0.0799999982118607;
            y[171] = 0.28999999165535;
            y[175] = 0.25;
            y[179] = 0.0700000002980232;
            y[183] = 0.0700000002980232;
            y[187] = 0.0399999991059303;
            y[191] = 0.0399999991059303;
            y[195] = 0.0599999986588955;
            y[199] = 0.0199999995529652;
            y[203] = 0.0274999998509884;
            y[207] = 0.0199999995529652;
            y[211] = 0.0350000001490116;
            y[215] = 0.0199999995529652;
            y[219] = 0.0274999998509884;
            y[223] = 0.0199999995529652;
            y[227] = 0.0350000001490116;

            Debug output;

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[0], y[1], y[2], y[3]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[4], y[5], y[6], y[7]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[8], y[9], y[10], y[11]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[12], y[13], y[14], y[15]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[16], y[17], y[18], y[19]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[20], y[21], y[22], y[23]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[24], y[25], y[26], y[27]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[28], y[29], y[30], y[31]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[32], y[33], y[34], y[35]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[36], y[37], y[38], y[39]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[40], y[41], y[42], y[43]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[44], y[45], y[46], y[47]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[48], y[49], y[50], y[51]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[52], y[53], y[54], y[55]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[56], y[57], y[58], y[59]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[60], y[61], y[62], y[63]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[64], y[65], y[66], y[67]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[68], y[69], y[70], y[71]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[72], y[73], y[74], y[75]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[76], y[77], y[78], y[79]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[80], y[81], y[82], y[83]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[84], y[85], y[86], y[87]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[88], y[89], y[90], y[91]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[92], y[93], y[94], y[95]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[96], y[97], y[98], y[99]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[100], y[101], y[102], y[103]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[104], y[105], y[106], y[107]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[108], y[109], y[110], y[111]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[112], y[113], y[114], y[115]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[116], y[117], y[118], y[119]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[120], y[121], y[122], y[123]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[124], y[125], y[126], y[127]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[128], y[129], y[130], y[131]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[132], y[133], y[134], y[135]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[136], y[137], y[138], y[139]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[140], y[141], y[142], y[143]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[144], y[145], y[146], y[147]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[148], y[149], y[150], y[151]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[152], y[153], y[154], y[155]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[156], y[157], y[158], y[159]));

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[8], y[9], y[10], y[11]))
            {
                output.second.emplace_back(0, 2);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[12], y[13], y[14], y[15]))
            {
                output.second.emplace_back(0, 3);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[16], y[17], y[18], y[19]))
            {
                output.second.emplace_back(0, 4);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[20], y[21], y[22], y[23]))
            {
                output.second.emplace_back(0, 5);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[24], y[25], y[26], y[27]))
            {
                output.second.emplace_back(0, 6);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[28], y[29], y[30], y[31]))
            {
                output.second.emplace_back(0, 7);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[32], y[33], y[34], y[35]))
            {
                output.second.emplace_back(0, 8);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[36], y[37], y[38], y[39]))
            {
                output.second.emplace_back(0, 9);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[40], y[41], y[42], y[43]))
            {
                output.second.emplace_back(0, 10);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[44], y[45], y[46], y[47]))
            {
                output.second.emplace_back(0, 11);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[48], y[49], y[50], y[51]))
            {
                output.second.emplace_back(0, 12);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[52], y[53], y[54], y[55]))
            {
                output.second.emplace_back(0, 13);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[56], y[57], y[58], y[59]))
            {
                output.second.emplace_back(0, 14);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[60], y[61], y[62], y[63]))
            {
                output.second.emplace_back(0, 15);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(0, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(0, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(0, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(0, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(0, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(0, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(0, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(0, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(0, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(0, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(0, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(0, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(0, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(0, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(0, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(0, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(0, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(0, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(0, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(0, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(0, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(0, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(0, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(0, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[28], y[29], y[30], y[31]))
            {
                output.second.emplace_back(1, 7);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[32], y[33], y[34], y[35]))
            {
                output.second.emplace_back(1, 8);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[36], y[37], y[38], y[39]))
            {
                output.second.emplace_back(1, 9);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[40], y[41], y[42], y[43]))
            {
                output.second.emplace_back(1, 10);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[44], y[45], y[46], y[47]))
            {
                output.second.emplace_back(1, 11);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[48], y[49], y[50], y[51]))
            {
                output.second.emplace_back(1, 12);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[52], y[53], y[54], y[55]))
            {
                output.second.emplace_back(1, 13);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[56], y[57], y[58], y[59]))
            {
                output.second.emplace_back(1, 14);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[60], y[61], y[62], y[63]))
            {
                output.second.emplace_back(1, 15);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(1, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(1, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(1, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(1, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(1, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(1, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(1, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(1, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(1, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(1, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(1, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(1, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(1, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(1, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(1, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(1, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(1, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(1, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(1, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(1, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(1, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(1, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(1, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(1, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(2, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(2, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(2, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(3, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(3, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(3, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(4, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(4, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(4, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(5, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(5, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(5, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(6, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(6, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(6, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(2, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(2, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(2, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(3, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(3, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(3, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(4, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(4, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(4, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(5, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(5, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(5, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(6, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(6, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(6, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(2, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(3, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(4, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(5, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(6, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(2, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(3, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(4, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(5, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(6, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(2, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(2, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(3, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(3, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(4, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(4, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(5, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(5, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(6, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(6, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(2, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(3, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(4, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(5, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(6, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(2, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(2, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(3, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(3, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(4, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(4, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(5, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(5, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(6, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(6, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(2, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(3, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(4, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(5, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(6, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(2, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(2, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(2, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(3, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(3, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(3, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(4, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(4, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(4, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(5, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(5, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(5, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(6, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(6, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(6, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(2, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(3, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(4, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(5, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(6, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(2, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(2, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(3, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(3, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(4, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(4, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(5, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(5, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(6, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(6, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(2, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(3, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(4, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(5, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(6, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(2, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(2, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(2, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(3, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(3, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(3, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(4, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(4, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(4, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(5, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(5, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(5, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(6, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(6, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(6, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(7, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(7, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(7, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(8, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(8, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(8, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(9, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(9, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(9, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(10, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(10, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(10, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(11, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(11, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(11, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(12, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(12, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(12, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(13, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(13, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(13, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(14, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(14, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(14, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(15, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(15, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(15, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(7, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(8, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(9, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(10, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(11, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(12, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(13, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(14, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(15, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(7, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(8, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(9, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(10, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(11, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(12, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(13, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(14, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(15, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(7, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(7, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(8, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(8, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(9, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(9, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(10, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(10, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(11, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(11, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(12, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(12, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(13, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(13, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(14, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(14, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(15, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(15, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(7, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(8, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(9, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(10, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(11, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(12, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(13, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(14, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(15, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(7, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(7, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(8, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(8, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(9, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(9, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(10, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(10, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(11, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(11, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(12, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(12, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(13, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(13, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(14, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(14, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(15, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(15, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(7, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(8, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(9, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(10, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(11, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(12, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(13, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(14, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(15, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(7, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(7, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(7, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(8, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(8, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(8, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(9, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(9, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(9, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(10, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(10, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(10, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(11, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(11, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(11, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(12, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(12, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(12, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(13, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(13, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(13, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(14, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(14, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(14, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(15, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(15, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(15, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(7, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(8, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(9, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(10, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(11, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(12, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(13, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(14, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(15, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(7, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(7, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(8, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(8, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(9, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(9, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(10, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(10, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(11, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(11, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(12, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(12, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(13, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(13, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(14, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(14, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(15, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(15, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(7, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(8, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(9, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(10, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(11, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(12, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(13, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(14, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(15, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(7, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(7, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(7, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(8, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(8, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(8, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(9, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(9, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(9, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(10, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(10, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(10, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(11, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(11, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(11, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(12, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(12, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(12, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(13, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(13, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(13, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(14, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(14, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(14, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(15, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(15, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(15, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(16, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(17, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(18, 23);
            }

            return output;
        }

        template <std::size_t rake>
        inline static bool fkcc(
            const vamp::collision::Environment<FloatVector<rake>> &environment,
            const ConfigurationBlock<rake> &x) noexcept
        {
            std::array<FloatVector<rake, 1>, 30> v;
            std::array<FloatVector<rake, 1>, 228> y;

            v[0] = cos(x[0]);
            v[1] = sin(x[0]);
            v[2] = 0.000796326710733264 * v[0] + -0.999999682931835 * v[1];
            v[3] = sin(x[1]);
            v[4] = cos(x[1]);
            v[5] = 1.79489656471077e-09 * v[3] + v[4];
            v[6] = v[2] * v[5];
            v[7] = -v[1];
            v[8] = 0.000796326710733264 * v[7] + -0.999999682931835 * v[0];
            y[24] = 0.13585 * v[8];
            y[8] = 0.105 * v[6] + y[24];
            v[1] = 0.999999682931835 * v[0] + 0.000796326710733264 * v[1];
            v[5] = v[1] * v[5];
            v[7] = 0.999999682931835 * v[7] + 0.000796326710733264 * v[0];
            y[25] = 0.13585 * v[7];
            y[9] = 0.105 * v[5] + y[25];
            v[0] = -1. * v[3] + 1.79489656471077e-09 * v[4];
            y[10] = 1.003559 + 0.105 * v[0];
            y[12] = 0.21 * v[6] + y[24];
            y[13] = 0.21 * v[5] + y[25];
            y[14] = 1.003559 + 0.21 * v[0];
            y[16] = 0.315 * v[6] + y[24];
            y[17] = 0.315 * v[5] + y[25];
            y[18] = 1.003559 + 0.315 * v[0];
            y[20] = 0.42 * v[6] + y[24];
            y[21] = 0.42 * v[5] + y[25];
            y[22] = 1.003559 + 0.42 * v[0];
            y[28] = y[24] + -0.1197 * v[8] + 0.425 * v[6];
            y[29] = y[25] + -0.1197 * v[7] + 0.425 * v[5];
            y[30] = 1.003559 + 0.425 * v[0];
            v[3] = -v[3];
            v[9] = 1.79489656471077e-09 * v[4] + v[3];
            v[2] = v[2] * v[9];
            v[10] = sin(x[2]);
            v[11] = cos(x[2]);
            v[12] = v[2] * v[10] + v[6] * v[11];
            y[32] = 0.1 * v[12] + y[28];
            v[9] = v[1] * v[9];
            v[1] = v[9] * v[10] + v[5] * v[11];
            y[33] = 0.1 * v[1] + y[29];
            v[3] = -1. * v[4] + 1.79489656471077e-09 * v[3];
            v[4] = v[3] * v[10] + v[0] * v[11];
            y[34] = 0.1 * v[4] + y[30];
            y[36] = 0.14 * v[12] + y[28];
            y[37] = 0.14 * v[1] + y[29];
            y[38] = 0.14 * v[4] + y[30];
            y[40] = 0.18 * v[12] + y[28];
            y[41] = 0.18 * v[1] + y[29];
            y[42] = 0.18 * v[4] + y[30];
            y[44] = 0.22 * v[12] + y[28];
            y[45] = 0.22 * v[1] + y[29];
            y[46] = 0.22 * v[4] + y[30];
            y[48] = 0.26 * v[12] + y[28];
            y[49] = 0.26 * v[1] + y[29];
            y[50] = 0.26 * v[4] + y[30];
            y[52] = 0.3 * v[12] + y[28];
            y[53] = 0.3 * v[1] + y[29];
            y[54] = 0.3 * v[4] + y[30];
            y[56] = 0.34 * v[12] + y[28];
            y[57] = 0.34 * v[1] + y[29];
            y[58] = 0.34 * v[4] + y[30];
            y[60] = 0.38 * v[12] + y[28];
            y[61] = 0.38 * v[1] + y[29];
            y[62] = 0.38 * v[4] + y[30];
            v[10] = -v[10];
            v[2] = v[2] * v[11] + v[6] * v[10];
            v[13] = sin(x[3]);
            v[14] = cos(x[3]);
            v[15] = 1.79489656471077e-09 * v[13] + v[14];
            v[16] = -1. * v[13] + 1.79489656471077e-09 * v[14];
            v[17] = v[2] * v[15] + v[12] * v[16];
            v[18] = y[28] + 0.39225 * v[12];
            y[64] = 0.09 * v[8] + 0.03 * v[17] + v[18];
            v[9] = v[9] * v[11] + v[5] * v[10];
            v[19] = v[9] * v[15] + v[1] * v[16];
            v[20] = y[29] + 0.39225 * v[1];
            y[65] = 0.09 * v[7] + 0.03 * v[19] + v[20];
            v[10] = v[3] * v[11] + v[0] * v[10];
            v[16] = v[10] * v[15] + v[4] * v[16];
            y[74] = y[30] + 0.39225 * v[4];
            y[66] = 0.03 * v[16] + y[74];
            y[68] = 0.09 * v[8] + -0.03 * v[17] + v[18];
            y[69] = 0.09 * v[7] + -0.03 * v[19] + v[20];
            y[70] = -0.03 * v[16] + y[74];
            y[72] = 0.09 * v[8] + v[18];
            y[73] = 0.09 * v[7] + v[20];
            v[13] = -v[13];
            v[15] = 1.79489656471077e-09 * v[14] + v[13];
            v[13] = -1. * v[14] + 1.79489656471077e-09 * v[13];
            v[2] = v[2] * v[15] + v[12] * v[13];
            v[14] = sin(x[4]);
            v[3] = -v[14];
            v[11] = cos(x[4]);
            v[21] = v[2] * v[3] + v[8] * v[11];
            v[22] = v[18] + 0.093 * v[8];
            y[76] = 0.03 * v[21] + 0.09 * v[17] + v[22];
            v[9] = v[9] * v[15] + v[1] * v[13];
            v[23] = v[9] * v[3] + v[7] * v[11];
            v[24] = v[20] + 0.093 * v[7];
            y[77] = 0.03 * v[23] + 0.09 * v[19] + v[24];
            v[13] = v[10] * v[15] + v[4] * v[13];
            v[3] = v[13] * v[3];
            y[78] = 0.03 * v[3] + 0.09 * v[16] + y[74];
            y[80] = -0.03 * v[21] + 0.09 * v[17] + v[22];
            y[81] = -0.03 * v[23] + 0.09 * v[19] + v[24];
            y[82] = -0.03 * v[3] + 0.09 * v[16] + y[74];
            y[84] = 0.09 * v[17] + v[22];
            y[85] = 0.09 * v[19] + v[24];
            y[86] = 0.09 * v[16] + y[74];
            v[15] = v[22] + 0.09465 * v[17];
            y[88] = 0.06 * v[21] + v[15];
            v[10] = v[24] + 0.09465 * v[19];
            y[89] = 0.06 * v[23] + v[10];
            v[25] = y[74] + 0.09465 * v[16];
            y[90] = 0.06 * v[3] + v[25];
            v[2] = v[2] * v[11] + v[8] * v[14];
            v[26] = cos(x[5]);
            v[27] = sin(x[5]);
            v[28] = -v[27];
            v[29] = v[2] * v[26] + v[17] * v[28];
            y[92] = 1.59265611381251e-05 * v[29] + 0.0973000063413347 * v[21] + v[15];
            v[9] = v[9] * v[11] + v[7] * v[14];
            v[14] = v[9] * v[26] + v[19] * v[28];
            y[93] = 1.59265611381251e-05 * v[14] + 0.0973000063413347 * v[23] + v[10];
            v[13] = v[13] * v[11];
            v[28] = v[13] * v[26] + v[16] * v[28];
            y[94] = 1.59265611381251e-05 * v[28] + 0.0973000063413347 * v[3] + v[25];
            v[2] = v[2] * v[27] + v[17] * v[26];
            y[96] = -4.77794169794995e-05 * v[29] + 0.177299961951912 * v[21] + 0.000547779602643996 * v[2] +
                    v[15];
            v[9] = v[9] * v[27] + v[19] * v[26];
            y[97] = -4.77794169794995e-05 * v[14] + 0.177299961951912 * v[23] + 0.000547779602643996 * v[9] +
                    v[10];
            v[13] = v[13] * v[27] + v[16] * v[26];
            y[98] = -4.77794169794995e-05 * v[28] + 0.177299961951912 * v[3] + 0.000547779602643996 * v[13] +
                    v[25];
            y[100] =
                -1.592643044558e-05 * v[29] + 0.137299987317304 * v[21] + 0.000515926534214666 * v[2] + v[15];
            y[101] =
                -1.592643044558e-05 * v[14] + 0.137299987317304 * v[23] + 0.000515926534214666 * v[9] + v[10];
            y[102] =
                -1.592643044558e-05 * v[28] + 0.137299987317304 * v[3] + 0.000515926534214666 * v[13] + v[25];
            y[104] =
                0.0326288719300173 * v[29] + 0.206633483191718 * v[21] + 0.000571117947718783 * v[2] + v[15];
            y[105] =
                0.0326288719300173 * v[14] + 0.206633483191718 * v[23] + 0.000571117947718783 * v[9] + v[10];
            y[106] =
                0.0326288719300173 * v[28] + 0.206633483191718 * v[3] + 0.000571117947718783 * v[13] + v[25];
            y[108] =
                0.0471739661448351 * v[29] + 0.257142085198042 * v[21] + 0.000611330073064923 * v[2] + v[15];
            y[109] =
                0.0471739661448351 * v[14] + 0.257142085198042 * v[23] + 0.000611330073064923 * v[9] + v[10];
            y[110] =
                0.0471739661448351 * v[28] + 0.257142085198042 * v[3] + 0.000611330073064923 * v[13] + v[25];
            y[112] =
                0.0471938742614188 * v[29] + 0.232142101051412 * v[21] + 0.000591421905296592 * v[2] + v[15];
            y[113] =
                0.0471938742614188 * v[14] + 0.232142101051412 * v[23] + 0.000591421905296592 * v[9] + v[10];
            y[114] =
                0.0471938742614188 * v[28] + 0.232142101051412 * v[3] + 0.000591421905296592 * v[13] + v[25];
            y[116] =
                0.0305511319550535 * v[29] + 0.180116344904696 * v[21] + 0.000550002959776232 * v[2] + v[15];
            y[117] =
                0.0305511319550535 * v[14] + 0.180116344904696 * v[23] + 0.000550002959776232 * v[9] + v[10];
            y[118] =
                0.0305511319550535 * v[28] + 0.180116344904696 * v[3] + 0.000550002959776232 * v[13] + v[25];
            y[120] =
                0.0622118532100359 * v[29] + 0.218207593376579 * v[21] + 0.000580315961208078 * v[2] + v[15];
            y[121] =
                0.0622118532100359 * v[14] + 0.218207593376579 * v[23] + 0.000580315961208078 * v[9] + v[10];
            y[122] =
                0.0622118532100359 * v[28] + 0.218207593376579 * v[3] + 0.000580315961208078 * v[13] + v[25];
            y[124] =
                0.0622437061965698 * v[29] + 0.178207618741971 * v[21] + 0.000548462892778747 * v[2] + v[15];
            y[125] =
                0.0622437061965698 * v[14] + 0.178207618741971 * v[23] + 0.000548462892778747 * v[9] + v[10];
            y[126] =
                0.0622437061965698 * v[28] + 0.178207618741971 * v[3] + 0.000548462892778747 * v[13] + v[25];
            y[128] =
                0.0622277797033028 * v[29] + 0.198207606059275 * v[21] + 0.000564389426993413 * v[2] + v[15];
            y[129] =
                0.0622277797033028 * v[14] + 0.198207606059275 * v[23] + 0.000564389426993413 * v[9] + v[10];
            y[130] =
                0.0622277797033028 * v[28] + 0.198207606059275 * v[3] + 0.000564389426993413 * v[13] + v[25];
            y[132] =
                -0.0327711073338182 * v[29] + 0.206581403542295 * v[21] + 0.000571117947718776 * v[2] + v[15];
            y[133] =
                -0.0327711073338182 * v[14] + 0.206581403542295 * v[23] + 0.000571117947718776 * v[9] + v[10];
            y[134] =
                -0.0327711073338182 * v[28] + 0.206581403542295 * v[3] + 0.000571117947718776 * v[13] + v[25];
            y[136] =
                -0.0469176493992621 * v[29] + 0.257427526148899 * v[21] + 0.000611617044105126 * v[2] + v[15];
            y[137] =
                -0.0469176493992621 * v[14] + 0.257427526148899 * v[23] + 0.000611617044105126 * v[9] + v[10];
            y[138] =
                -0.0469176493992621 * v[28] + 0.257427526148899 * v[3] + 0.000611617044105126 * v[13] + v[25];
            y[140] =
                -0.0468977412826784 * v[29] + 0.232427542002269 * v[21] + 0.000591708876336795 * v[2] + v[15];
            y[141] =
                -0.0468977412826784 * v[14] + 0.232427542002269 * v[23] + 0.000591708876336795 * v[9] + v[10];
            y[142] =
                -0.0468977412826784 * v[28] + 0.232427542002269 * v[3] + 0.000591708876336795 * v[13] + v[25];
            y[144] =
                -0.0306511374916797 * v[29] + 0.180067607997177 * v[21] + 0.000550002959776232 * v[2] + v[15];
            y[145] =
                -0.0306511374916797 * v[14] + 0.180067607997177 * v[23] + 0.000550002959776232 * v[9] + v[10];
            y[146] =
                -0.0306511374916797 * v[28] + 0.180067607997177 * v[3] + 0.000550002959776232 * v[13] + v[25];
            y[148] =
                -0.0623912961612096 * v[29] + 0.218440976014459 * v[21] + 0.000580580825254713 * v[2] + v[15];
            y[149] =
                -0.0623912961612096 * v[14] + 0.218440976014459 * v[23] + 0.000580580825254713 * v[9] + v[10];
            y[150] =
                -0.0623912961612096 * v[28] + 0.218440976014459 * v[3] + 0.000580580825254713 * v[13] + v[25];
            y[152] =
                -0.0623594431746757 * v[29] + 0.178441001379851 * v[21] + 0.000548727756825382 * v[2] + v[15];
            y[153] =
                -0.0623594431746757 * v[14] + 0.178441001379851 * v[23] + 0.000548727756825382 * v[9] + v[10];
            y[154] =
                -0.0623594431746757 * v[28] + 0.178441001379851 * v[3] + 0.000548727756825382 * v[13] + v[25];
            y[156] =
                -0.0623753696679426 * v[29] + 0.198440988697155 * v[21] + 0.000564654291040047 * v[2] + v[15];
            y[157] =
                -0.0623753696679426 * v[14] + 0.198440988697155 * v[23] + 0.000564654291040047 * v[9] + v[10];
            y[158] =
                -0.0623753696679426 * v[28] + 0.198440988697155 * v[3] + 0.000564654291040047 * v[13] + v[25];
            y[168] = 0.209999993443489 * v[6] + y[24];
            y[169] = 0.209999993443489 * v[5] + y[25];
            y[170] = 1.003559 + 0.209999993443489 * v[0];
            y[172] = 0.170000001788139 * v[12] + y[28];
            y[173] = 0.170000001788139 * v[1] + y[29];
            y[174] = 0.170000001788139 * v[4] + y[30];
            y[176] = 0.0900000035762787 * v[8] + v[18];
            y[177] = 0.0900000035762787 * v[7] + v[20];
            y[180] = 0.0900000035762787 * v[17] + v[22];
            y[181] = 0.0900000035762787 * v[19] + v[24];
            y[182] = 0.0900000035762787 * v[16] + y[74];
            y[184] = 0.0599999986588955 * v[21] + v[15];
            y[185] = 0.0599999986588955 * v[23] + v[10];
            y[186] = 0.0599999986588955 * v[3] + v[25];
            y[188] = 1.59265619004145e-05 * v[29] + 0.0973000079393387 * v[21] + v[15];
            y[189] = 1.59265619004145e-05 * v[14] + 0.0973000079393387 * v[23] + v[10];
            y[190] = 1.59265619004145e-05 * v[28] + 0.0973000079393387 * v[3] + v[25];
            y[192] = -3.18529237119947e-05 * v[29] + 0.157299980521202 * v[21] + 0.000531853060238063 * v[2] +
                     v[15];
            y[193] = -3.18529237119947e-05 * v[14] + 0.157299980521202 * v[23] + 0.000531853060238063 * v[9] +
                     v[10];
            y[194] = -3.18529237119947e-05 * v[28] + 0.157299980521202 * v[3] + 0.000531853060238063 * v[13] +
                     v[25];
            y[196] =
                0.0326288715004921 * v[29] + 0.206633478403091 * v[21] + 0.000571117969229817 * v[2] + v[15];
            y[197] =
                0.0326288715004921 * v[14] + 0.206633478403091 * v[23] + 0.000571117969229817 * v[9] + v[10];
            y[198] =
                0.0326288715004921 * v[28] + 0.206633478403091 * v[3] + 0.000571117969229817 * v[13] + v[25];
            y[200] = 0.047183919698 * v[29] + 0.244642093777657 * v[21] + 0.000601375999394804 * v[2] + v[15];
            y[201] = 0.047183919698 * v[14] + 0.244642093777657 * v[23] + 0.000601375999394804 * v[9] + v[10];
            y[202] = 0.047183919698 * v[28] + 0.244642093777657 * v[3] + 0.000601375999394804 * v[13] + v[25];
            y[204] =
                0.0305511318147182 * v[29] + 0.180116340517998 * v[21] + 0.000550002965610474 * v[2] + v[15];
            y[205] =
                0.0305511318147182 * v[14] + 0.180116340517998 * v[23] + 0.000550002965610474 * v[9] + v[10];
            y[206] =
                0.0305511318147182 * v[28] + 0.180116340517998 * v[3] + 0.000550002965610474 * v[13] + v[25];
            y[208] =
                0.0622277781367302 * v[29] + 0.198207601904869 * v[21] + 0.000564389454666525 * v[2] + v[15];
            y[209] =
                0.0622277781367302 * v[14] + 0.198207601904869 * v[23] + 0.000564389454666525 * v[9] + v[10];
            y[210] =
                0.0622277781367302 * v[28] + 0.198207601904869 * v[3] + 0.000564389454666525 * v[13] + v[25];
            y[212] =
                -0.0327711068093777 * v[29] + 0.206581398844719 * v[21] + 0.000571117969229817 * v[2] + v[15];
            y[213] =
                -0.0327711068093777 * v[14] + 0.206581398844719 * v[23] + 0.000571117969229817 * v[9] + v[10];
            y[214] =
                -0.0327711068093777 * v[28] + 0.206581398844719 * v[3] + 0.000571117969229817 * v[13] + v[25];
            y[216] =
                -0.0469076968729496 * v[29] + 0.244927540421486 * v[21] + 0.000601662963163108 * v[2] + v[15];
            y[217] =
                -0.0469076968729496 * v[14] + 0.244927540421486 * v[23] + 0.000601662963163108 * v[9] + v[10];
            y[218] =
                -0.0469076968729496 * v[28] + 0.244927540421486 * v[3] + 0.000601662963163108 * v[13] + v[25];
            y[220] =
                -0.0306511372327805 * v[29] + 0.180067613720894 * v[21] + 0.000550002965610474 * v[2] + v[15];
            y[221] =
                -0.0306511372327805 * v[14] + 0.180067613720894 * v[23] + 0.000550002965610474 * v[9] + v[10];
            y[222] =
                -0.0306511372327805 * v[28] + 0.180067613720894 * v[3] + 0.000550002965610474 * v[13] + v[25];
            y[224] =
                -0.062375370413065 * v[29] + 0.198440983891487 * v[21] + 0.000564654299523681 * v[2] + v[15];
            y[225] =
                -0.062375370413065 * v[14] + 0.198440983891487 * v[23] + 0.000564654299523681 * v[9] + v[10];
            y[226] =
                -0.062375370413065 * v[28] + 0.198440983891487 * v[3] + 0.000564654299523681 * v[13] + v[25];
            // variable duplicates: 1
            y[178] = y[74];
            // dependent variables without operations
            y[0] = 0.;
            y[1] = 0.;
            y[2] = 0.9144;
            y[3] = 0.0799999982118607;
            y[4] = 0.;
            y[5] = 0.;
            y[6] = 1.003559;
            y[7] = 0.0799999982118607;
            y[11] = 0.0799999982118607;
            y[15] = 0.0799999982118607;
            y[19] = 0.0799999982118607;
            y[23] = 0.0799999982118607;
            y[26] = 1.003559;
            y[27] = 0.0799999982118607;
            y[31] = 0.0799999982118607;
            y[35] = 0.0399999991059303;
            y[39] = 0.0399999991059303;
            y[43] = 0.0399999991059303;
            y[47] = 0.0399999991059303;
            y[51] = 0.0399999991059303;
            y[55] = 0.0399999991059303;
            y[59] = 0.0399999991059303;
            y[63] = 0.0399999991059303;
            y[67] = 0.0399999991059303;
            y[71] = 0.0399999991059303;
            y[75] = 0.0399999991059303;
            y[79] = 0.0399999991059303;
            y[83] = 0.0399999991059303;
            y[87] = 0.0399999991059303;
            y[91] = 0.0399999991059303;
            y[95] = 0.0399999991059303;
            y[99] = 0.0399999991059303;
            y[103] = 0.0399999991059303;
            y[107] = 0.0199999995529652;
            y[111] = 0.0149999996647239;
            y[115] = 0.0149999996647239;
            y[119] = 0.0199999995529652;
            y[123] = 0.0149999996647239;
            y[127] = 0.0149999996647239;
            y[131] = 0.0149999996647239;
            y[135] = 0.0199999995529652;
            y[139] = 0.0149999996647239;
            y[143] = 0.0149999996647239;
            y[147] = 0.0199999995529652;
            y[151] = 0.0149999996647239;
            y[155] = 0.0149999996647239;
            y[159] = 0.0149999996647239;
            y[160] = 0.;
            y[161] = 0.;
            y[162] = 0.914399981498718;
            y[163] = 0.0799999982118607;
            y[164] = 0.;
            y[165] = 0.;
            y[166] = 1.003559;
            y[167] = 0.0799999982118607;
            y[171] = 0.28999999165535;
            y[175] = 0.25;
            y[179] = 0.0700000002980232;
            y[183] = 0.0700000002980232;
            y[187] = 0.0399999991059303;
            y[191] = 0.0399999991059303;
            y[195] = 0.0599999986588955;
            y[199] = 0.0199999995529652;
            y[203] = 0.0274999998509884;
            y[207] = 0.0199999995529652;
            y[211] = 0.0350000001490116;
            y[215] = 0.0199999995529652;
            y[219] = 0.0274999998509884;
            y[223] = 0.0199999995529652;
            y[227] = 0.0350000001490116;

            //
            // environment vs. robot collisions
            //

            // robotiq_85_right_finger_link
            if (sphere_environment_in_collision(environment, y[224], y[225], y[226], y[227]))
            {
                if (sphere_environment_in_collision(environment, y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_right_knuckle_link
            if (sphere_environment_in_collision(environment, y[220], y[221], y[222], y[223]))
            {
                if (sphere_environment_in_collision(environment, y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_right_finger_tip_link
            if (sphere_environment_in_collision(environment, y[216], y[217], y[218], y[219]))
            {
                if (sphere_environment_in_collision(environment, y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[140], y[141], y[142], y[143]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_right_inner_knuckle_link
            if (sphere_environment_in_collision(environment, y[212], y[213], y[214], y[215]))
            {
                if (sphere_environment_in_collision(environment, y[132], y[133], y[134], y[135]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_left_finger_link
            if (sphere_environment_in_collision(environment, y[208], y[209], y[210], y[211]))
            {
                if (sphere_environment_in_collision(environment, y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[128], y[129], y[130], y[131]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_left_knuckle_link
            if (sphere_environment_in_collision(environment, y[204], y[205], y[206], y[207]))
            {
                if (sphere_environment_in_collision(environment, y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_left_finger_tip_link
            if (sphere_environment_in_collision(environment, y[200], y[201], y[202], y[203]))
            {
                if (sphere_environment_in_collision(environment, y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_left_inner_knuckle_link
            if (sphere_environment_in_collision(environment, y[196], y[197], y[198], y[199]))
            {
                if (sphere_environment_in_collision(environment, y[104], y[105], y[106], y[107]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_base_link
            if (sphere_environment_in_collision(environment, y[192], y[193], y[194], y[195]))
            {
                if (sphere_environment_in_collision(environment, y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // fts_robotside
            if (sphere_environment_in_collision(environment, y[188], y[189], y[190], y[191]))
            {
                if (sphere_environment_in_collision(environment, y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // wrist_3_link
            if (sphere_environment_in_collision(environment, y[184], y[185], y[186], y[187]))
            {
                if (sphere_environment_in_collision(environment, y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // wrist_2_link
            if (sphere_environment_in_collision(environment, y[180], y[181], y[182], y[183]))
            {
                if (sphere_environment_in_collision(environment, y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[84], y[85], y[86], y[87]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // wrist_1_link
            if (sphere_environment_in_collision(environment, y[176], y[177], y[178], y[179]))
            {
                if (sphere_environment_in_collision(environment, y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // forearm_link
            if (sphere_environment_in_collision(environment, y[172], y[173], y[174], y[175]))
            {
                if (sphere_environment_in_collision(environment, y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[32], y[33], y[34], y[35]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[36], y[37], y[38], y[39]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // upper_arm_link
            if (sphere_environment_in_collision(environment, y[168], y[169], y[170], y[171]))
            {
                if (sphere_environment_in_collision(environment, y[8], y[9], y[10], y[11]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[12], y[13], y[14], y[15]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[16], y[17], y[18], y[19]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[20], y[21], y[22], y[23]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[24], y[25], y[26], y[27]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // shoulder_link
            if (sphere_environment_in_collision(environment, y[164], y[165], y[166], y[167]))
            {
                if (sphere_environment_in_collision(environment, y[4], y[5], y[6], y[7]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // base_link
            if (sphere_environment_in_collision(environment, y[160], y[161], y[162], y[163]))
            {
                if (sphere_environment_in_collision(environment, y[0], y[1], y[2], y[3]))
                {
                    return false;
                }
            }

            //
            // robot self-collisions
            //

            // base_link vs. upper_arm_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[168], y[169], y[170], y[171]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[8], y[9], y[10], y[11]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[12], y[13], y[14], y[15]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[16], y[17], y[18], y[19]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[20], y[21], y[22], y[23]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[24], y[25], y[26], y[27]))
                {
                    return false;
                }
            }

            // base_link vs. forearm_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[172], y[173], y[174], y[175]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[32], y[33], y[34], y[35]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[36], y[37], y[38], y[39]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            // base_link vs. wrist_1_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[176], y[177], y[178], y[179]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // base_link vs. wrist_2_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[180], y[181], y[182], y[183]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }
            }

            // base_link vs. wrist_3_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[184], y[185], y[186], y[187]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // base_link vs. fts_robotside
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[188], y[189], y[190], y[191]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_base_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[192], y[193], y[194], y[195]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_left_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[196], y[197], y[198], y[199]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_left_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[200], y[201], y[202], y[203]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_left_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[204], y[205], y[206], y[207]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_left_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[208], y[209], y[210], y[211]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_right_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[212], y[213], y[214], y[215]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_right_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[216], y[217], y[218], y[219]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_right_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[220], y[221], y[222], y[223]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_right_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[224], y[225], y[226], y[227]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // shoulder_link vs. forearm_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[172], y[173], y[174], y[175]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[32], y[33], y[34], y[35]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[36], y[37], y[38], y[39]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            // shoulder_link vs. wrist_1_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[176], y[177], y[178], y[179]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // shoulder_link vs. wrist_2_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[180], y[181], y[182], y[183]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }
            }

            // shoulder_link vs. wrist_3_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[184], y[185], y[186], y[187]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // shoulder_link vs. fts_robotside
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[188], y[189], y[190], y[191]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_base_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[192], y[193], y[194], y[195]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_left_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[196], y[197], y[198], y[199]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_left_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[200], y[201], y[202], y[203]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_left_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[204], y[205], y[206], y[207]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_left_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[208], y[209], y[210], y[211]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_right_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[212], y[213], y[214], y[215]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_right_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[216], y[217], y[218], y[219]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_right_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[220], y[221], y[222], y[223]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_right_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[224], y[225], y[226], y[227]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. wrist_1_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[176], y[177], y[178], y[179]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. wrist_2_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[180], y[181], y[182], y[183]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. wrist_3_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[184], y[185], y[186], y[187]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. fts_robotside
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[188], y[189], y[190], y[191]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_base_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[192], y[193], y[194], y[195]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_left_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[196], y[197], y[198], y[199]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_left_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[200], y[201], y[202], y[203]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_left_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[204], y[205], y[206], y[207]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_left_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[208], y[209], y[210], y[211]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_right_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[212], y[213], y[214], y[215]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_right_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[216], y[217], y[218], y[219]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_right_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[220], y[221], y[222], y[223]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_right_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[224], y[225], y[226], y[227]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // forearm_link vs. wrist_2_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[180], y[181], y[182], y[183]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }
            }

            // forearm_link vs. wrist_3_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[184], y[185], y[186], y[187]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // forearm_link vs. fts_robotside
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[188], y[189], y[190], y[191]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_base_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[192], y[193], y[194], y[195]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_left_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[196], y[197], y[198], y[199]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_left_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[200], y[201], y[202], y[203]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_left_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[204], y[205], y[206], y[207]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_left_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[208], y[209], y[210], y[211]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_right_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[212], y[213], y[214], y[215]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_right_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[216], y[217], y[218], y[219]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_right_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[220], y[221], y[222], y[223]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_right_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[224], y[225], y[226], y[227]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // wrist_1_link vs. fts_robotside
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[188], y[189], y[190], y[191]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            return true;
        }

        template <std::size_t rake>
        inline static bool fkcc_attach(
            const vamp::collision::Environment<FloatVector<rake>> &environment,
            const ConfigurationBlock<rake> &x) noexcept
        {
            std::array<FloatVector<rake, 1>, 30> v;
            std::array<FloatVector<rake, 1>, 240> y;

            v[0] = cos(x[0]);
            v[1] = sin(x[0]);
            v[2] = 0.000796326710733264 * v[0] + -0.999999682931835 * v[1];
            v[3] = sin(x[1]);
            v[4] = cos(x[1]);
            v[5] = 1.79489656471077e-09 * v[3] + v[4];
            v[6] = v[2] * v[5];
            v[7] = -v[1];
            v[8] = 0.000796326710733264 * v[7] + -0.999999682931835 * v[0];
            y[24] = 0.13585 * v[8];
            y[8] = 0.105 * v[6] + y[24];
            v[1] = 0.999999682931835 * v[0] + 0.000796326710733264 * v[1];
            v[5] = v[1] * v[5];
            v[7] = 0.999999682931835 * v[7] + 0.000796326710733264 * v[0];
            y[25] = 0.13585 * v[7];
            y[9] = 0.105 * v[5] + y[25];
            v[0] = -1. * v[3] + 1.79489656471077e-09 * v[4];
            y[10] = 1.003559 + 0.105 * v[0];
            y[12] = 0.21 * v[6] + y[24];
            y[13] = 0.21 * v[5] + y[25];
            y[14] = 1.003559 + 0.21 * v[0];
            y[16] = 0.315 * v[6] + y[24];
            y[17] = 0.315 * v[5] + y[25];
            y[18] = 1.003559 + 0.315 * v[0];
            y[20] = 0.42 * v[6] + y[24];
            y[21] = 0.42 * v[5] + y[25];
            y[22] = 1.003559 + 0.42 * v[0];
            y[28] = y[24] + -0.1197 * v[8] + 0.425 * v[6];
            y[29] = y[25] + -0.1197 * v[7] + 0.425 * v[5];
            y[30] = 1.003559 + 0.425 * v[0];
            v[3] = -v[3];
            v[9] = 1.79489656471077e-09 * v[4] + v[3];
            v[2] = v[2] * v[9];
            v[10] = sin(x[2]);
            v[11] = cos(x[2]);
            v[12] = v[2] * v[10] + v[6] * v[11];
            y[32] = 0.1 * v[12] + y[28];
            v[9] = v[1] * v[9];
            v[1] = v[9] * v[10] + v[5] * v[11];
            y[33] = 0.1 * v[1] + y[29];
            v[3] = -1. * v[4] + 1.79489656471077e-09 * v[3];
            v[4] = v[3] * v[10] + v[0] * v[11];
            y[34] = 0.1 * v[4] + y[30];
            y[36] = 0.14 * v[12] + y[28];
            y[37] = 0.14 * v[1] + y[29];
            y[38] = 0.14 * v[4] + y[30];
            y[40] = 0.18 * v[12] + y[28];
            y[41] = 0.18 * v[1] + y[29];
            y[42] = 0.18 * v[4] + y[30];
            y[44] = 0.22 * v[12] + y[28];
            y[45] = 0.22 * v[1] + y[29];
            y[46] = 0.22 * v[4] + y[30];
            y[48] = 0.26 * v[12] + y[28];
            y[49] = 0.26 * v[1] + y[29];
            y[50] = 0.26 * v[4] + y[30];
            y[52] = 0.3 * v[12] + y[28];
            y[53] = 0.3 * v[1] + y[29];
            y[54] = 0.3 * v[4] + y[30];
            y[56] = 0.34 * v[12] + y[28];
            y[57] = 0.34 * v[1] + y[29];
            y[58] = 0.34 * v[4] + y[30];
            y[60] = 0.38 * v[12] + y[28];
            y[61] = 0.38 * v[1] + y[29];
            y[62] = 0.38 * v[4] + y[30];
            v[10] = -v[10];
            v[2] = v[2] * v[11] + v[6] * v[10];
            v[13] = sin(x[3]);
            v[14] = cos(x[3]);
            v[15] = 1.79489656471077e-09 * v[13] + v[14];
            v[16] = -1. * v[13] + 1.79489656471077e-09 * v[14];
            v[17] = v[2] * v[15] + v[12] * v[16];
            v[18] = y[28] + 0.39225 * v[12];
            y[64] = 0.09 * v[8] + 0.03 * v[17] + v[18];
            v[9] = v[9] * v[11] + v[5] * v[10];
            v[19] = v[9] * v[15] + v[1] * v[16];
            v[20] = y[29] + 0.39225 * v[1];
            y[65] = 0.09 * v[7] + 0.03 * v[19] + v[20];
            v[10] = v[3] * v[11] + v[0] * v[10];
            v[16] = v[10] * v[15] + v[4] * v[16];
            y[74] = y[30] + 0.39225 * v[4];
            y[66] = 0.03 * v[16] + y[74];
            y[68] = 0.09 * v[8] + -0.03 * v[17] + v[18];
            y[69] = 0.09 * v[7] + -0.03 * v[19] + v[20];
            y[70] = -0.03 * v[16] + y[74];
            y[72] = 0.09 * v[8] + v[18];
            y[73] = 0.09 * v[7] + v[20];
            v[13] = -v[13];
            v[15] = 1.79489656471077e-09 * v[14] + v[13];
            v[13] = -1. * v[14] + 1.79489656471077e-09 * v[13];
            v[2] = v[2] * v[15] + v[12] * v[13];
            v[14] = sin(x[4]);
            v[3] = -v[14];
            v[11] = cos(x[4]);
            v[21] = v[2] * v[3] + v[8] * v[11];
            v[22] = v[18] + 0.093 * v[8];
            y[76] = 0.03 * v[21] + 0.09 * v[17] + v[22];
            v[9] = v[9] * v[15] + v[1] * v[13];
            v[23] = v[9] * v[3] + v[7] * v[11];
            v[24] = v[20] + 0.093 * v[7];
            y[77] = 0.03 * v[23] + 0.09 * v[19] + v[24];
            v[13] = v[10] * v[15] + v[4] * v[13];
            v[3] = v[13] * v[3];
            y[78] = 0.03 * v[3] + 0.09 * v[16] + y[74];
            y[80] = -0.03 * v[21] + 0.09 * v[17] + v[22];
            y[81] = -0.03 * v[23] + 0.09 * v[19] + v[24];
            y[82] = -0.03 * v[3] + 0.09 * v[16] + y[74];
            y[84] = 0.09 * v[17] + v[22];
            y[85] = 0.09 * v[19] + v[24];
            y[86] = 0.09 * v[16] + y[74];
            v[15] = v[22] + 0.09465 * v[17];
            y[88] = 0.06 * v[21] + v[15];
            v[10] = v[24] + 0.09465 * v[19];
            y[89] = 0.06 * v[23] + v[10];
            v[25] = y[74] + 0.09465 * v[16];
            y[90] = 0.06 * v[3] + v[25];
            v[2] = v[2] * v[11] + v[8] * v[14];
            v[26] = cos(x[5]);
            v[27] = sin(x[5]);
            v[28] = -v[27];
            v[29] = v[2] * v[26] + v[17] * v[28];
            y[92] = 1.59265611381251e-05 * v[29] + 0.0973000063413347 * v[21] + v[15];
            v[9] = v[9] * v[11] + v[7] * v[14];
            v[14] = v[9] * v[26] + v[19] * v[28];
            y[93] = 1.59265611381251e-05 * v[14] + 0.0973000063413347 * v[23] + v[10];
            v[13] = v[13] * v[11];
            v[28] = v[13] * v[26] + v[16] * v[28];
            y[94] = 1.59265611381251e-05 * v[28] + 0.0973000063413347 * v[3] + v[25];
            v[2] = v[2] * v[27] + v[17] * v[26];
            y[96] = -4.77794169794995e-05 * v[29] + 0.177299961951912 * v[21] + 0.000547779602643996 * v[2] +
                    v[15];
            v[9] = v[9] * v[27] + v[19] * v[26];
            y[97] = -4.77794169794995e-05 * v[14] + 0.177299961951912 * v[23] + 0.000547779602643996 * v[9] +
                    v[10];
            v[13] = v[13] * v[27] + v[16] * v[26];
            y[98] = -4.77794169794995e-05 * v[28] + 0.177299961951912 * v[3] + 0.000547779602643996 * v[13] +
                    v[25];
            y[100] =
                -1.592643044558e-05 * v[29] + 0.137299987317304 * v[21] + 0.000515926534214666 * v[2] + v[15];
            y[101] =
                -1.592643044558e-05 * v[14] + 0.137299987317304 * v[23] + 0.000515926534214666 * v[9] + v[10];
            y[102] =
                -1.592643044558e-05 * v[28] + 0.137299987317304 * v[3] + 0.000515926534214666 * v[13] + v[25];
            y[104] =
                0.0326288719300173 * v[29] + 0.206633483191718 * v[21] + 0.000571117947718783 * v[2] + v[15];
            y[105] =
                0.0326288719300173 * v[14] + 0.206633483191718 * v[23] + 0.000571117947718783 * v[9] + v[10];
            y[106] =
                0.0326288719300173 * v[28] + 0.206633483191718 * v[3] + 0.000571117947718783 * v[13] + v[25];
            y[108] =
                0.0471739661448351 * v[29] + 0.257142085198042 * v[21] + 0.000611330073064923 * v[2] + v[15];
            y[109] =
                0.0471739661448351 * v[14] + 0.257142085198042 * v[23] + 0.000611330073064923 * v[9] + v[10];
            y[110] =
                0.0471739661448351 * v[28] + 0.257142085198042 * v[3] + 0.000611330073064923 * v[13] + v[25];
            y[112] =
                0.0471938742614188 * v[29] + 0.232142101051412 * v[21] + 0.000591421905296592 * v[2] + v[15];
            y[113] =
                0.0471938742614188 * v[14] + 0.232142101051412 * v[23] + 0.000591421905296592 * v[9] + v[10];
            y[114] =
                0.0471938742614188 * v[28] + 0.232142101051412 * v[3] + 0.000591421905296592 * v[13] + v[25];
            y[116] =
                0.0305511319550535 * v[29] + 0.180116344904696 * v[21] + 0.000550002959776232 * v[2] + v[15];
            y[117] =
                0.0305511319550535 * v[14] + 0.180116344904696 * v[23] + 0.000550002959776232 * v[9] + v[10];
            y[118] =
                0.0305511319550535 * v[28] + 0.180116344904696 * v[3] + 0.000550002959776232 * v[13] + v[25];
            y[120] =
                0.0622118532100359 * v[29] + 0.218207593376579 * v[21] + 0.000580315961208078 * v[2] + v[15];
            y[121] =
                0.0622118532100359 * v[14] + 0.218207593376579 * v[23] + 0.000580315961208078 * v[9] + v[10];
            y[122] =
                0.0622118532100359 * v[28] + 0.218207593376579 * v[3] + 0.000580315961208078 * v[13] + v[25];
            y[124] =
                0.0622437061965698 * v[29] + 0.178207618741971 * v[21] + 0.000548462892778747 * v[2] + v[15];
            y[125] =
                0.0622437061965698 * v[14] + 0.178207618741971 * v[23] + 0.000548462892778747 * v[9] + v[10];
            y[126] =
                0.0622437061965698 * v[28] + 0.178207618741971 * v[3] + 0.000548462892778747 * v[13] + v[25];
            y[128] =
                0.0622277797033028 * v[29] + 0.198207606059275 * v[21] + 0.000564389426993413 * v[2] + v[15];
            y[129] =
                0.0622277797033028 * v[14] + 0.198207606059275 * v[23] + 0.000564389426993413 * v[9] + v[10];
            y[130] =
                0.0622277797033028 * v[28] + 0.198207606059275 * v[3] + 0.000564389426993413 * v[13] + v[25];
            y[132] =
                -0.0327711073338182 * v[29] + 0.206581403542295 * v[21] + 0.000571117947718776 * v[2] + v[15];
            y[133] =
                -0.0327711073338182 * v[14] + 0.206581403542295 * v[23] + 0.000571117947718776 * v[9] + v[10];
            y[134] =
                -0.0327711073338182 * v[28] + 0.206581403542295 * v[3] + 0.000571117947718776 * v[13] + v[25];
            y[136] =
                -0.0469176493992621 * v[29] + 0.257427526148899 * v[21] + 0.000611617044105126 * v[2] + v[15];
            y[137] =
                -0.0469176493992621 * v[14] + 0.257427526148899 * v[23] + 0.000611617044105126 * v[9] + v[10];
            y[138] =
                -0.0469176493992621 * v[28] + 0.257427526148899 * v[3] + 0.000611617044105126 * v[13] + v[25];
            y[140] =
                -0.0468977412826784 * v[29] + 0.232427542002269 * v[21] + 0.000591708876336795 * v[2] + v[15];
            y[141] =
                -0.0468977412826784 * v[14] + 0.232427542002269 * v[23] + 0.000591708876336795 * v[9] + v[10];
            y[142] =
                -0.0468977412826784 * v[28] + 0.232427542002269 * v[3] + 0.000591708876336795 * v[13] + v[25];
            y[144] =
                -0.0306511374916797 * v[29] + 0.180067607997177 * v[21] + 0.000550002959776232 * v[2] + v[15];
            y[145] =
                -0.0306511374916797 * v[14] + 0.180067607997177 * v[23] + 0.000550002959776232 * v[9] + v[10];
            y[146] =
                -0.0306511374916797 * v[28] + 0.180067607997177 * v[3] + 0.000550002959776232 * v[13] + v[25];
            y[148] =
                -0.0623912961612096 * v[29] + 0.218440976014459 * v[21] + 0.000580580825254713 * v[2] + v[15];
            y[149] =
                -0.0623912961612096 * v[14] + 0.218440976014459 * v[23] + 0.000580580825254713 * v[9] + v[10];
            y[150] =
                -0.0623912961612096 * v[28] + 0.218440976014459 * v[3] + 0.000580580825254713 * v[13] + v[25];
            y[152] =
                -0.0623594431746757 * v[29] + 0.178441001379851 * v[21] + 0.000548727756825382 * v[2] + v[15];
            y[153] =
                -0.0623594431746757 * v[14] + 0.178441001379851 * v[23] + 0.000548727756825382 * v[9] + v[10];
            y[154] =
                -0.0623594431746757 * v[28] + 0.178441001379851 * v[3] + 0.000548727756825382 * v[13] + v[25];
            y[156] =
                -0.0623753696679426 * v[29] + 0.198440988697155 * v[21] + 0.000564654291040047 * v[2] + v[15];
            y[157] =
                -0.0623753696679426 * v[14] + 0.198440988697155 * v[23] + 0.000564654291040047 * v[9] + v[10];
            y[158] =
                -0.0623753696679426 * v[28] + 0.198440988697155 * v[3] + 0.000564654291040047 * v[13] + v[25];
            y[168] = 0.209999993443489 * v[6] + y[24];
            y[169] = 0.209999993443489 * v[5] + y[25];
            y[170] = 1.003559 + 0.209999993443489 * v[0];
            y[172] = 0.170000001788139 * v[12] + y[28];
            y[173] = 0.170000001788139 * v[1] + y[29];
            y[174] = 0.170000001788139 * v[4] + y[30];
            y[176] = 0.0900000035762787 * v[8] + v[18];
            y[177] = 0.0900000035762787 * v[7] + v[20];
            y[180] = 0.0900000035762787 * v[17] + v[22];
            y[181] = 0.0900000035762787 * v[19] + v[24];
            y[182] = 0.0900000035762787 * v[16] + y[74];
            y[184] = 0.0599999986588955 * v[21] + v[15];
            y[185] = 0.0599999986588955 * v[23] + v[10];
            y[186] = 0.0599999986588955 * v[3] + v[25];
            y[188] = 1.59265619004145e-05 * v[29] + 0.0973000079393387 * v[21] + v[15];
            y[189] = 1.59265619004145e-05 * v[14] + 0.0973000079393387 * v[23] + v[10];
            y[190] = 1.59265619004145e-05 * v[28] + 0.0973000079393387 * v[3] + v[25];
            y[192] = -3.18529237119947e-05 * v[29] + 0.157299980521202 * v[21] + 0.000531853060238063 * v[2] +
                     v[15];
            y[193] = -3.18529237119947e-05 * v[14] + 0.157299980521202 * v[23] + 0.000531853060238063 * v[9] +
                     v[10];
            y[194] = -3.18529237119947e-05 * v[28] + 0.157299980521202 * v[3] + 0.000531853060238063 * v[13] +
                     v[25];
            y[196] =
                0.0326288715004921 * v[29] + 0.206633478403091 * v[21] + 0.000571117969229817 * v[2] + v[15];
            y[197] =
                0.0326288715004921 * v[14] + 0.206633478403091 * v[23] + 0.000571117969229817 * v[9] + v[10];
            y[198] =
                0.0326288715004921 * v[28] + 0.206633478403091 * v[3] + 0.000571117969229817 * v[13] + v[25];
            y[200] = 0.047183919698 * v[29] + 0.244642093777657 * v[21] + 0.000601375999394804 * v[2] + v[15];
            y[201] = 0.047183919698 * v[14] + 0.244642093777657 * v[23] + 0.000601375999394804 * v[9] + v[10];
            y[202] = 0.047183919698 * v[28] + 0.244642093777657 * v[3] + 0.000601375999394804 * v[13] + v[25];
            y[204] =
                0.0305511318147182 * v[29] + 0.180116340517998 * v[21] + 0.000550002965610474 * v[2] + v[15];
            y[205] =
                0.0305511318147182 * v[14] + 0.180116340517998 * v[23] + 0.000550002965610474 * v[9] + v[10];
            y[206] =
                0.0305511318147182 * v[28] + 0.180116340517998 * v[3] + 0.000550002965610474 * v[13] + v[25];
            y[208] =
                0.0622277781367302 * v[29] + 0.198207601904869 * v[21] + 0.000564389454666525 * v[2] + v[15];
            y[209] =
                0.0622277781367302 * v[14] + 0.198207601904869 * v[23] + 0.000564389454666525 * v[9] + v[10];
            y[210] =
                0.0622277781367302 * v[28] + 0.198207601904869 * v[3] + 0.000564389454666525 * v[13] + v[25];
            y[212] =
                -0.0327711068093777 * v[29] + 0.206581398844719 * v[21] + 0.000571117969229817 * v[2] + v[15];
            y[213] =
                -0.0327711068093777 * v[14] + 0.206581398844719 * v[23] + 0.000571117969229817 * v[9] + v[10];
            y[214] =
                -0.0327711068093777 * v[28] + 0.206581398844719 * v[3] + 0.000571117969229817 * v[13] + v[25];
            y[216] =
                -0.0469076968729496 * v[29] + 0.244927540421486 * v[21] + 0.000601662963163108 * v[2] + v[15];
            y[217] =
                -0.0469076968729496 * v[14] + 0.244927540421486 * v[23] + 0.000601662963163108 * v[9] + v[10];
            y[218] =
                -0.0469076968729496 * v[28] + 0.244927540421486 * v[3] + 0.000601662963163108 * v[13] + v[25];
            y[220] =
                -0.0306511372327805 * v[29] + 0.180067613720894 * v[21] + 0.000550002965610474 * v[2] + v[15];
            y[221] =
                -0.0306511372327805 * v[14] + 0.180067613720894 * v[23] + 0.000550002965610474 * v[9] + v[10];
            y[222] =
                -0.0306511372327805 * v[28] + 0.180067613720894 * v[3] + 0.000550002965610474 * v[13] + v[25];
            y[224] =
                -0.062375370413065 * v[29] + 0.198440983891487 * v[21] + 0.000564654299523681 * v[2] + v[15];
            y[225] =
                -0.062375370413065 * v[14] + 0.198440983891487 * v[23] + 0.000564654299523681 * v[9] + v[10];
            y[226] =
                -0.062375370413065 * v[28] + 0.198440983891487 * v[3] + 0.000564654299523681 * v[13] + v[25];
            y[228] = v[15] + 6.28213797648769e-11 * v[29] + 0.1173 * v[21] + 0.0005 * v[2];
            y[229] = v[10] + 6.28213797648769e-11 * v[14] + 0.1173 * v[23] + 0.0005 * v[9];
            y[230] = v[25] + 6.28213797648769e-11 * v[28] + 0.1173 * v[3] + 0.0005 * v[13];
            y[231] = 0.999999682933264 * v[29] + 0.000796324915837268 * v[21];
            y[232] = 0.999999682933264 * v[14] + 0.000796324915837268 * v[23];
            y[233] = 0.999999682933264 * v[28] + 0.000796324915837268 * v[3];
            y[234] = -6.34134800903635e-07 * v[29] + 0.000796326458244553 * v[21] + -0.999999682931835 * v[2];
            y[235] = -6.34134800903635e-07 * v[14] + 0.000796326458244553 * v[23] + -0.999999682931835 * v[9];
            y[236] = -6.34134800903635e-07 * v[28] + 0.000796326458244553 * v[3] + -0.999999682931835 * v[13];
            y[237] = -0.000796324663347988 * v[29] + 0.999999365865199 * v[21] + 0.000796326710733264 * v[2];
            y[238] = -0.000796324663347988 * v[14] + 0.999999365865199 * v[23] + 0.000796326710733264 * v[9];
            y[239] = -0.000796324663347988 * v[28] + 0.999999365865199 * v[3] + 0.000796326710733264 * v[13];
            // variable duplicates: 1
            y[178] = y[74];
            // dependent variables without operations
            y[0] = 0.;
            y[1] = 0.;
            y[2] = 0.9144;
            y[3] = 0.0799999982118607;
            y[4] = 0.;
            y[5] = 0.;
            y[6] = 1.003559;
            y[7] = 0.0799999982118607;
            y[11] = 0.0799999982118607;
            y[15] = 0.0799999982118607;
            y[19] = 0.0799999982118607;
            y[23] = 0.0799999982118607;
            y[26] = 1.003559;
            y[27] = 0.0799999982118607;
            y[31] = 0.0799999982118607;
            y[35] = 0.0399999991059303;
            y[39] = 0.0399999991059303;
            y[43] = 0.0399999991059303;
            y[47] = 0.0399999991059303;
            y[51] = 0.0399999991059303;
            y[55] = 0.0399999991059303;
            y[59] = 0.0399999991059303;
            y[63] = 0.0399999991059303;
            y[67] = 0.0399999991059303;
            y[71] = 0.0399999991059303;
            y[75] = 0.0399999991059303;
            y[79] = 0.0399999991059303;
            y[83] = 0.0399999991059303;
            y[87] = 0.0399999991059303;
            y[91] = 0.0399999991059303;
            y[95] = 0.0399999991059303;
            y[99] = 0.0399999991059303;
            y[103] = 0.0399999991059303;
            y[107] = 0.0199999995529652;
            y[111] = 0.0149999996647239;
            y[115] = 0.0149999996647239;
            y[119] = 0.0199999995529652;
            y[123] = 0.0149999996647239;
            y[127] = 0.0149999996647239;
            y[131] = 0.0149999996647239;
            y[135] = 0.0199999995529652;
            y[139] = 0.0149999996647239;
            y[143] = 0.0149999996647239;
            y[147] = 0.0199999995529652;
            y[151] = 0.0149999996647239;
            y[155] = 0.0149999996647239;
            y[159] = 0.0149999996647239;
            y[160] = 0.;
            y[161] = 0.;
            y[162] = 0.914399981498718;
            y[163] = 0.0799999982118607;
            y[164] = 0.;
            y[165] = 0.;
            y[166] = 1.003559;
            y[167] = 0.0799999982118607;
            y[171] = 0.28999999165535;
            y[175] = 0.25;
            y[179] = 0.0700000002980232;
            y[183] = 0.0700000002980232;
            y[187] = 0.0399999991059303;
            y[191] = 0.0399999991059303;
            y[195] = 0.0599999986588955;
            y[199] = 0.0199999995529652;
            y[203] = 0.0274999998509884;
            y[207] = 0.0199999995529652;
            y[211] = 0.0350000001490116;
            y[215] = 0.0199999995529652;
            y[219] = 0.0274999998509884;
            y[223] = 0.0199999995529652;
            y[227] = 0.0350000001490116;

            //
            // environment vs. robot collisions
            //

            // robotiq_85_right_finger_link
            if (sphere_environment_in_collision(environment, y[224], y[225], y[226], y[227]))
            {
                if (sphere_environment_in_collision(environment, y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_right_knuckle_link
            if (sphere_environment_in_collision(environment, y[220], y[221], y[222], y[223]))
            {
                if (sphere_environment_in_collision(environment, y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_right_finger_tip_link
            if (sphere_environment_in_collision(environment, y[216], y[217], y[218], y[219]))
            {
                if (sphere_environment_in_collision(environment, y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[140], y[141], y[142], y[143]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_right_inner_knuckle_link
            if (sphere_environment_in_collision(environment, y[212], y[213], y[214], y[215]))
            {
                if (sphere_environment_in_collision(environment, y[132], y[133], y[134], y[135]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_left_finger_link
            if (sphere_environment_in_collision(environment, y[208], y[209], y[210], y[211]))
            {
                if (sphere_environment_in_collision(environment, y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[128], y[129], y[130], y[131]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_left_knuckle_link
            if (sphere_environment_in_collision(environment, y[204], y[205], y[206], y[207]))
            {
                if (sphere_environment_in_collision(environment, y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_left_finger_tip_link
            if (sphere_environment_in_collision(environment, y[200], y[201], y[202], y[203]))
            {
                if (sphere_environment_in_collision(environment, y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_left_inner_knuckle_link
            if (sphere_environment_in_collision(environment, y[196], y[197], y[198], y[199]))
            {
                if (sphere_environment_in_collision(environment, y[104], y[105], y[106], y[107]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // robotiq_85_base_link
            if (sphere_environment_in_collision(environment, y[192], y[193], y[194], y[195]))
            {
                if (sphere_environment_in_collision(environment, y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // fts_robotside
            if (sphere_environment_in_collision(environment, y[188], y[189], y[190], y[191]))
            {
                if (sphere_environment_in_collision(environment, y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // wrist_3_link
            if (sphere_environment_in_collision(environment, y[184], y[185], y[186], y[187]))
            {
                if (sphere_environment_in_collision(environment, y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // wrist_2_link
            if (sphere_environment_in_collision(environment, y[180], y[181], y[182], y[183]))
            {
                if (sphere_environment_in_collision(environment, y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[84], y[85], y[86], y[87]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // wrist_1_link
            if (sphere_environment_in_collision(environment, y[176], y[177], y[178], y[179]))
            {
                if (sphere_environment_in_collision(environment, y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // forearm_link
            if (sphere_environment_in_collision(environment, y[172], y[173], y[174], y[175]))
            {
                if (sphere_environment_in_collision(environment, y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[32], y[33], y[34], y[35]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[36], y[37], y[38], y[39]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // upper_arm_link
            if (sphere_environment_in_collision(environment, y[168], y[169], y[170], y[171]))
            {
                if (sphere_environment_in_collision(environment, y[8], y[9], y[10], y[11]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[12], y[13], y[14], y[15]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[16], y[17], y[18], y[19]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[20], y[21], y[22], y[23]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[24], y[25], y[26], y[27]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // shoulder_link
            if (sphere_environment_in_collision(environment, y[164], y[165], y[166], y[167]))
            {
                if (sphere_environment_in_collision(environment, y[4], y[5], y[6], y[7]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // base_link
            if (sphere_environment_in_collision(environment, y[160], y[161], y[162], y[163]))
            {
                if (sphere_environment_in_collision(environment, y[0], y[1], y[2], y[3]))
                {
                    return false;
                }
            }

            //
            // robot self-collisions
            //

            // base_link vs. upper_arm_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[168], y[169], y[170], y[171]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[8], y[9], y[10], y[11]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[12], y[13], y[14], y[15]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[16], y[17], y[18], y[19]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[20], y[21], y[22], y[23]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[24], y[25], y[26], y[27]))
                {
                    return false;
                }
            }

            // base_link vs. forearm_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[172], y[173], y[174], y[175]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[32], y[33], y[34], y[35]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[36], y[37], y[38], y[39]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            // base_link vs. wrist_1_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[176], y[177], y[178], y[179]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // base_link vs. wrist_2_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[180], y[181], y[182], y[183]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }
            }

            // base_link vs. wrist_3_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[184], y[185], y[186], y[187]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // base_link vs. fts_robotside
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[188], y[189], y[190], y[191]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_base_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[192], y[193], y[194], y[195]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_left_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[196], y[197], y[198], y[199]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_left_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[200], y[201], y[202], y[203]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_left_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[204], y[205], y[206], y[207]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_left_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[208], y[209], y[210], y[211]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_right_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[212], y[213], y[214], y[215]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_right_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[216], y[217], y[218], y[219]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_right_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[220], y[221], y[222], y[223]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // base_link vs. robotiq_85_right_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[224], y[225], y[226], y[227]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // shoulder_link vs. forearm_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[172], y[173], y[174], y[175]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[32], y[33], y[34], y[35]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[36], y[37], y[38], y[39]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            // shoulder_link vs. wrist_1_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[176], y[177], y[178], y[179]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // shoulder_link vs. wrist_2_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[180], y[181], y[182], y[183]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }
            }

            // shoulder_link vs. wrist_3_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[184], y[185], y[186], y[187]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // shoulder_link vs. fts_robotside
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[188], y[189], y[190], y[191]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_base_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[192], y[193], y[194], y[195]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_left_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[196], y[197], y[198], y[199]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_left_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[200], y[201], y[202], y[203]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_left_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[204], y[205], y[206], y[207]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_left_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[208], y[209], y[210], y[211]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_right_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[212], y[213], y[214], y[215]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_right_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[216], y[217], y[218], y[219]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_right_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[220], y[221], y[222], y[223]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // shoulder_link vs. robotiq_85_right_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[224], y[225], y[226], y[227]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. wrist_1_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[176], y[177], y[178], y[179]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. wrist_2_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[180], y[181], y[182], y[183]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. wrist_3_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[184], y[185], y[186], y[187]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. fts_robotside
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[188], y[189], y[190], y[191]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_base_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[192], y[193], y[194], y[195]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_left_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[196], y[197], y[198], y[199]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_left_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[200], y[201], y[202], y[203]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_left_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[204], y[205], y[206], y[207]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_left_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[208], y[209], y[210], y[211]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_right_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[212], y[213], y[214], y[215]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_right_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[216], y[217], y[218], y[219]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_right_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[220], y[221], y[222], y[223]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // upper_arm_link vs. robotiq_85_right_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[224], y[225], y[226], y[227]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // forearm_link vs. wrist_2_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[180], y[181], y[182], y[183]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }
            }

            // forearm_link vs. wrist_3_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[184], y[185], y[186], y[187]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // forearm_link vs. fts_robotside
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[188], y[189], y[190], y[191]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_base_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[192], y[193], y[194], y[195]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_left_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[196], y[197], y[198], y[199]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_left_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[200], y[201], y[202], y[203]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_left_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[204], y[205], y[206], y[207]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_left_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[208], y[209], y[210], y[211]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_right_inner_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[212], y[213], y[214], y[215]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_right_finger_tip_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[216], y[217], y[218], y[219]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_right_knuckle_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[220], y[221], y[222], y[223]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // forearm_link vs. robotiq_85_right_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[224], y[225], y[226], y[227]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // wrist_1_link vs. fts_robotside
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[188], y[189], y[190], y[191]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }
            }

            // attaching at robotiq_85_base_link
            set_attachment_pose(environment, to_isometry(&y[228]));

            //
            // attachment vs. environment collisions
            //
            if (attachment_environment_collision(environment))
            {
                return false;
            }

            //
            // attachment vs. robot collisions
            //

            // Attachment vs. base_link
            if (attachment_sphere_collision<decltype(x[0])>(environment, y[160], y[161], y[162], y[163]))
            {
                if (attachment_sphere_collision<decltype(x[0])>(environment, y[0], y[1], y[2], y[3]))
                {
                    return false;
                }
            }

            // Attachment vs. shoulder_link
            if (attachment_sphere_collision<decltype(x[0])>(environment, y[164], y[165], y[166], y[167]))
            {
                if (attachment_sphere_collision<decltype(x[0])>(environment, y[4], y[5], y[6], y[7]))
                {
                    return false;
                }
            }

            // Attachment vs. upper_arm_link
            if (attachment_sphere_collision<decltype(x[0])>(environment, y[168], y[169], y[170], y[171]))
            {
                if (attachment_sphere_collision<decltype(x[0])>(environment, y[8], y[9], y[10], y[11]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[12], y[13], y[14], y[15]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[16], y[17], y[18], y[19]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[20], y[21], y[22], y[23]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[24], y[25], y[26], y[27]))
                {
                    return false;
                }
            }

            // Attachment vs. forearm_link
            if (attachment_sphere_collision<decltype(x[0])>(environment, y[172], y[173], y[174], y[175]))
            {
                if (attachment_sphere_collision<decltype(x[0])>(environment, y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[32], y[33], y[34], y[35]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[36], y[37], y[38], y[39]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            // Attachment vs. wrist_1_link
            if (attachment_sphere_collision<decltype(x[0])>(environment, y[176], y[177], y[178], y[179]))
            {
                if (attachment_sphere_collision<decltype(x[0])>(environment, y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            return true;
        }

        inline static auto eefk(const std::array<float, 6> &x) noexcept -> Eigen::Isometry3f
        {
            std::array<float, 28> v;
            std::array<float, 12> y;

            v[0] = sin(x[0]);
            v[1] = -v[0];
            v[2] = cos(x[0]);
            v[3] = 0.000796326710733264 * v[1] + -0.999999682931835 * v[2];
            v[4] = 0.000796326710733264 * v[2] + -0.999999682931835 * v[0];
            v[5] = sin(x[1]);
            v[6] = cos(x[1]);
            v[7] = 1.79489656471077e-09 * v[5] + v[6];
            v[8] = v[4] * v[7];
            v[9] = -v[5];
            v[10] = 1.79489656471077e-09 * v[6] + v[9];
            v[4] = v[4] * v[10];
            v[11] = sin(x[2]);
            v[12] = cos(x[2]);
            v[13] = v[4] * v[11] + v[8] * v[12];
            v[14] = -v[11];
            v[4] = v[4] * v[12] + v[8] * v[14];
            v[15] = sin(x[3]);
            v[16] = cos(x[3]);
            v[17] = 1.79489656471077e-09 * v[15] + v[16];
            v[18] = -1. * v[15] + 1.79489656471077e-09 * v[16];
            v[19] = v[4] * v[17] + v[13] * v[18];
            v[15] = -v[15];
            v[20] = 1.79489656471077e-09 * v[16] + v[15];
            v[15] = -1. * v[16] + 1.79489656471077e-09 * v[15];
            v[4] = v[4] * v[20] + v[13] * v[15];
            v[16] = cos(x[4]);
            v[21] = sin(x[4]);
            v[22] = v[4] * v[16] + v[3] * v[21];
            v[23] = cos(x[5]);
            v[24] = sin(x[5]);
            v[25] = -v[24];
            v[26] = v[22] * v[23] + v[19] * v[25];
            v[27] = -v[21];
            v[4] = v[4] * v[27] + v[3] * v[16];
            v[22] = v[22] * v[24] + v[19] * v[23];
            y[0] = 0.13585 * v[3] + -0.1197 * v[3] + 0.425 * v[8] + 0.39225 * v[13] + 0.093 * v[3] +
                   0.09465 * v[19] + 6.28213797648769e-11 * v[26] + 0.1173 * v[4] + 0.0005 * v[22];
            v[1] = 0.999999682931835 * v[1] + 0.000796326710733264 * v[2];
            v[2] = 0.999999682931835 * v[2] + 0.000796326710733264 * v[0];
            v[7] = v[2] * v[7];
            v[2] = v[2] * v[10];
            v[10] = v[2] * v[11] + v[7] * v[12];
            v[2] = v[2] * v[12] + v[7] * v[14];
            v[0] = v[2] * v[17] + v[10] * v[18];
            v[2] = v[2] * v[20] + v[10] * v[15];
            v[21] = v[2] * v[16] + v[1] * v[21];
            v[19] = v[21] * v[23] + v[0] * v[25];
            v[2] = v[2] * v[27] + v[1] * v[16];
            v[21] = v[21] * v[24] + v[0] * v[23];
            y[1] = 0.13585 * v[1] + -0.1197 * v[1] + 0.425 * v[7] + 0.39225 * v[10] + 0.093 * v[1] +
                   0.09465 * v[0] + 6.28213797648769e-11 * v[19] + 0.1173 * v[2] + 0.0005 * v[21];
            v[5] = -1. * v[5] + 1.79489656471077e-09 * v[6];
            v[9] = -1. * v[6] + 1.79489656471077e-09 * v[9];
            v[11] = v[9] * v[11] + v[5] * v[12];
            v[9] = v[9] * v[12] + v[5] * v[14];
            v[18] = v[9] * v[17] + v[11] * v[18];
            v[9] = v[9] * v[20] + v[11] * v[15];
            v[16] = v[9] * v[16];
            v[25] = v[16] * v[23] + v[18] * v[25];
            v[9] = v[9] * v[27];
            v[16] = v[16] * v[24] + v[18] * v[23];
            y[2] = 1.003559 + 0.425 * v[5] + 0.39225 * v[11] + 0.09465 * v[18] +
                   6.28213797648769e-11 * v[25] + 0.1173 * v[9] + 0.0005 * v[16];
            y[3] = 0.999999682933264 * v[26] + 0.000796324915837268 * v[4];
            y[4] = 0.999999682933264 * v[19] + 0.000796324915837268 * v[2];
            y[5] = 0.999999682933264 * v[25] + 0.000796324915837268 * v[9];
            y[6] = -6.34134800903635e-07 * v[26] + 0.000796326458244553 * v[4] + -0.999999682931835 * v[22];
            y[7] = -6.34134800903635e-07 * v[19] + 0.000796326458244553 * v[2] + -0.999999682931835 * v[21];
            y[8] = -6.34134800903635e-07 * v[25] + 0.000796326458244553 * v[9] + -0.999999682931835 * v[16];
            y[9] = -0.000796324663347988 * v[26] + 0.999999365865199 * v[4] + 0.000796326710733264 * v[22];
            y[10] = -0.000796324663347988 * v[19] + 0.999999365865199 * v[2] + 0.000796326710733264 * v[21];
            y[11] = -0.000796324663347988 * v[25] + 0.999999365865199 * v[9] + 0.000796326710733264 * v[16];

            return to_isometry(y.data());
        }
    };
}  // namespace vamp::robots

// NOLINTEND(*-magic-numbers)
