#pragma once

#include <vamp/vector.hh>
#include <vamp/vector/math.hh>
#include <vamp/collision/environment.hh>
#include <vamp/collision/validity.hh>

// NOLINTBEGIN(*-magic-numbers)
namespace vamp::robots
{
    struct Baxter
    {
        static constexpr char *name = "baxter";
        static constexpr std::size_t dimension = 14;
        static constexpr std::size_t n_spheres = 75;
        static constexpr float min_radius = 0.012000000104308128;
        static constexpr float max_radius = 0.5;
        static constexpr std::size_t resolution = 64;

        static constexpr std::array<std::string_view, dimension> joint_names = {
            "left_s0",
            "left_s1",
            "left_e0",
            "left_e1",
            "left_w0",
            "left_w1",
            "left_w2",
            "right_s0",
            "right_s1",
            "right_e0",
            "right_e1",
            "right_w0",
            "right_w1",
            "right_w2"};
        static constexpr char *end_effector = "right_gripper";

        using Configuration = FloatVector<dimension>;
        using ConfigurationArray = std::array<FloatT, dimension>;

        struct alignas(FloatVectorAlignment) ConfigurationBuffer
          : std::array<float, Configuration::num_scalars_rounded>
        {
        };

        template <std::size_t rake>
        using ConfigurationBlock = FloatVector<rake, dimension>;

        template <std::size_t rake>
        struct Spheres
        {
            FloatVector<rake, n_spheres> x;
            FloatVector<rake, n_spheres> y;
            FloatVector<rake, n_spheres> z;
            FloatVector<rake, n_spheres> r;
        };

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> s_m{
            3.403359889984131,
            3.194000005722046,
            6.1083598136901855,
            2.6679999828338623,
            6.118000030517578,
            3.6647963523864746,
            6.118000030517578,
            3.403359889984131,
            3.194000005722046,
            6.1083598136901855,
            2.6679999828338623,
            6.118000030517578,
            3.6647963523864746,
            6.118000030517578};

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> s_a{
            -1.7016799449920654,
            -2.1470000743865967,
            -3.0541799068450928,
            -0.05000000074505806,
            -3.059000015258789,
            -1.5707963705062866,
            -3.059000015258789,
            -1.7016799449920654,
            -2.1470000743865967,
            -3.0541799068450928,
            -0.05000000074505806,
            -3.059000015258789,
            -1.5707963705062866,
            -3.059000015258789};

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> d_m{
            0.2938272953033447,
            0.3130870461463928,
            0.1637100726366043,
            0.37481260299682617,
            0.16345210373401642,
            0.27286645770072937,
            0.16345210373401642,
            0.2938272953033447,
            0.3130870461463928,
            0.1637100726366043,
            0.37481260299682617,
            0.16345210373401642,
            0.27286645770072937,
            0.16345210373401642};

        inline static void scale_configuration(Configuration &q) noexcept
        {
            q = q * Configuration(s_m) + Configuration(s_a);
        }

        inline static void descale_configuration(Configuration &q) noexcept
        {
            q = (q - Configuration(s_a)) * Configuration(d_m);
        }

        template <std::size_t rake>
        inline static void scale_configuration_block(ConfigurationBlock<rake> &q) noexcept
        {
            q[0] = -1.7016799449920654 + (q[0] * 3.403359889984131);
            q[1] = -2.1470000743865967 + (q[1] * 3.194000005722046);
            q[2] = -3.0541799068450928 + (q[2] * 6.1083598136901855);
            q[3] = -0.05000000074505806 + (q[3] * 2.6679999828338623);
            q[4] = -3.059000015258789 + (q[4] * 6.118000030517578);
            q[5] = -1.5707963705062866 + (q[5] * 3.6647963523864746);
            q[6] = -3.059000015258789 + (q[6] * 6.118000030517578);
            q[7] = -1.7016799449920654 + (q[7] * 3.403359889984131);
            q[8] = -2.1470000743865967 + (q[8] * 3.194000005722046);
            q[9] = -3.0541799068450928 + (q[9] * 6.1083598136901855);
            q[10] = -0.05000000074505806 + (q[10] * 2.6679999828338623);
            q[11] = -3.059000015258789 + (q[11] * 6.118000030517578);
            q[12] = -1.5707963705062866 + (q[12] * 3.6647963523864746);
            q[13] = -3.059000015258789 + (q[13] * 6.118000030517578);
        }

        template <std::size_t rake>
        inline static void descale_configuration_block(ConfigurationBlock<rake> &q) noexcept
        {
            q[0] = 0.2938272953033447 * (q[0] - -1.7016799449920654);
            q[1] = 0.3130870461463928 * (q[1] - -2.1470000743865967);
            q[2] = 0.1637100726366043 * (q[2] - -3.0541799068450928);
            q[3] = 0.37481260299682617 * (q[3] - -0.05000000074505806);
            q[4] = 0.16345210373401642 * (q[4] - -3.059000015258789);
            q[5] = 0.27286645770072937 * (q[5] - -1.5707963705062866);
            q[6] = 0.16345210373401642 * (q[6] - -3.059000015258789);
            q[7] = 0.2938272953033447 * (q[7] - -1.7016799449920654);
            q[8] = 0.3130870461463928 * (q[8] - -2.1470000743865967);
            q[9] = 0.1637100726366043 * (q[9] - -3.0541799068450928);
            q[10] = 0.37481260299682617 * (q[10] - -0.05000000074505806);
            q[11] = 0.16345210373401642 * (q[11] - -3.059000015258789);
            q[12] = 0.27286645770072937 * (q[12] - -1.5707963705062866);
            q[13] = 0.16345210373401642 * (q[13] - -3.059000015258789);
        }

        inline static auto space_measure() noexcept -> float
        {
            return 590532810.7756369;
        }

        template <std::size_t rake>
        inline static void sphere_fk(const ConfigurationBlock<rake> &x, Spheres<rake> &out) noexcept
        {
            std::array<FloatVector<rake, 1>, 23> v;
            std::array<FloatVector<rake, 1>, 300> y;

            v[0] = cos(x[0]);
            v[1] = sin(x[0]);
            v[2] = 0.707105482511236 * v[0] + -0.707108079859474 * v[1];
            y[24] = 0.0640272398484633 + 0.069 * v[2];
            v[3] = 0.707108079859474 * v[0] + 0.707105482511236 * v[1];
            y[25] = 0.259027384507773 + 0.069 * v[3];
            v[4] = cos(x[1]);
            v[1] = -v[1];
            v[5] = 0.707105482511236 * v[1] + -0.707108079859474 * v[0];
            v[6] = sin(x[1]);
            v[7] = 4.89663865010925e-12 * v[6];
            v[8] = v[2] * v[4] + v[5] * v[7];
            v[9] = cos(x[2]);
            v[10] = sin(x[2]);
            v[11] = 4.89663865010925e-12 * v[9] + -4.89658313895802e-12 * v[10];
            v[12] = -v[6];
            v[13] = 4.89663865010925e-12 * v[4];
            v[2] = v[2] * v[12] + v[5] * v[13];
            v[14] = 5.55111512312578e-17 * v[9] + v[10];
            v[15] = v[8] * v[11] + v[2] * v[9] + v[5] * v[14];
            v[16] = v[8] + -4.89658313895802e-12 * v[2] + 4.89663865010925e-12 * v[5];
            y[36] = y[24] + 0.102 * v[8];
            y[28] = -0.02 * v[15] + 0.22 * v[16] + y[36];
            v[1] = 0.707108079859474 * v[1] + 0.707105482511236 * v[0];
            v[7] = v[3] * v[4] + v[1] * v[7];
            v[13] = v[3] * v[12] + v[1] * v[13];
            v[12] = v[7] * v[11] + v[13] * v[9] + v[1] * v[14];
            v[3] = v[7] + -4.89658313895802e-12 * v[13] + 4.89663865010925e-12 * v[1];
            y[37] = y[25] + 0.102 * v[7];
            y[29] = -0.02 * v[12] + 0.22 * v[3] + y[37];
            v[6] = -1. * v[6];
            v[4] = -1. * v[4];
            v[14] = v[6] * v[11] + v[4] * v[9] + 4.89663865010925e-12 * v[14];
            v[11] = v[6] + 2.39770700697438e-23 + -4.89658313895802e-12 * v[4];
            y[38] = 0.399976 + 0.102 * v[6];
            y[30] = -0.02 * v[14] + 0.22 * v[11] + y[38];
            y[32] = -0.01 * v[15] + 0.11 * v[16] + y[36];
            y[33] = -0.01 * v[12] + 0.11 * v[3] + y[37];
            y[34] = -0.01 * v[14] + 0.11 * v[11] + y[38];
            y[40] = y[36] + 0.069 * v[15] + 0.26242 * v[16];
            y[41] = y[37] + 0.069 * v[12] + 0.26242 * v[3];
            y[42] = y[38] + 0.069 * v[14] + 0.26242 * v[11];
            v[0] = cos(x[3]);
            v[17] = sin(x[3]);
            v[18] = 4.89663865010925e-12 * v[0] + v[17];
            v[10] = -v[10];
            v[19] = 4.89663865010925e-12 * v[10] + -4.89658313895802e-12 * v[9];
            v[9] = 5.55111512312578e-17 * v[10] + v[9];
            v[2] = v[8] * v[19] + v[2] * v[10] + v[5] * v[9];
            v[8] = 5.55111512312578e-17 * v[0] + 4.89663865010925e-12 * v[17];
            v[5] = v[0] + -4.89658313895802e-12 * v[17];
            v[20] = v[15] * v[18] + v[2] * v[8] + v[16] * v[5];
            y[44] = y[40] + 0.10359 * v[20];
            v[13] = v[7] * v[19] + v[13] * v[10] + v[1] * v[9];
            v[7] = v[12] * v[18] + v[13] * v[8] + v[3] * v[5];
            y[45] = y[41] + 0.10359 * v[7];
            v[9] = v[6] * v[19] + v[4] * v[10] + 4.89663865010925e-12 * v[9];
            v[5] = v[14] * v[18] + v[9] * v[8] + v[11] * v[5];
            y[46] = y[42] + 0.10359 * v[5];
            v[17] = -v[17];
            v[8] = 4.89663865010925e-12 * v[17] + v[0];
            v[18] = 5.55111512312578e-17 * v[17] + 4.89663865010925e-12 * v[0];
            v[17] = v[17] + -4.89658313895802e-12 * v[0];
            v[16] = v[15] * v[8] + v[2] * v[18] + v[16] * v[17];
            v[2] = -4.89658313895802e-12 * v[15] + v[2];
            v[15] = v[20] + -4.89658313895802e-12 * v[16] + 4.89663865010925e-12 * v[2];
            y[48] = 0.22 * v[15] + y[44];
            v[3] = v[12] * v[8] + v[13] * v[18] + v[3] * v[17];
            v[13] = -4.89658313895802e-12 * v[12] + v[13];
            v[12] = v[7] + -4.89658313895802e-12 * v[3] + 4.89663865010925e-12 * v[13];
            y[49] = 0.22 * v[12] + y[45];
            v[17] = v[14] * v[8] + v[9] * v[18] + v[11] * v[17];
            v[9] = -4.89658313895802e-12 * v[14] + v[9];
            v[14] = v[5] + -4.89658313895802e-12 * v[17] + 4.89663865010925e-12 * v[9];
            y[50] = 0.22 * v[14] + y[46];
            y[52] = 0.11 * v[15] + y[44];
            y[53] = 0.11 * v[12] + y[45];
            y[54] = 0.11 * v[14] + y[46];
            v[18] = cos(x[4]);
            v[8] = sin(x[4]);
            v[11] = 4.89663865010925e-12 * v[18] + -4.89658313895802e-12 * v[8];
            v[0] = 5.55111512312578e-17 * v[18] + v[8];
            v[19] = v[20] * v[11] + v[16] * v[18] + v[2] * v[0];
            v[8] = -v[8];
            v[10] = 4.89663865010925e-12 * v[8] + -4.89658313895802e-12 * v[18];
            v[4] = 5.55111512312578e-17 * v[8] + v[18];
            v[2] = v[20] * v[10] + v[16] * v[8] + v[2] * v[4];
            v[16] = -4.89658313895802e-12 * v[19] + v[2];
            v[20] = y[44] + 0.01 * v[19] + 0.2707 * v[15];
            y[56] = 0.03 * v[16] + v[20];
            v[6] = v[7] * v[11] + v[3] * v[18] + v[13] * v[0];
            v[13] = v[7] * v[10] + v[3] * v[8] + v[13] * v[4];
            v[3] = -4.89658313895802e-12 * v[6] + v[13];
            v[7] = y[45] + 0.01 * v[6] + 0.2707 * v[12];
            y[57] = 0.03 * v[3] + v[7];
            v[0] = v[5] * v[11] + v[17] * v[18] + v[9] * v[0];
            v[4] = v[5] * v[10] + v[17] * v[8] + v[9] * v[4];
            v[10] = -4.89658313895802e-12 * v[0] + v[4];
            v[8] = y[46] + 0.01 * v[0] + 0.2707 * v[14];
            y[58] = 0.03 * v[10] + v[8];
            y[60] = -0.03 * v[16] + v[20];
            y[61] = -0.03 * v[3] + v[7];
            y[62] = -0.03 * v[10] + v[8];
            v[9] = cos(x[5]);
            v[17] = sin(x[5]);
            v[5] = 4.89663865010925e-12 * v[9] + v[17];
            v[11] = 5.55111512312578e-17 * v[9] + 4.89663865010925e-12 * v[17];
            v[18] = v[9] + -4.89658313895802e-12 * v[17];
            v[1] = v[19] * v[5] + v[2] * v[11] + v[15] * v[18];
            v[17] = -v[17];
            v[21] = 4.89663865010925e-12 * v[17] + v[9];
            v[22] = 5.55111512312578e-17 * v[17] + 4.89663865010925e-12 * v[9];
            v[17] = v[17] + -4.89658313895802e-12 * v[9];
            v[2] = v[19] * v[21] + v[2] * v[22] + v[15] * v[17];
            v[19] = v[1] + -4.89658313895802e-12 * v[2] + 4.89663865010925e-12 * v[16];
            v[20] = v[20] + 0.115975 * v[1];
            y[64] = 0.02 * v[19] + v[20];
            v[15] = v[6] * v[5] + v[13] * v[11] + v[12] * v[18];
            v[13] = v[6] * v[21] + v[13] * v[22] + v[12] * v[17];
            v[6] = v[15] + -4.89658313895802e-12 * v[13] + 4.89663865010925e-12 * v[3];
            v[7] = v[7] + 0.115975 * v[15];
            y[65] = 0.02 * v[6] + v[7];
            v[18] = v[0] * v[5] + v[4] * v[11] + v[14] * v[18];
            v[17] = v[0] * v[21] + v[4] * v[22] + v[14] * v[17];
            v[22] = v[18] + -4.89658313895802e-12 * v[17] + 4.89663865010925e-12 * v[10];
            v[8] = v[8] + 0.115975 * v[18];
            y[66] = 0.02 * v[22] + v[8];
            y[68] = -0.04 * v[19] + v[20];
            y[69] = -0.04 * v[6] + v[7];
            y[70] = -0.04 * v[22] + v[8];
            v[21] = cos(x[6]);
            v[4] = sin(x[6]);
            v[0] = 4.89663865010925e-12 * v[21] + -4.89658313895802e-12 * v[4];
            v[14] = 5.55111512312578e-17 * v[21] + v[4];
            v[11] = v[1] * v[0] + v[2] * v[21] + v[16] * v[14];
            y[72] = 0.01 * v[11] + 0.09355 * v[19] + v[20];
            v[5] = v[15] * v[0] + v[13] * v[21] + v[3] * v[14];
            y[73] = 0.01 * v[5] + 0.09355 * v[6] + v[7];
            v[14] = v[18] * v[0] + v[17] * v[21] + v[10] * v[14];
            y[74] = 0.01 * v[14] + 0.09355 * v[22] + v[8];
            v[4] = -v[4];
            v[0] = 4.89663865010925e-12 * v[4] + -4.89658313895802e-12 * v[21];
            v[21] = 5.55111512312578e-17 * v[4] + v[21];
            v[2] = v[1] * v[0] + v[2] * v[4] + v[16] * v[21];
            y[76] = 0.02 * v[2] + 0.13855 * v[19] + v[20];
            v[13] = v[15] * v[0] + v[13] * v[4] + v[3] * v[21];
            y[77] = 0.02 * v[13] + 0.13855 * v[6] + v[7];
            v[21] = v[18] * v[0] + v[17] * v[4] + v[10] * v[21];
            y[78] = 0.02 * v[21] + 0.13855 * v[22] + v[8];
            y[80] = -0.02 * v[2] + 0.13855 * v[19] + v[20];
            y[81] = -0.02 * v[13] + 0.13855 * v[6] + v[7];
            y[82] = -0.02 * v[21] + 0.13855 * v[22] + v[8];
            y[84] = -0.005 * v[11] + 0.081333 * v[2] + 0.16655 * v[19] + v[20];
            y[85] = -0.005 * v[5] + 0.081333 * v[13] + 0.16655 * v[6] + v[7];
            y[86] = -0.005 * v[14] + 0.081333 * v[21] + 0.16655 * v[22] + v[8];
            y[88] = -0.005 * v[11] + 0.057333 * v[2] + 0.16655 * v[19] + v[20];
            y[89] = -0.005 * v[5] + 0.057333 * v[13] + 0.16655 * v[6] + v[7];
            y[90] = -0.005 * v[14] + 0.057333 * v[21] + 0.16655 * v[22] + v[8];
            y[92] = 0.086583 * v[2] + 0.18855 * v[19] + v[20];
            y[93] = 0.086583 * v[13] + 0.18855 * v[6] + v[7];
            y[94] = 0.086583 * v[21] + 0.18855 * v[22] + v[8];
            y[96] = 0.086583 * v[2] + 0.20855 * v[19] + v[20];
            y[97] = 0.086583 * v[13] + 0.20855 * v[6] + v[7];
            y[98] = 0.086583 * v[21] + 0.20855 * v[22] + v[8];
            y[100] = 0.086583 * v[2] + 0.22855 * v[19] + v[20];
            y[101] = 0.086583 * v[13] + 0.22855 * v[6] + v[7];
            y[102] = 0.086583 * v[21] + 0.22855 * v[22] + v[8];
            y[104] = 0.01 * v[11] + 0.082083 * v[2] + 0.26625 * v[19] + v[20];
            y[105] = 0.01 * v[5] + 0.082083 * v[13] + 0.26625 * v[6] + v[7];
            y[106] = 0.01 * v[14] + 0.082083 * v[21] + 0.26625 * v[22] + v[8];
            y[108] = -0.01 * v[11] + 0.082083 * v[2] + 0.26625 * v[19] + v[20];
            y[109] = -0.01 * v[5] + 0.082083 * v[13] + 0.26625 * v[6] + v[7];
            y[110] = -0.01 * v[14] + 0.082083 * v[21] + 0.26625 * v[22] + v[8];
            y[112] = -0.01 * v[11] + 0.082083 * v[2] + 0.24625 * v[19] + v[20];
            y[113] = -0.01 * v[5] + 0.082083 * v[13] + 0.24625 * v[6] + v[7];
            y[114] = -0.01 * v[14] + 0.082083 * v[21] + 0.24625 * v[22] + v[8];
            y[116] = 0.01 * v[11] + 0.082083 * v[2] + 0.24625 * v[19] + v[20];
            y[117] = 0.01 * v[5] + 0.082083 * v[13] + 0.24625 * v[6] + v[7];
            y[118] = 0.01 * v[14] + 0.082083 * v[21] + 0.24625 * v[22] + v[8];
            y[120] = 0.005 * v[11] + -0.059333 * v[2] + 0.16655 * v[19] + v[20];
            y[121] = 0.005 * v[5] + -0.059333 * v[13] + 0.16655 * v[6] + v[7];
            y[122] = 0.005 * v[14] + -0.059333 * v[21] + 0.16655 * v[22] + v[8];
            y[124] = 0.005 * v[11] + -0.079333 * v[2] + 0.16655 * v[19] + v[20];
            y[125] = 0.005 * v[5] + -0.079333 * v[13] + 0.16655 * v[6] + v[7];
            y[126] = 0.005 * v[14] + -0.079333 * v[21] + 0.16655 * v[22] + v[8];
            y[128] = -0.086583 * v[2] + 0.18855 * v[19] + v[20];
            y[129] = -0.086583 * v[13] + 0.18855 * v[6] + v[7];
            y[130] = -0.086583 * v[21] + 0.18855 * v[22] + v[8];
            y[132] = -0.086583 * v[2] + 0.20855 * v[19] + v[20];
            y[133] = -0.086583 * v[13] + 0.20855 * v[6] + v[7];
            y[134] = -0.086583 * v[21] + 0.20855 * v[22] + v[8];
            y[136] = -0.086583 * v[2] + 0.22855 * v[19] + v[20];
            y[137] = -0.086583 * v[13] + 0.22855 * v[6] + v[7];
            y[138] = -0.086583 * v[21] + 0.22855 * v[22] + v[8];
            y[140] = 0.01 * v[11] + -0.082083 * v[2] + 0.26625 * v[19] + v[20];
            y[141] = 0.01 * v[5] + -0.082083 * v[13] + 0.26625 * v[6] + v[7];
            y[142] = 0.01 * v[14] + -0.082083 * v[21] + 0.26625 * v[22] + v[8];
            y[144] = -0.01 * v[11] + -0.082083 * v[2] + 0.26625 * v[19] + v[20];
            y[145] = -0.01 * v[5] + -0.082083 * v[13] + 0.26625 * v[6] + v[7];
            y[146] = -0.01 * v[14] + -0.082083 * v[21] + 0.26625 * v[22] + v[8];
            y[148] = -0.01 * v[11] + -0.082083 * v[2] + 0.24625 * v[19] + v[20];
            y[149] = -0.01 * v[5] + -0.082083 * v[13] + 0.24625 * v[6] + v[7];
            y[150] = -0.01 * v[14] + -0.082083 * v[21] + 0.24625 * v[22] + v[8];
            y[152] = 0.01 * v[11] + -0.082083 * v[2] + 0.24625 * v[19] + v[20];
            y[153] = 0.01 * v[5] + -0.082083 * v[13] + 0.24625 * v[6] + v[7];
            y[154] = 0.01 * v[14] + -0.082083 * v[21] + 0.24625 * v[22] + v[8];
            v[21] = cos(x[7]);
            v[14] = sin(x[7]);
            v[8] = 0.707105482511236 * v[21] + 0.707108079859474 * v[14];
            y[168] = 0.0640272398484633 + 0.069 * v[8];
            v[22] = -0.707108079859474 * v[21] + 0.707105482511236 * v[14];
            y[169] = -0.259027384507773 + 0.069 * v[22];
            v[13] = cos(x[8]);
            v[14] = -v[14];
            v[5] = 0.707105482511236 * v[14] + 0.707108079859474 * v[21];
            v[7] = sin(x[8]);
            v[6] = 4.89663865010925e-12 * v[7];
            v[2] = v[8] * v[13] + v[5] * v[6];
            v[11] = cos(x[9]);
            v[20] = sin(x[9]);
            v[19] = 4.89663865010925e-12 * v[11] + -4.89658313895802e-12 * v[20];
            v[0] = -v[7];
            v[4] = 4.89663865010925e-12 * v[13];
            v[8] = v[8] * v[0] + v[5] * v[4];
            v[17] = 5.55111512312578e-17 * v[11] + v[20];
            v[18] = v[2] * v[19] + v[8] * v[11] + v[5] * v[17];
            v[10] = v[2] + -4.89658313895802e-12 * v[8] + 4.89663865010925e-12 * v[5];
            y[180] = y[168] + 0.102 * v[2];
            y[172] = -0.02 * v[18] + 0.22 * v[10] + y[180];
            v[14] = -0.707108079859474 * v[14] + 0.707105482511236 * v[21];
            v[6] = v[22] * v[13] + v[14] * v[6];
            v[4] = v[22] * v[0] + v[14] * v[4];
            v[0] = v[6] * v[19] + v[4] * v[11] + v[14] * v[17];
            v[22] = v[6] + -4.89658313895802e-12 * v[4] + 4.89663865010925e-12 * v[14];
            y[181] = y[169] + 0.102 * v[6];
            y[173] = -0.02 * v[0] + 0.22 * v[22] + y[181];
            v[7] = -1. * v[7];
            v[13] = -1. * v[13];
            v[17] = v[7] * v[19] + v[13] * v[11] + 4.89663865010925e-12 * v[17];
            v[19] = v[7] + 2.39770700697438e-23 + -4.89658313895802e-12 * v[13];
            y[182] = 0.399976 + 0.102 * v[7];
            y[174] = -0.02 * v[17] + 0.22 * v[19] + y[182];
            y[176] = -0.01 * v[18] + 0.11 * v[10] + y[180];
            y[177] = -0.01 * v[0] + 0.11 * v[22] + y[181];
            y[178] = -0.01 * v[17] + 0.11 * v[19] + y[182];
            y[184] = y[180] + 0.069 * v[18] + 0.26242 * v[10];
            y[185] = y[181] + 0.069 * v[0] + 0.26242 * v[22];
            y[186] = y[182] + 0.069 * v[17] + 0.26242 * v[19];
            v[21] = cos(x[10]);
            v[15] = sin(x[10]);
            v[3] = 4.89663865010925e-12 * v[21] + v[15];
            v[20] = -v[20];
            v[1] = 4.89663865010925e-12 * v[20] + -4.89658313895802e-12 * v[11];
            v[11] = 5.55111512312578e-17 * v[20] + v[11];
            v[8] = v[2] * v[1] + v[8] * v[20] + v[5] * v[11];
            v[2] = 5.55111512312578e-17 * v[21] + 4.89663865010925e-12 * v[15];
            v[5] = v[21] + -4.89658313895802e-12 * v[15];
            v[16] = v[18] * v[3] + v[8] * v[2] + v[10] * v[5];
            y[188] = y[184] + 0.10359 * v[16];
            v[4] = v[6] * v[1] + v[4] * v[20] + v[14] * v[11];
            v[6] = v[0] * v[3] + v[4] * v[2] + v[22] * v[5];
            y[189] = y[185] + 0.10359 * v[6];
            v[11] = v[7] * v[1] + v[13] * v[20] + 4.89663865010925e-12 * v[11];
            v[5] = v[17] * v[3] + v[11] * v[2] + v[19] * v[5];
            y[190] = y[186] + 0.10359 * v[5];
            v[15] = -v[15];
            v[2] = 4.89663865010925e-12 * v[15] + v[21];
            v[3] = 5.55111512312578e-17 * v[15] + 4.89663865010925e-12 * v[21];
            v[15] = v[15] + -4.89658313895802e-12 * v[21];
            v[10] = v[18] * v[2] + v[8] * v[3] + v[10] * v[15];
            v[8] = -4.89658313895802e-12 * v[18] + v[8];
            v[18] = v[16] + -4.89658313895802e-12 * v[10] + 4.89663865010925e-12 * v[8];
            y[192] = 0.22 * v[18] + y[188];
            v[22] = v[0] * v[2] + v[4] * v[3] + v[22] * v[15];
            v[4] = -4.89658313895802e-12 * v[0] + v[4];
            v[0] = v[6] + -4.89658313895802e-12 * v[22] + 4.89663865010925e-12 * v[4];
            y[193] = 0.22 * v[0] + y[189];
            v[15] = v[17] * v[2] + v[11] * v[3] + v[19] * v[15];
            v[11] = -4.89658313895802e-12 * v[17] + v[11];
            v[17] = v[5] + -4.89658313895802e-12 * v[15] + 4.89663865010925e-12 * v[11];
            y[194] = 0.22 * v[17] + y[190];
            y[196] = 0.11 * v[18] + y[188];
            y[197] = 0.11 * v[0] + y[189];
            y[198] = 0.11 * v[17] + y[190];
            v[3] = cos(x[11]);
            v[2] = sin(x[11]);
            v[19] = 4.89663865010925e-12 * v[3] + -4.89658313895802e-12 * v[2];
            v[21] = 5.55111512312578e-17 * v[3] + v[2];
            v[1] = v[16] * v[19] + v[10] * v[3] + v[8] * v[21];
            v[2] = -v[2];
            v[20] = 4.89663865010925e-12 * v[2] + -4.89658313895802e-12 * v[3];
            v[13] = 5.55111512312578e-17 * v[2] + v[3];
            v[8] = v[16] * v[20] + v[10] * v[2] + v[8] * v[13];
            v[10] = -4.89658313895802e-12 * v[1] + v[8];
            v[16] = y[188] + 0.01 * v[1] + 0.2707 * v[18];
            y[200] = 0.03 * v[10] + v[16];
            v[7] = v[6] * v[19] + v[22] * v[3] + v[4] * v[21];
            v[4] = v[6] * v[20] + v[22] * v[2] + v[4] * v[13];
            v[22] = -4.89658313895802e-12 * v[7] + v[4];
            v[6] = y[189] + 0.01 * v[7] + 0.2707 * v[0];
            y[201] = 0.03 * v[22] + v[6];
            v[21] = v[5] * v[19] + v[15] * v[3] + v[11] * v[21];
            v[13] = v[5] * v[20] + v[15] * v[2] + v[11] * v[13];
            v[20] = -4.89658313895802e-12 * v[21] + v[13];
            v[2] = y[190] + 0.01 * v[21] + 0.2707 * v[17];
            y[202] = 0.03 * v[20] + v[2];
            y[204] = -0.03 * v[10] + v[16];
            y[205] = -0.03 * v[22] + v[6];
            y[206] = -0.03 * v[20] + v[2];
            v[11] = cos(x[12]);
            v[15] = sin(x[12]);
            v[5] = 4.89663865010925e-12 * v[11] + v[15];
            v[19] = 5.55111512312578e-17 * v[11] + 4.89663865010925e-12 * v[15];
            v[3] = v[11] + -4.89658313895802e-12 * v[15];
            v[14] = v[1] * v[5] + v[8] * v[19] + v[18] * v[3];
            v[15] = -v[15];
            v[12] = 4.89663865010925e-12 * v[15] + v[11];
            v[9] = 5.55111512312578e-17 * v[15] + 4.89663865010925e-12 * v[11];
            v[15] = v[15] + -4.89658313895802e-12 * v[11];
            v[8] = v[1] * v[12] + v[8] * v[9] + v[18] * v[15];
            v[1] = v[14] + -4.89658313895802e-12 * v[8] + 4.89663865010925e-12 * v[10];
            v[16] = v[16] + 0.115975 * v[14];
            y[208] = 0.02 * v[1] + v[16];
            v[18] = v[7] * v[5] + v[4] * v[19] + v[0] * v[3];
            v[4] = v[7] * v[12] + v[4] * v[9] + v[0] * v[15];
            v[7] = v[18] + -4.89658313895802e-12 * v[4] + 4.89663865010925e-12 * v[22];
            v[6] = v[6] + 0.115975 * v[18];
            y[209] = 0.02 * v[7] + v[6];
            v[3] = v[21] * v[5] + v[13] * v[19] + v[17] * v[3];
            v[15] = v[21] * v[12] + v[13] * v[9] + v[17] * v[15];
            v[9] = v[3] + -4.89658313895802e-12 * v[15] + 4.89663865010925e-12 * v[20];
            v[2] = v[2] + 0.115975 * v[3];
            y[210] = 0.02 * v[9] + v[2];
            y[212] = -0.04 * v[1] + v[16];
            y[213] = -0.04 * v[7] + v[6];
            y[214] = -0.04 * v[9] + v[2];
            v[12] = cos(x[13]);
            v[13] = sin(x[13]);
            v[21] = 4.89663865010925e-12 * v[12] + -4.89658313895802e-12 * v[13];
            v[17] = 5.55111512312578e-17 * v[12] + v[13];
            v[19] = v[14] * v[21] + v[8] * v[12] + v[10] * v[17];
            y[216] = 0.01 * v[19] + 0.09355 * v[1] + v[16];
            v[5] = v[18] * v[21] + v[4] * v[12] + v[22] * v[17];
            y[217] = 0.01 * v[5] + 0.09355 * v[7] + v[6];
            v[17] = v[3] * v[21] + v[15] * v[12] + v[20] * v[17];
            y[218] = 0.01 * v[17] + 0.09355 * v[9] + v[2];
            v[13] = -v[13];
            v[21] = 4.89663865010925e-12 * v[13] + -4.89658313895802e-12 * v[12];
            v[12] = 5.55111512312578e-17 * v[13] + v[12];
            v[8] = v[14] * v[21] + v[8] * v[13] + v[10] * v[12];
            y[220] = 0.02 * v[8] + 0.13855 * v[1] + v[16];
            v[4] = v[18] * v[21] + v[4] * v[13] + v[22] * v[12];
            y[221] = 0.02 * v[4] + 0.13855 * v[7] + v[6];
            v[12] = v[3] * v[21] + v[15] * v[13] + v[20] * v[12];
            y[222] = 0.02 * v[12] + 0.13855 * v[9] + v[2];
            y[224] = -0.02 * v[8] + 0.13855 * v[1] + v[16];
            y[225] = -0.02 * v[4] + 0.13855 * v[7] + v[6];
            y[226] = -0.02 * v[12] + 0.13855 * v[9] + v[2];
            y[228] = -0.005 * v[19] + 0.081333 * v[8] + 0.16655 * v[1] + v[16];
            y[229] = -0.005 * v[5] + 0.081333 * v[4] + 0.16655 * v[7] + v[6];
            y[230] = -0.005 * v[17] + 0.081333 * v[12] + 0.16655 * v[9] + v[2];
            y[232] = -0.005 * v[19] + 0.057333 * v[8] + 0.16655 * v[1] + v[16];
            y[233] = -0.005 * v[5] + 0.057333 * v[4] + 0.16655 * v[7] + v[6];
            y[234] = -0.005 * v[17] + 0.057333 * v[12] + 0.16655 * v[9] + v[2];
            y[236] = 0.086583 * v[8] + 0.18855 * v[1] + v[16];
            y[237] = 0.086583 * v[4] + 0.18855 * v[7] + v[6];
            y[238] = 0.086583 * v[12] + 0.18855 * v[9] + v[2];
            y[240] = 0.086583 * v[8] + 0.20855 * v[1] + v[16];
            y[241] = 0.086583 * v[4] + 0.20855 * v[7] + v[6];
            y[242] = 0.086583 * v[12] + 0.20855 * v[9] + v[2];
            y[244] = 0.086583 * v[8] + 0.22855 * v[1] + v[16];
            y[245] = 0.086583 * v[4] + 0.22855 * v[7] + v[6];
            y[246] = 0.086583 * v[12] + 0.22855 * v[9] + v[2];
            y[248] = 0.01 * v[19] + 0.082083 * v[8] + 0.26625 * v[1] + v[16];
            y[249] = 0.01 * v[5] + 0.082083 * v[4] + 0.26625 * v[7] + v[6];
            y[250] = 0.01 * v[17] + 0.082083 * v[12] + 0.26625 * v[9] + v[2];
            y[252] = -0.01 * v[19] + 0.082083 * v[8] + 0.26625 * v[1] + v[16];
            y[253] = -0.01 * v[5] + 0.082083 * v[4] + 0.26625 * v[7] + v[6];
            y[254] = -0.01 * v[17] + 0.082083 * v[12] + 0.26625 * v[9] + v[2];
            y[256] = -0.01 * v[19] + 0.082083 * v[8] + 0.24625 * v[1] + v[16];
            y[257] = -0.01 * v[5] + 0.082083 * v[4] + 0.24625 * v[7] + v[6];
            y[258] = -0.01 * v[17] + 0.082083 * v[12] + 0.24625 * v[9] + v[2];
            y[260] = 0.01 * v[19] + 0.082083 * v[8] + 0.24625 * v[1] + v[16];
            y[261] = 0.01 * v[5] + 0.082083 * v[4] + 0.24625 * v[7] + v[6];
            y[262] = 0.01 * v[17] + 0.082083 * v[12] + 0.24625 * v[9] + v[2];
            y[264] = 0.005 * v[19] + -0.059333 * v[8] + 0.16655 * v[1] + v[16];
            y[265] = 0.005 * v[5] + -0.059333 * v[4] + 0.16655 * v[7] + v[6];
            y[266] = 0.005 * v[17] + -0.059333 * v[12] + 0.16655 * v[9] + v[2];
            y[268] = 0.005 * v[19] + -0.079333 * v[8] + 0.16655 * v[1] + v[16];
            y[269] = 0.005 * v[5] + -0.079333 * v[4] + 0.16655 * v[7] + v[6];
            y[270] = 0.005 * v[17] + -0.079333 * v[12] + 0.16655 * v[9] + v[2];
            y[272] = -0.086583 * v[8] + 0.18855 * v[1] + v[16];
            y[273] = -0.086583 * v[4] + 0.18855 * v[7] + v[6];
            y[274] = -0.086583 * v[12] + 0.18855 * v[9] + v[2];
            y[276] = -0.086583 * v[8] + 0.20855 * v[1] + v[16];
            y[277] = -0.086583 * v[4] + 0.20855 * v[7] + v[6];
            y[278] = -0.086583 * v[12] + 0.20855 * v[9] + v[2];
            y[280] = -0.086583 * v[8] + 0.22855 * v[1] + v[16];
            y[281] = -0.086583 * v[4] + 0.22855 * v[7] + v[6];
            y[282] = -0.086583 * v[12] + 0.22855 * v[9] + v[2];
            y[284] = 0.01 * v[19] + -0.082083 * v[8] + 0.26625 * v[1] + v[16];
            y[285] = 0.01 * v[5] + -0.082083 * v[4] + 0.26625 * v[7] + v[6];
            y[286] = 0.01 * v[17] + -0.082083 * v[12] + 0.26625 * v[9] + v[2];
            y[288] = -0.01 * v[19] + -0.082083 * v[8] + 0.26625 * v[1] + v[16];
            y[289] = -0.01 * v[5] + -0.082083 * v[4] + 0.26625 * v[7] + v[6];
            y[290] = -0.01 * v[17] + -0.082083 * v[12] + 0.26625 * v[9] + v[2];
            y[292] = -0.01 * v[19] + -0.082083 * v[8] + 0.24625 * v[1] + v[16];
            y[293] = -0.01 * v[5] + -0.082083 * v[4] + 0.24625 * v[7] + v[6];
            y[294] = -0.01 * v[17] + -0.082083 * v[12] + 0.24625 * v[9] + v[2];
            y[296] = 0.01 * v[19] + -0.082083 * v[8] + 0.24625 * v[1] + v[16];
            y[297] = 0.01 * v[5] + -0.082083 * v[4] + 0.24625 * v[7] + v[6];
            y[298] = 0.01 * v[17] + -0.082083 * v[12] + 0.24625 * v[9] + v[2];
            // dependent variables without operations
            y[0] = -0.025;
            y[1] = -0.1;
            y[2] = 0.1;
            y[3] = 0.25;
            y[4] = -0.025;
            y[5] = 0.1;
            y[6] = 0.1;
            y[7] = 0.25;
            y[8] = -0.065;
            y[9] = 0.;
            y[10] = 0.4;
            y[11] = 0.230000004172325;
            y[12] = 0.04;
            y[13] = 0.;
            y[14] = 0.686;
            y[15] = 0.200000002980232;
            y[16] = 0.0640272398484633;
            y[17] = 0.259027384507773;
            y[18] = 0.379626;
            y[19] = 0.100000001490116;
            y[20] = 0.0640272398484633;
            y[21] = 0.259027384507773;
            y[22] = 0.229626;
            y[23] = 0.100000001490116;
            y[26] = 0.399976;
            y[27] = 0.100000001490116;
            y[31] = 0.0799999982118607;
            y[35] = 0.0799999982118607;
            y[39] = 0.0799999982118607;
            y[43] = 0.100000001490116;
            y[47] = 0.0799999982118607;
            y[51] = 0.0799999982118607;
            y[55] = 0.0799999982118607;
            y[59] = 0.0700000002980232;
            y[63] = 0.0700000002980232;
            y[67] = 0.0700000002980232;
            y[71] = 0.0799999982118607;
            y[75] = 0.0500000007450581;
            y[79] = 0.0399999991059303;
            y[83] = 0.0399999991059303;
            y[87] = 0.0149999996647239;
            y[91] = 0.0149999996647239;
            y[95] = 0.0120000001043081;
            y[99] = 0.0120000001043081;
            y[103] = 0.0120000001043081;
            y[107] = 0.0140000004321337;
            y[111] = 0.0140000004321337;
            y[115] = 0.0140000004321337;
            y[119] = 0.0140000004321337;
            y[123] = 0.0149999996647239;
            y[127] = 0.0149999996647239;
            y[131] = 0.0120000001043081;
            y[135] = 0.0120000001043081;
            y[139] = 0.0120000001043081;
            y[143] = 0.0140000004321337;
            y[147] = 0.0140000004321337;
            y[151] = 0.0140000004321337;
            y[155] = 0.0140000004321337;
            y[156] = 0.;
            y[157] = 0.;
            y[158] = -0.6;
            y[159] = 0.5;
            y[160] = 0.0640272398484633;
            y[161] = -0.259027384507773;
            y[162] = 0.379626;
            y[163] = 0.100000001490116;
            y[164] = 0.0640272398484633;
            y[165] = -0.259027384507773;
            y[166] = 0.229626;
            y[167] = 0.100000001490116;
            y[170] = 0.399976;
            y[171] = 0.100000001490116;
            y[175] = 0.0799999982118607;
            y[179] = 0.0799999982118607;
            y[183] = 0.0799999982118607;
            y[187] = 0.100000001490116;
            y[191] = 0.0799999982118607;
            y[195] = 0.0799999982118607;
            y[199] = 0.0799999982118607;
            y[203] = 0.0700000002980232;
            y[207] = 0.0700000002980232;
            y[211] = 0.0700000002980232;
            y[215] = 0.0799999982118607;
            y[219] = 0.0500000007450581;
            y[223] = 0.0399999991059303;
            y[227] = 0.0399999991059303;
            y[231] = 0.0149999996647239;
            y[235] = 0.0149999996647239;
            y[239] = 0.0120000001043081;
            y[243] = 0.0120000001043081;
            y[247] = 0.0120000001043081;
            y[251] = 0.0140000004321337;
            y[255] = 0.0140000004321337;
            y[259] = 0.0140000004321337;
            y[263] = 0.0140000004321337;
            y[267] = 0.0149999996647239;
            y[271] = 0.0149999996647239;
            y[275] = 0.0120000001043081;
            y[279] = 0.0120000001043081;
            y[283] = 0.0120000001043081;
            y[287] = 0.0140000004321337;
            y[291] = 0.0140000004321337;
            y[295] = 0.0140000004321337;
            y[299] = 0.0140000004321337;

            for (auto i = 0U; i < 75; ++i)
            {
                out.x[i] = y[i * 4 + 0];
                out.y[i] = y[i * 4 + 1];
                out.z[i] = y[i * 4 + 2];
                out.r[i] = y[i * 4 + 3];
            }
        }

        using Debug = std::
            pair<std::vector<std::vector<std::string>>, std::vector<std::pair<std::size_t, std::size_t>>>;

        template <std::size_t rake>
        inline static auto fkcc_debug(
            const vamp::collision::Environment<FloatVector<rake>> &environment,
            const ConfigurationBlock<rake> &x) noexcept -> Debug
        {
            std::array<FloatVector<rake, 1>, 51> v;
            std::array<FloatVector<rake, 1>, 432> y;

            v[0] = cos(x[0]);
            v[1] = sin(x[0]);
            v[2] = 0.707105482511236 * v[0] + -0.707108079859474 * v[1];
            y[24] = 0.0640272398484633 + 0.069 * v[2];
            v[3] = 0.707108079859474 * v[0] + 0.707105482511236 * v[1];
            y[25] = 0.259027384507773 + 0.069 * v[3];
            v[4] = cos(x[1]);
            v[1] = -v[1];
            v[5] = 0.707105482511236 * v[1] + -0.707108079859474 * v[0];
            v[6] = sin(x[1]);
            v[7] = 4.89663865010925e-12 * v[6];
            v[8] = v[2] * v[4] + v[5] * v[7];
            v[9] = cos(x[2]);
            v[10] = sin(x[2]);
            v[11] = 4.89663865010925e-12 * v[9] + -4.89658313895802e-12 * v[10];
            v[12] = -v[6];
            v[13] = 4.89663865010925e-12 * v[4];
            v[2] = v[2] * v[12] + v[5] * v[13];
            v[14] = 5.55111512312578e-17 * v[9] + v[10];
            v[15] = v[8] * v[11] + v[2] * v[9] + v[5] * v[14];
            v[16] = v[8] + -4.89658313895802e-12 * v[2] + 4.89663865010925e-12 * v[5];
            y[36] = y[24] + 0.102 * v[8];
            y[28] = -0.02 * v[15] + 0.22 * v[16] + y[36];
            v[1] = 0.707108079859474 * v[1] + 0.707105482511236 * v[0];
            v[7] = v[3] * v[4] + v[1] * v[7];
            v[13] = v[3] * v[12] + v[1] * v[13];
            v[12] = v[7] * v[11] + v[13] * v[9] + v[1] * v[14];
            v[3] = v[7] + -4.89658313895802e-12 * v[13] + 4.89663865010925e-12 * v[1];
            y[37] = y[25] + 0.102 * v[7];
            y[29] = -0.02 * v[12] + 0.22 * v[3] + y[37];
            v[6] = -1. * v[6];
            v[4] = -1. * v[4];
            v[14] = v[6] * v[11] + v[4] * v[9] + 4.89663865010925e-12 * v[14];
            v[11] = v[6] + 2.39770700697438e-23 + -4.89658313895802e-12 * v[4];
            y[38] = 0.399976 + 0.102 * v[6];
            y[30] = -0.02 * v[14] + 0.22 * v[11] + y[38];
            y[32] = -0.01 * v[15] + 0.11 * v[16] + y[36];
            y[33] = -0.01 * v[12] + 0.11 * v[3] + y[37];
            y[34] = -0.01 * v[14] + 0.11 * v[11] + y[38];
            y[40] = y[36] + 0.069 * v[15] + 0.26242 * v[16];
            y[41] = y[37] + 0.069 * v[12] + 0.26242 * v[3];
            y[42] = y[38] + 0.069 * v[14] + 0.26242 * v[11];
            v[0] = cos(x[3]);
            v[17] = sin(x[3]);
            v[18] = 4.89663865010925e-12 * v[0] + v[17];
            v[10] = -v[10];
            v[19] = 4.89663865010925e-12 * v[10] + -4.89658313895802e-12 * v[9];
            v[9] = 5.55111512312578e-17 * v[10] + v[9];
            v[2] = v[8] * v[19] + v[2] * v[10] + v[5] * v[9];
            v[8] = 5.55111512312578e-17 * v[0] + 4.89663865010925e-12 * v[17];
            v[5] = v[0] + -4.89658313895802e-12 * v[17];
            v[20] = v[15] * v[18] + v[2] * v[8] + v[16] * v[5];
            y[44] = y[40] + 0.10359 * v[20];
            v[13] = v[7] * v[19] + v[13] * v[10] + v[1] * v[9];
            v[7] = v[12] * v[18] + v[13] * v[8] + v[3] * v[5];
            y[45] = y[41] + 0.10359 * v[7];
            v[9] = v[6] * v[19] + v[4] * v[10] + 4.89663865010925e-12 * v[9];
            v[5] = v[14] * v[18] + v[9] * v[8] + v[11] * v[5];
            y[46] = y[42] + 0.10359 * v[5];
            v[17] = -v[17];
            v[8] = 4.89663865010925e-12 * v[17] + v[0];
            v[18] = 5.55111512312578e-17 * v[17] + 4.89663865010925e-12 * v[0];
            v[17] = v[17] + -4.89658313895802e-12 * v[0];
            v[0] = v[15] * v[8] + v[2] * v[18] + v[16] * v[17];
            v[2] = -4.89658313895802e-12 * v[15] + v[2];
            v[19] = v[20] + -4.89658313895802e-12 * v[0] + 4.89663865010925e-12 * v[2];
            y[48] = 0.22 * v[19] + y[44];
            v[10] = v[12] * v[8] + v[13] * v[18] + v[3] * v[17];
            v[13] = -4.89658313895802e-12 * v[12] + v[13];
            v[4] = v[7] + -4.89658313895802e-12 * v[10] + 4.89663865010925e-12 * v[13];
            y[49] = 0.22 * v[4] + y[45];
            v[17] = v[14] * v[8] + v[9] * v[18] + v[11] * v[17];
            v[9] = -4.89658313895802e-12 * v[14] + v[9];
            v[18] = v[5] + -4.89658313895802e-12 * v[17] + 4.89663865010925e-12 * v[9];
            y[50] = 0.22 * v[18] + y[46];
            y[52] = 0.11 * v[19] + y[44];
            y[53] = 0.11 * v[4] + y[45];
            y[54] = 0.11 * v[18] + y[46];
            v[8] = cos(x[4]);
            v[6] = sin(x[4]);
            v[1] = 4.89663865010925e-12 * v[8] + -4.89658313895802e-12 * v[6];
            v[21] = 5.55111512312578e-17 * v[8] + v[6];
            v[22] = v[20] * v[1] + v[0] * v[8] + v[2] * v[21];
            v[6] = -v[6];
            v[23] = 4.89663865010925e-12 * v[6] + -4.89658313895802e-12 * v[8];
            v[24] = 5.55111512312578e-17 * v[6] + v[8];
            v[2] = v[20] * v[23] + v[0] * v[6] + v[2] * v[24];
            v[0] = -4.89658313895802e-12 * v[22] + v[2];
            y[328] = y[44] + 0.01 * v[22] + 0.2707 * v[19];
            y[56] = 0.03 * v[0] + y[328];
            v[20] = v[7] * v[1] + v[10] * v[8] + v[13] * v[21];
            v[13] = v[7] * v[23] + v[10] * v[6] + v[13] * v[24];
            v[10] = -4.89658313895802e-12 * v[20] + v[13];
            y[329] = y[45] + 0.01 * v[20] + 0.2707 * v[4];
            y[57] = 0.03 * v[10] + y[329];
            v[21] = v[5] * v[1] + v[17] * v[8] + v[9] * v[21];
            v[24] = v[5] * v[23] + v[17] * v[6] + v[9] * v[24];
            v[23] = -4.89658313895802e-12 * v[21] + v[24];
            y[330] = y[46] + 0.01 * v[21] + 0.2707 * v[18];
            y[58] = 0.03 * v[23] + y[330];
            y[60] = -0.03 * v[0] + y[328];
            y[61] = -0.03 * v[10] + y[329];
            y[62] = -0.03 * v[23] + y[330];
            v[6] = cos(x[5]);
            v[9] = sin(x[5]);
            v[17] = 4.89663865010925e-12 * v[6] + v[9];
            v[5] = 5.55111512312578e-17 * v[6] + 4.89663865010925e-12 * v[9];
            v[1] = v[6] + -4.89658313895802e-12 * v[9];
            v[8] = v[22] * v[17] + v[2] * v[5] + v[19] * v[1];
            v[9] = -v[9];
            v[7] = 4.89663865010925e-12 * v[9] + v[6];
            v[25] = 5.55111512312578e-17 * v[9] + 4.89663865010925e-12 * v[6];
            v[9] = v[9] + -4.89658313895802e-12 * v[6];
            v[2] = v[22] * v[7] + v[2] * v[25] + v[19] * v[9];
            v[22] = v[8] + -4.89658313895802e-12 * v[2] + 4.89663865010925e-12 * v[0];
            v[6] = y[328] + 0.115975 * v[8];
            y[64] = 0.02 * v[22] + v[6];
            v[26] = v[20] * v[17] + v[13] * v[5] + v[4] * v[1];
            v[13] = v[20] * v[7] + v[13] * v[25] + v[4] * v[9];
            v[20] = v[26] + -4.89658313895802e-12 * v[13] + 4.89663865010925e-12 * v[10];
            v[27] = y[329] + 0.115975 * v[26];
            y[65] = 0.02 * v[20] + v[27];
            v[1] = v[21] * v[17] + v[24] * v[5] + v[18] * v[1];
            v[9] = v[21] * v[7] + v[24] * v[25] + v[18] * v[9];
            v[25] = v[1] + -4.89658313895802e-12 * v[9] + 4.89663865010925e-12 * v[23];
            v[7] = y[330] + 0.115975 * v[1];
            y[66] = 0.02 * v[25] + v[7];
            y[68] = -0.04 * v[22] + v[6];
            y[69] = -0.04 * v[20] + v[27];
            y[70] = -0.04 * v[25] + v[7];
            v[24] = cos(x[6]);
            v[21] = sin(x[6]);
            v[5] = 4.89663865010925e-12 * v[24] + -4.89658313895802e-12 * v[21];
            v[17] = 5.55111512312578e-17 * v[24] + v[21];
            v[28] = v[8] * v[5] + v[2] * v[24] + v[0] * v[17];
            y[72] = 0.01 * v[28] + 0.09355 * v[22] + v[6];
            v[29] = v[26] * v[5] + v[13] * v[24] + v[10] * v[17];
            y[73] = 0.01 * v[29] + 0.09355 * v[20] + v[27];
            v[17] = v[1] * v[5] + v[9] * v[24] + v[23] * v[17];
            y[74] = 0.01 * v[17] + 0.09355 * v[25] + v[7];
            v[21] = -v[21];
            v[5] = 4.89663865010925e-12 * v[21] + -4.89658313895802e-12 * v[24];
            v[24] = 5.55111512312578e-17 * v[21] + v[24];
            v[2] = v[8] * v[5] + v[2] * v[21] + v[0] * v[24];
            y[76] = 0.02 * v[2] + 0.13855 * v[22] + v[6];
            v[13] = v[26] * v[5] + v[13] * v[21] + v[10] * v[24];
            y[77] = 0.02 * v[13] + 0.13855 * v[20] + v[27];
            v[24] = v[1] * v[5] + v[9] * v[21] + v[23] * v[24];
            y[78] = 0.02 * v[24] + 0.13855 * v[25] + v[7];
            y[80] = -0.02 * v[2] + 0.13855 * v[22] + v[6];
            y[81] = -0.02 * v[13] + 0.13855 * v[20] + v[27];
            y[82] = -0.02 * v[24] + 0.13855 * v[25] + v[7];
            y[84] = -0.005 * v[28] + 0.081333 * v[2] + 0.16655 * v[22] + v[6];
            y[85] = -0.005 * v[29] + 0.081333 * v[13] + 0.16655 * v[20] + v[27];
            y[86] = -0.005 * v[17] + 0.081333 * v[24] + 0.16655 * v[25] + v[7];
            y[88] = -0.005 * v[28] + 0.057333 * v[2] + 0.16655 * v[22] + v[6];
            y[89] = -0.005 * v[29] + 0.057333 * v[13] + 0.16655 * v[20] + v[27];
            y[90] = -0.005 * v[17] + 0.057333 * v[24] + 0.16655 * v[25] + v[7];
            y[92] = 0.086583 * v[2] + 0.18855 * v[22] + v[6];
            y[93] = 0.086583 * v[13] + 0.18855 * v[20] + v[27];
            y[94] = 0.086583 * v[24] + 0.18855 * v[25] + v[7];
            y[96] = 0.086583 * v[2] + 0.20855 * v[22] + v[6];
            y[97] = 0.086583 * v[13] + 0.20855 * v[20] + v[27];
            y[98] = 0.086583 * v[24] + 0.20855 * v[25] + v[7];
            y[100] = 0.086583 * v[2] + 0.22855 * v[22] + v[6];
            y[101] = 0.086583 * v[13] + 0.22855 * v[20] + v[27];
            y[102] = 0.086583 * v[24] + 0.22855 * v[25] + v[7];
            y[104] = 0.01 * v[28] + 0.082083 * v[2] + 0.26625 * v[22] + v[6];
            y[105] = 0.01 * v[29] + 0.082083 * v[13] + 0.26625 * v[20] + v[27];
            y[106] = 0.01 * v[17] + 0.082083 * v[24] + 0.26625 * v[25] + v[7];
            y[108] = -0.01 * v[28] + 0.082083 * v[2] + 0.26625 * v[22] + v[6];
            y[109] = -0.01 * v[29] + 0.082083 * v[13] + 0.26625 * v[20] + v[27];
            y[110] = -0.01 * v[17] + 0.082083 * v[24] + 0.26625 * v[25] + v[7];
            y[112] = -0.01 * v[28] + 0.082083 * v[2] + 0.24625 * v[22] + v[6];
            y[113] = -0.01 * v[29] + 0.082083 * v[13] + 0.24625 * v[20] + v[27];
            y[114] = -0.01 * v[17] + 0.082083 * v[24] + 0.24625 * v[25] + v[7];
            y[116] = 0.01 * v[28] + 0.082083 * v[2] + 0.24625 * v[22] + v[6];
            y[117] = 0.01 * v[29] + 0.082083 * v[13] + 0.24625 * v[20] + v[27];
            y[118] = 0.01 * v[17] + 0.082083 * v[24] + 0.24625 * v[25] + v[7];
            y[120] = 0.005 * v[28] + -0.059333 * v[2] + 0.16655 * v[22] + v[6];
            y[121] = 0.005 * v[29] + -0.059333 * v[13] + 0.16655 * v[20] + v[27];
            y[122] = 0.005 * v[17] + -0.059333 * v[24] + 0.16655 * v[25] + v[7];
            y[124] = 0.005 * v[28] + -0.079333 * v[2] + 0.16655 * v[22] + v[6];
            y[125] = 0.005 * v[29] + -0.079333 * v[13] + 0.16655 * v[20] + v[27];
            y[126] = 0.005 * v[17] + -0.079333 * v[24] + 0.16655 * v[25] + v[7];
            y[128] = -0.086583 * v[2] + 0.18855 * v[22] + v[6];
            y[129] = -0.086583 * v[13] + 0.18855 * v[20] + v[27];
            y[130] = -0.086583 * v[24] + 0.18855 * v[25] + v[7];
            y[132] = -0.086583 * v[2] + 0.20855 * v[22] + v[6];
            y[133] = -0.086583 * v[13] + 0.20855 * v[20] + v[27];
            y[134] = -0.086583 * v[24] + 0.20855 * v[25] + v[7];
            y[136] = -0.086583 * v[2] + 0.22855 * v[22] + v[6];
            y[137] = -0.086583 * v[13] + 0.22855 * v[20] + v[27];
            y[138] = -0.086583 * v[24] + 0.22855 * v[25] + v[7];
            y[140] = 0.01 * v[28] + -0.082083 * v[2] + 0.26625 * v[22] + v[6];
            y[141] = 0.01 * v[29] + -0.082083 * v[13] + 0.26625 * v[20] + v[27];
            y[142] = 0.01 * v[17] + -0.082083 * v[24] + 0.26625 * v[25] + v[7];
            y[144] = -0.01 * v[28] + -0.082083 * v[2] + 0.26625 * v[22] + v[6];
            y[145] = -0.01 * v[29] + -0.082083 * v[13] + 0.26625 * v[20] + v[27];
            y[146] = -0.01 * v[17] + -0.082083 * v[24] + 0.26625 * v[25] + v[7];
            y[148] = -0.01 * v[28] + -0.082083 * v[2] + 0.24625 * v[22] + v[6];
            y[149] = -0.01 * v[29] + -0.082083 * v[13] + 0.24625 * v[20] + v[27];
            y[150] = -0.01 * v[17] + -0.082083 * v[24] + 0.24625 * v[25] + v[7];
            y[152] = 0.01 * v[28] + -0.082083 * v[2] + 0.24625 * v[22] + v[6];
            y[153] = 0.01 * v[29] + -0.082083 * v[13] + 0.24625 * v[20] + v[27];
            y[154] = 0.01 * v[17] + -0.082083 * v[24] + 0.24625 * v[25] + v[7];
            v[5] = cos(x[7]);
            v[21] = sin(x[7]);
            v[9] = 0.707105482511236 * v[5] + 0.707108079859474 * v[21];
            y[168] = 0.0640272398484633 + 0.069 * v[9];
            v[1] = -0.707108079859474 * v[5] + 0.707105482511236 * v[21];
            y[169] = -0.259027384507773 + 0.069 * v[1];
            v[23] = cos(x[8]);
            v[21] = -v[21];
            v[26] = 0.707105482511236 * v[21] + 0.707108079859474 * v[5];
            v[10] = sin(x[8]);
            v[8] = 4.89663865010925e-12 * v[10];
            v[0] = v[9] * v[23] + v[26] * v[8];
            v[30] = cos(x[9]);
            v[31] = sin(x[9]);
            v[32] = 4.89663865010925e-12 * v[30] + -4.89658313895802e-12 * v[31];
            v[33] = -v[10];
            v[34] = 4.89663865010925e-12 * v[23];
            v[9] = v[9] * v[33] + v[26] * v[34];
            v[35] = 5.55111512312578e-17 * v[30] + v[31];
            v[36] = v[0] * v[32] + v[9] * v[30] + v[26] * v[35];
            v[37] = v[0] + -4.89658313895802e-12 * v[9] + 4.89663865010925e-12 * v[26];
            y[180] = y[168] + 0.102 * v[0];
            y[172] = -0.02 * v[36] + 0.22 * v[37] + y[180];
            v[21] = -0.707108079859474 * v[21] + 0.707105482511236 * v[5];
            v[8] = v[1] * v[23] + v[21] * v[8];
            v[34] = v[1] * v[33] + v[21] * v[34];
            v[33] = v[8] * v[32] + v[34] * v[30] + v[21] * v[35];
            v[1] = v[8] + -4.89658313895802e-12 * v[34] + 4.89663865010925e-12 * v[21];
            y[181] = y[169] + 0.102 * v[8];
            y[173] = -0.02 * v[33] + 0.22 * v[1] + y[181];
            v[10] = -1. * v[10];
            v[23] = -1. * v[23];
            v[35] = v[10] * v[32] + v[23] * v[30] + 4.89663865010925e-12 * v[35];
            v[32] = v[10] + 2.39770700697438e-23 + -4.89658313895802e-12 * v[23];
            y[182] = 0.399976 + 0.102 * v[10];
            y[174] = -0.02 * v[35] + 0.22 * v[32] + y[182];
            y[176] = -0.01 * v[36] + 0.11 * v[37] + y[180];
            y[177] = -0.01 * v[33] + 0.11 * v[1] + y[181];
            y[178] = -0.01 * v[35] + 0.11 * v[32] + y[182];
            y[184] = y[180] + 0.069 * v[36] + 0.26242 * v[37];
            y[185] = y[181] + 0.069 * v[33] + 0.26242 * v[1];
            y[186] = y[182] + 0.069 * v[35] + 0.26242 * v[32];
            v[5] = cos(x[10]);
            v[38] = sin(x[10]);
            v[39] = 4.89663865010925e-12 * v[5] + v[38];
            v[31] = -v[31];
            v[40] = 4.89663865010925e-12 * v[31] + -4.89658313895802e-12 * v[30];
            v[30] = 5.55111512312578e-17 * v[31] + v[30];
            v[9] = v[0] * v[40] + v[9] * v[31] + v[26] * v[30];
            v[0] = 5.55111512312578e-17 * v[5] + 4.89663865010925e-12 * v[38];
            v[26] = v[5] + -4.89658313895802e-12 * v[38];
            v[41] = v[36] * v[39] + v[9] * v[0] + v[37] * v[26];
            y[188] = y[184] + 0.10359 * v[41];
            v[34] = v[8] * v[40] + v[34] * v[31] + v[21] * v[30];
            v[8] = v[33] * v[39] + v[34] * v[0] + v[1] * v[26];
            y[189] = y[185] + 0.10359 * v[8];
            v[30] = v[10] * v[40] + v[23] * v[31] + 4.89663865010925e-12 * v[30];
            v[26] = v[35] * v[39] + v[30] * v[0] + v[32] * v[26];
            y[190] = y[186] + 0.10359 * v[26];
            v[38] = -v[38];
            v[0] = 4.89663865010925e-12 * v[38] + v[5];
            v[39] = 5.55111512312578e-17 * v[38] + 4.89663865010925e-12 * v[5];
            v[38] = v[38] + -4.89658313895802e-12 * v[5];
            v[5] = v[36] * v[0] + v[9] * v[39] + v[37] * v[38];
            v[9] = -4.89658313895802e-12 * v[36] + v[9];
            v[40] = v[41] + -4.89658313895802e-12 * v[5] + 4.89663865010925e-12 * v[9];
            y[192] = 0.22 * v[40] + y[188];
            v[31] = v[33] * v[0] + v[34] * v[39] + v[1] * v[38];
            v[34] = -4.89658313895802e-12 * v[33] + v[34];
            v[23] = v[8] + -4.89658313895802e-12 * v[31] + 4.89663865010925e-12 * v[34];
            y[193] = 0.22 * v[23] + y[189];
            v[38] = v[35] * v[0] + v[30] * v[39] + v[32] * v[38];
            v[30] = -4.89658313895802e-12 * v[35] + v[30];
            v[39] = v[26] + -4.89658313895802e-12 * v[38] + 4.89663865010925e-12 * v[30];
            y[194] = 0.22 * v[39] + y[190];
            y[196] = 0.11 * v[40] + y[188];
            y[197] = 0.11 * v[23] + y[189];
            y[198] = 0.11 * v[39] + y[190];
            v[0] = cos(x[11]);
            v[10] = sin(x[11]);
            v[21] = 4.89663865010925e-12 * v[0] + -4.89658313895802e-12 * v[10];
            v[42] = 5.55111512312578e-17 * v[0] + v[10];
            v[43] = v[41] * v[21] + v[5] * v[0] + v[9] * v[42];
            v[10] = -v[10];
            v[44] = 4.89663865010925e-12 * v[10] + -4.89658313895802e-12 * v[0];
            v[45] = 5.55111512312578e-17 * v[10] + v[0];
            v[9] = v[41] * v[44] + v[5] * v[10] + v[9] * v[45];
            v[5] = -4.89658313895802e-12 * v[43] + v[9];
            y[392] = y[188] + 0.01 * v[43] + 0.2707 * v[40];
            y[200] = 0.03 * v[5] + y[392];
            v[41] = v[8] * v[21] + v[31] * v[0] + v[34] * v[42];
            v[34] = v[8] * v[44] + v[31] * v[10] + v[34] * v[45];
            v[31] = -4.89658313895802e-12 * v[41] + v[34];
            y[393] = y[189] + 0.01 * v[41] + 0.2707 * v[23];
            y[201] = 0.03 * v[31] + y[393];
            v[42] = v[26] * v[21] + v[38] * v[0] + v[30] * v[42];
            v[45] = v[26] * v[44] + v[38] * v[10] + v[30] * v[45];
            v[44] = -4.89658313895802e-12 * v[42] + v[45];
            y[394] = y[190] + 0.01 * v[42] + 0.2707 * v[39];
            y[202] = 0.03 * v[44] + y[394];
            y[204] = -0.03 * v[5] + y[392];
            y[205] = -0.03 * v[31] + y[393];
            y[206] = -0.03 * v[44] + y[394];
            v[10] = cos(x[12]);
            v[30] = sin(x[12]);
            v[38] = 4.89663865010925e-12 * v[10] + v[30];
            v[26] = 5.55111512312578e-17 * v[10] + 4.89663865010925e-12 * v[30];
            v[21] = v[10] + -4.89658313895802e-12 * v[30];
            v[0] = v[43] * v[38] + v[9] * v[26] + v[40] * v[21];
            v[30] = -v[30];
            v[8] = 4.89663865010925e-12 * v[30] + v[10];
            v[46] = 5.55111512312578e-17 * v[30] + 4.89663865010925e-12 * v[10];
            v[30] = v[30] + -4.89658313895802e-12 * v[10];
            v[9] = v[43] * v[8] + v[9] * v[46] + v[40] * v[30];
            v[43] = v[0] + -4.89658313895802e-12 * v[9] + 4.89663865010925e-12 * v[5];
            v[10] = y[392] + 0.115975 * v[0];
            y[208] = 0.02 * v[43] + v[10];
            v[47] = v[41] * v[38] + v[34] * v[26] + v[23] * v[21];
            v[34] = v[41] * v[8] + v[34] * v[46] + v[23] * v[30];
            v[41] = v[47] + -4.89658313895802e-12 * v[34] + 4.89663865010925e-12 * v[31];
            v[48] = y[393] + 0.115975 * v[47];
            y[209] = 0.02 * v[41] + v[48];
            v[21] = v[42] * v[38] + v[45] * v[26] + v[39] * v[21];
            v[30] = v[42] * v[8] + v[45] * v[46] + v[39] * v[30];
            v[46] = v[21] + -4.89658313895802e-12 * v[30] + 4.89663865010925e-12 * v[44];
            v[8] = y[394] + 0.115975 * v[21];
            y[210] = 0.02 * v[46] + v[8];
            y[212] = -0.04 * v[43] + v[10];
            y[213] = -0.04 * v[41] + v[48];
            y[214] = -0.04 * v[46] + v[8];
            v[45] = cos(x[13]);
            v[42] = sin(x[13]);
            v[26] = 4.89663865010925e-12 * v[45] + -4.89658313895802e-12 * v[42];
            v[38] = 5.55111512312578e-17 * v[45] + v[42];
            v[49] = v[0] * v[26] + v[9] * v[45] + v[5] * v[38];
            y[216] = 0.01 * v[49] + 0.09355 * v[43] + v[10];
            v[50] = v[47] * v[26] + v[34] * v[45] + v[31] * v[38];
            y[217] = 0.01 * v[50] + 0.09355 * v[41] + v[48];
            v[38] = v[21] * v[26] + v[30] * v[45] + v[44] * v[38];
            y[218] = 0.01 * v[38] + 0.09355 * v[46] + v[8];
            v[42] = -v[42];
            v[26] = 4.89663865010925e-12 * v[42] + -4.89658313895802e-12 * v[45];
            v[45] = 5.55111512312578e-17 * v[42] + v[45];
            v[9] = v[0] * v[26] + v[9] * v[42] + v[5] * v[45];
            y[220] = 0.02 * v[9] + 0.13855 * v[43] + v[10];
            v[34] = v[47] * v[26] + v[34] * v[42] + v[31] * v[45];
            y[221] = 0.02 * v[34] + 0.13855 * v[41] + v[48];
            v[45] = v[21] * v[26] + v[30] * v[42] + v[44] * v[45];
            y[222] = 0.02 * v[45] + 0.13855 * v[46] + v[8];
            y[224] = -0.02 * v[9] + 0.13855 * v[43] + v[10];
            y[225] = -0.02 * v[34] + 0.13855 * v[41] + v[48];
            y[226] = -0.02 * v[45] + 0.13855 * v[46] + v[8];
            y[228] = -0.005 * v[49] + 0.081333 * v[9] + 0.16655 * v[43] + v[10];
            y[229] = -0.005 * v[50] + 0.081333 * v[34] + 0.16655 * v[41] + v[48];
            y[230] = -0.005 * v[38] + 0.081333 * v[45] + 0.16655 * v[46] + v[8];
            y[232] = -0.005 * v[49] + 0.057333 * v[9] + 0.16655 * v[43] + v[10];
            y[233] = -0.005 * v[50] + 0.057333 * v[34] + 0.16655 * v[41] + v[48];
            y[234] = -0.005 * v[38] + 0.057333 * v[45] + 0.16655 * v[46] + v[8];
            y[236] = 0.086583 * v[9] + 0.18855 * v[43] + v[10];
            y[237] = 0.086583 * v[34] + 0.18855 * v[41] + v[48];
            y[238] = 0.086583 * v[45] + 0.18855 * v[46] + v[8];
            y[240] = 0.086583 * v[9] + 0.20855 * v[43] + v[10];
            y[241] = 0.086583 * v[34] + 0.20855 * v[41] + v[48];
            y[242] = 0.086583 * v[45] + 0.20855 * v[46] + v[8];
            y[244] = 0.086583 * v[9] + 0.22855 * v[43] + v[10];
            y[245] = 0.086583 * v[34] + 0.22855 * v[41] + v[48];
            y[246] = 0.086583 * v[45] + 0.22855 * v[46] + v[8];
            y[248] = 0.01 * v[49] + 0.082083 * v[9] + 0.26625 * v[43] + v[10];
            y[249] = 0.01 * v[50] + 0.082083 * v[34] + 0.26625 * v[41] + v[48];
            y[250] = 0.01 * v[38] + 0.082083 * v[45] + 0.26625 * v[46] + v[8];
            y[252] = -0.01 * v[49] + 0.082083 * v[9] + 0.26625 * v[43] + v[10];
            y[253] = -0.01 * v[50] + 0.082083 * v[34] + 0.26625 * v[41] + v[48];
            y[254] = -0.01 * v[38] + 0.082083 * v[45] + 0.26625 * v[46] + v[8];
            y[256] = -0.01 * v[49] + 0.082083 * v[9] + 0.24625 * v[43] + v[10];
            y[257] = -0.01 * v[50] + 0.082083 * v[34] + 0.24625 * v[41] + v[48];
            y[258] = -0.01 * v[38] + 0.082083 * v[45] + 0.24625 * v[46] + v[8];
            y[260] = 0.01 * v[49] + 0.082083 * v[9] + 0.24625 * v[43] + v[10];
            y[261] = 0.01 * v[50] + 0.082083 * v[34] + 0.24625 * v[41] + v[48];
            y[262] = 0.01 * v[38] + 0.082083 * v[45] + 0.24625 * v[46] + v[8];
            y[264] = 0.005 * v[49] + -0.059333 * v[9] + 0.16655 * v[43] + v[10];
            y[265] = 0.005 * v[50] + -0.059333 * v[34] + 0.16655 * v[41] + v[48];
            y[266] = 0.005 * v[38] + -0.059333 * v[45] + 0.16655 * v[46] + v[8];
            y[268] = 0.005 * v[49] + -0.079333 * v[9] + 0.16655 * v[43] + v[10];
            y[269] = 0.005 * v[50] + -0.079333 * v[34] + 0.16655 * v[41] + v[48];
            y[270] = 0.005 * v[38] + -0.079333 * v[45] + 0.16655 * v[46] + v[8];
            y[272] = -0.086583 * v[9] + 0.18855 * v[43] + v[10];
            y[273] = -0.086583 * v[34] + 0.18855 * v[41] + v[48];
            y[274] = -0.086583 * v[45] + 0.18855 * v[46] + v[8];
            y[276] = -0.086583 * v[9] + 0.20855 * v[43] + v[10];
            y[277] = -0.086583 * v[34] + 0.20855 * v[41] + v[48];
            y[278] = -0.086583 * v[45] + 0.20855 * v[46] + v[8];
            y[280] = -0.086583 * v[9] + 0.22855 * v[43] + v[10];
            y[281] = -0.086583 * v[34] + 0.22855 * v[41] + v[48];
            y[282] = -0.086583 * v[45] + 0.22855 * v[46] + v[8];
            y[284] = 0.01 * v[49] + -0.082083 * v[9] + 0.26625 * v[43] + v[10];
            y[285] = 0.01 * v[50] + -0.082083 * v[34] + 0.26625 * v[41] + v[48];
            y[286] = 0.01 * v[38] + -0.082083 * v[45] + 0.26625 * v[46] + v[8];
            y[288] = -0.01 * v[49] + -0.082083 * v[9] + 0.26625 * v[43] + v[10];
            y[289] = -0.01 * v[50] + -0.082083 * v[34] + 0.26625 * v[41] + v[48];
            y[290] = -0.01 * v[38] + -0.082083 * v[45] + 0.26625 * v[46] + v[8];
            y[292] = -0.01 * v[49] + -0.082083 * v[9] + 0.24625 * v[43] + v[10];
            y[293] = -0.01 * v[50] + -0.082083 * v[34] + 0.24625 * v[41] + v[48];
            y[294] = -0.01 * v[38] + -0.082083 * v[45] + 0.24625 * v[46] + v[8];
            y[296] = 0.01 * v[49] + -0.082083 * v[9] + 0.24625 * v[43] + v[10];
            y[297] = 0.01 * v[50] + -0.082083 * v[34] + 0.24625 * v[41] + v[48];
            y[298] = 0.01 * v[38] + -0.082083 * v[45] + 0.24625 * v[46] + v[8];
            y[316] = -0.00999999977648258 * v[15] + 0.109999999403954 * v[16] + y[36];
            y[317] = -0.00999999977648258 * v[12] + 0.109999999403954 * v[3] + y[37];
            y[318] = -0.00999999977648258 * v[14] + 0.109999999403954 * v[11] + y[38];
            y[324] = 0.109999999403954 * v[19] + y[44];
            y[325] = 0.109999999403954 * v[4] + y[45];
            y[326] = 0.109999999403954 * v[18] + y[46];
            y[332] = -0.0149999987334013 * v[22] + v[6];
            y[333] = -0.0149999987334013 * v[20] + v[27];
            y[334] = -0.0149999987334013 * v[25] + v[7];
            y[336] = 0.00999999977648258 * v[28] + 0.0935499966144562 * v[22] + v[6];
            y[337] = 0.00999999977648258 * v[29] + 0.0935499966144562 * v[20] + v[27];
            y[338] = 0.00999999977648258 * v[17] + 0.0935499966144562 * v[25] + v[7];
            y[340] = 0.138549998402596 * v[22] + v[6];
            y[341] = 0.138549998402596 * v[20] + v[27];
            y[342] = 0.138549998402596 * v[25] + v[7];
            y[344] =
                -0.00499999988824129 * v[28] + 0.0693330019712448 * v[2] + 0.166549995541573 * v[22] + v[6];
            y[345] =
                -0.00499999988824129 * v[29] + 0.0693330019712448 * v[13] + 0.166549995541573 * v[20] + v[27];
            y[346] =
                -0.00499999988824129 * v[17] + 0.0693330019712448 * v[24] + 0.166549995541573 * v[25] + v[7];
            y[348] = 0.0865830034017563 * v[2] + 0.208550006151199 * v[22] + v[6];
            y[349] = 0.0865830034017563 * v[13] + 0.208550006151199 * v[20] + v[27];
            y[350] = 0.0865830034017563 * v[24] + 0.208550006151199 * v[25] + v[7];
            y[352] = 0.0820830017328262 * v[2] + 0.256249994039536 * v[22] + v[6];
            y[353] = 0.0820830017328262 * v[13] + 0.256249994039536 * v[20] + v[27];
            y[354] = 0.0820830017328262 * v[24] + 0.256249994039536 * v[25] + v[7];
            y[356] =
                0.00499999988824129 * v[28] + -0.0693330019712448 * v[2] + 0.166549995541573 * v[22] + v[6];
            y[357] =
                0.00499999988824129 * v[29] + -0.0693330019712448 * v[13] + 0.166549995541573 * v[20] + v[27];
            y[358] =
                0.00499999988824129 * v[17] + -0.0693330019712448 * v[24] + 0.166549995541573 * v[25] + v[7];
            y[360] = -0.0865830034017563 * v[2] + 0.208550006151199 * v[22] + v[6];
            y[361] = -0.0865830034017563 * v[13] + 0.208550006151199 * v[20] + v[27];
            y[362] = -0.0865830034017563 * v[24] + 0.208550006151199 * v[25] + v[7];
            y[364] = -0.0820830017328262 * v[2] + 0.256249994039536 * v[22] + v[6];
            y[365] = -0.0820830017328262 * v[13] + 0.256249994039536 * v[20] + v[27];
            y[366] = -0.0820830017328262 * v[24] + 0.256249994039536 * v[25] + v[7];
            y[380] = -0.00999999977648258 * v[36] + 0.109999999403954 * v[37] + y[180];
            y[381] = -0.00999999977648258 * v[33] + 0.109999999403954 * v[1] + y[181];
            y[382] = -0.00999999977648258 * v[35] + 0.109999999403954 * v[32] + y[182];
            y[388] = 0.109999999403954 * v[40] + y[188];
            y[389] = 0.109999999403954 * v[23] + y[189];
            y[390] = 0.109999999403954 * v[39] + y[190];
            y[396] = -0.0149999987334013 * v[43] + v[10];
            y[397] = -0.0149999987334013 * v[41] + v[48];
            y[398] = -0.0149999987334013 * v[46] + v[8];
            y[400] = 0.00999999977648258 * v[49] + 0.0935499966144562 * v[43] + v[10];
            y[401] = 0.00999999977648258 * v[50] + 0.0935499966144562 * v[41] + v[48];
            y[402] = 0.00999999977648258 * v[38] + 0.0935499966144562 * v[46] + v[8];
            y[404] = 0.138549998402596 * v[43] + v[10];
            y[405] = 0.138549998402596 * v[41] + v[48];
            y[406] = 0.138549998402596 * v[46] + v[8];
            y[408] =
                -0.00499999988824129 * v[49] + 0.0693330019712448 * v[9] + 0.166549995541573 * v[43] + v[10];
            y[409] =
                -0.00499999988824129 * v[50] + 0.0693330019712448 * v[34] + 0.166549995541573 * v[41] + v[48];
            y[410] =
                -0.00499999988824129 * v[38] + 0.0693330019712448 * v[45] + 0.166549995541573 * v[46] + v[8];
            y[412] = 0.0865830034017563 * v[9] + 0.208550006151199 * v[43] + v[10];
            y[413] = 0.0865830034017563 * v[34] + 0.208550006151199 * v[41] + v[48];
            y[414] = 0.0865830034017563 * v[45] + 0.208550006151199 * v[46] + v[8];
            y[416] = 0.0820830017328262 * v[9] + 0.256249994039536 * v[43] + v[10];
            y[417] = 0.0820830017328262 * v[34] + 0.256249994039536 * v[41] + v[48];
            y[418] = 0.0820830017328262 * v[45] + 0.256249994039536 * v[46] + v[8];
            y[420] =
                0.00499999988824129 * v[49] + -0.0693330019712448 * v[9] + 0.166549995541573 * v[43] + v[10];
            y[421] =
                0.00499999988824129 * v[50] + -0.0693330019712448 * v[34] + 0.166549995541573 * v[41] + v[48];
            y[422] =
                0.00499999988824129 * v[38] + -0.0693330019712448 * v[45] + 0.166549995541573 * v[46] + v[8];
            y[424] = -0.0865830034017563 * v[9] + 0.208550006151199 * v[43] + v[10];
            y[425] = -0.0865830034017563 * v[34] + 0.208550006151199 * v[41] + v[48];
            y[426] = -0.0865830034017563 * v[45] + 0.208550006151199 * v[46] + v[8];
            y[428] = -0.0820830017328262 * v[9] + 0.256249994039536 * v[43] + v[10];
            y[429] = -0.0820830017328262 * v[34] + 0.256249994039536 * v[41] + v[48];
            y[430] = -0.0820830017328262 * v[45] + 0.256249994039536 * v[46] + v[8];
            // variable duplicates: 10
            y[312] = y[24];
            y[313] = y[25];
            y[320] = y[40];
            y[321] = y[41];
            y[322] = y[42];
            y[376] = y[168];
            y[377] = y[169];
            y[384] = y[184];
            y[385] = y[185];
            y[386] = y[186];
            // dependent variables without operations
            y[0] = -0.025;
            y[1] = -0.1;
            y[2] = 0.1;
            y[3] = 0.25;
            y[4] = -0.025;
            y[5] = 0.1;
            y[6] = 0.1;
            y[7] = 0.25;
            y[8] = -0.065;
            y[9] = 0.;
            y[10] = 0.4;
            y[11] = 0.230000004172325;
            y[12] = 0.04;
            y[13] = 0.;
            y[14] = 0.686;
            y[15] = 0.200000002980232;
            y[16] = 0.0640272398484633;
            y[17] = 0.259027384507773;
            y[18] = 0.379626;
            y[19] = 0.100000001490116;
            y[20] = 0.0640272398484633;
            y[21] = 0.259027384507773;
            y[22] = 0.229626;
            y[23] = 0.100000001490116;
            y[26] = 0.399976;
            y[27] = 0.100000001490116;
            y[31] = 0.0799999982118607;
            y[35] = 0.0799999982118607;
            y[39] = 0.0799999982118607;
            y[43] = 0.100000001490116;
            y[47] = 0.0799999982118607;
            y[51] = 0.0799999982118607;
            y[55] = 0.0799999982118607;
            y[59] = 0.0700000002980232;
            y[63] = 0.0700000002980232;
            y[67] = 0.0700000002980232;
            y[71] = 0.0799999982118607;
            y[75] = 0.0500000007450581;
            y[79] = 0.0399999991059303;
            y[83] = 0.0399999991059303;
            y[87] = 0.0149999996647239;
            y[91] = 0.0149999996647239;
            y[95] = 0.0120000001043081;
            y[99] = 0.0120000001043081;
            y[103] = 0.0120000001043081;
            y[107] = 0.0140000004321337;
            y[111] = 0.0140000004321337;
            y[115] = 0.0140000004321337;
            y[119] = 0.0140000004321337;
            y[123] = 0.0149999996647239;
            y[127] = 0.0149999996647239;
            y[131] = 0.0120000001043081;
            y[135] = 0.0120000001043081;
            y[139] = 0.0120000001043081;
            y[143] = 0.0140000004321337;
            y[147] = 0.0140000004321337;
            y[151] = 0.0140000004321337;
            y[155] = 0.0140000004321337;
            y[156] = 0.;
            y[157] = 0.;
            y[158] = -0.6;
            y[159] = 0.5;
            y[160] = 0.0640272398484633;
            y[161] = -0.259027384507773;
            y[162] = 0.379626;
            y[163] = 0.100000001490116;
            y[164] = 0.0640272398484633;
            y[165] = -0.259027384507773;
            y[166] = 0.229626;
            y[167] = 0.100000001490116;
            y[170] = 0.399976;
            y[171] = 0.100000001490116;
            y[175] = 0.0799999982118607;
            y[179] = 0.0799999982118607;
            y[183] = 0.0799999982118607;
            y[187] = 0.100000001490116;
            y[191] = 0.0799999982118607;
            y[195] = 0.0799999982118607;
            y[199] = 0.0799999982118607;
            y[203] = 0.0700000002980232;
            y[207] = 0.0700000002980232;
            y[211] = 0.0700000002980232;
            y[215] = 0.0799999982118607;
            y[219] = 0.0500000007450581;
            y[223] = 0.0399999991059303;
            y[227] = 0.0399999991059303;
            y[231] = 0.0149999996647239;
            y[235] = 0.0149999996647239;
            y[239] = 0.0120000001043081;
            y[243] = 0.0120000001043081;
            y[247] = 0.0120000001043081;
            y[251] = 0.0140000004321337;
            y[255] = 0.0140000004321337;
            y[259] = 0.0140000004321337;
            y[263] = 0.0140000004321337;
            y[267] = 0.0149999996647239;
            y[271] = 0.0149999996647239;
            y[275] = 0.0120000001043081;
            y[279] = 0.0120000001043081;
            y[283] = 0.0120000001043081;
            y[287] = 0.0140000004321337;
            y[291] = 0.0140000004321337;
            y[295] = 0.0140000004321337;
            y[299] = 0.0140000004321337;
            y[300] = -0.0413404628634453;
            y[301] = 0.;
            y[302] = 0.222553476691246;
            y[303] = 0.409016877412796;
            y[304] = 0.0399999991059303;
            y[305] = 0.;
            y[306] = 0.685999989509583;
            y[307] = 0.200000002980232;
            y[308] = 0.0640272398484633;
            y[309] = 0.259027384507773;
            y[310] = 0.304625997019768;
            y[311] = 0.174999997019768;
            y[314] = 0.399976;
            y[315] = 0.100000001490116;
            y[319] = 0.190453603863716;
            y[323] = 0.100000001490116;
            y[327] = 0.189999997615814;
            y[331] = 0.100000001490116;
            y[335] = 0.104999996721745;
            y[339] = 0.0500000007450581;
            y[343] = 0.0599999986588955;
            y[347] = 0.0269999988377094;
            y[351] = 0.0320000015199184;
            y[355] = 0.0281421355903149;
            y[359] = 0.025000000372529;
            y[363] = 0.0320000015199184;
            y[367] = 0.0281421355903149;
            y[368] = 0.;
            y[369] = 0.;
            y[370] = -0.600000023841858;
            y[371] = 0.5;
            y[372] = 0.0640272398484633;
            y[373] = -0.259027384507773;
            y[374] = 0.304625997019768;
            y[375] = 0.174999997019768;
            y[378] = 0.399976;
            y[379] = 0.100000001490116;
            y[383] = 0.190453603863716;
            y[387] = 0.100000001490116;
            y[391] = 0.189999997615814;
            y[395] = 0.100000001490116;
            y[399] = 0.104999996721745;
            y[403] = 0.0500000007450581;
            y[407] = 0.0599999986588955;
            y[411] = 0.0269999988377094;
            y[415] = 0.0320000015199184;
            y[419] = 0.0281421355903149;
            y[423] = 0.025000000372529;
            y[427] = 0.0320000015199184;
            y[431] = 0.0281421355903149;

            Debug output;

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[0], y[1], y[2], y[3]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[4], y[5], y[6], y[7]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[8], y[9], y[10], y[11]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[12], y[13], y[14], y[15]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[16], y[17], y[18], y[19]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[20], y[21], y[22], y[23]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[24], y[25], y[26], y[27]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[28], y[29], y[30], y[31]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[32], y[33], y[34], y[35]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[36], y[37], y[38], y[39]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[40], y[41], y[42], y[43]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[44], y[45], y[46], y[47]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[48], y[49], y[50], y[51]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[52], y[53], y[54], y[55]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[56], y[57], y[58], y[59]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[60], y[61], y[62], y[63]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[64], y[65], y[66], y[67]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[68], y[69], y[70], y[71]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[72], y[73], y[74], y[75]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[76], y[77], y[78], y[79]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[80], y[81], y[82], y[83]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[84], y[85], y[86], y[87]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[88], y[89], y[90], y[91]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[92], y[93], y[94], y[95]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[96], y[97], y[98], y[99]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[100], y[101], y[102], y[103]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[104], y[105], y[106], y[107]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[108], y[109], y[110], y[111]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[112], y[113], y[114], y[115]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[116], y[117], y[118], y[119]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[120], y[121], y[122], y[123]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[124], y[125], y[126], y[127]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[128], y[129], y[130], y[131]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[132], y[133], y[134], y[135]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[136], y[137], y[138], y[139]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[140], y[141], y[142], y[143]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[144], y[145], y[146], y[147]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[148], y[149], y[150], y[151]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[152], y[153], y[154], y[155]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[156], y[157], y[158], y[159]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[160], y[161], y[162], y[163]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[164], y[165], y[166], y[167]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[168], y[169], y[170], y[171]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[172], y[173], y[174], y[175]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[176], y[177], y[178], y[179]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[180], y[181], y[182], y[183]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[184], y[185], y[186], y[187]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[188], y[189], y[190], y[191]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[192], y[193], y[194], y[195]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[196], y[197], y[198], y[199]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[200], y[201], y[202], y[203]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[204], y[205], y[206], y[207]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[208], y[209], y[210], y[211]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[212], y[213], y[214], y[215]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[216], y[217], y[218], y[219]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[220], y[221], y[222], y[223]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[224], y[225], y[226], y[227]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[228], y[229], y[230], y[231]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[232], y[233], y[234], y[235]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[236], y[237], y[238], y[239]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[240], y[241], y[242], y[243]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[244], y[245], y[246], y[247]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[248], y[249], y[250], y[251]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[252], y[253], y[254], y[255]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[256], y[257], y[258], y[259]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[260], y[261], y[262], y[263]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[264], y[265], y[266], y[267]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[268], y[269], y[270], y[271]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[272], y[273], y[274], y[275]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[276], y[277], y[278], y[279]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[280], y[281], y[282], y[283]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[284], y[285], y[286], y[287]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[288], y[289], y[290], y[291]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[292], y[293], y[294], y[295]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[296], y[297], y[298], y[299]));

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[40], y[41], y[42], y[43]))
            {
                output.second.emplace_back(0, 10);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[40], y[41], y[42], y[43]))
            {
                output.second.emplace_back(1, 10);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[40], y[41], y[42], y[43]))
            {
                output.second.emplace_back(2, 10);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[44], y[45], y[46], y[47]))
            {
                output.second.emplace_back(0, 11);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[48], y[49], y[50], y[51]))
            {
                output.second.emplace_back(0, 12);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[52], y[53], y[54], y[55]))
            {
                output.second.emplace_back(0, 13);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[44], y[45], y[46], y[47]))
            {
                output.second.emplace_back(1, 11);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[48], y[49], y[50], y[51]))
            {
                output.second.emplace_back(1, 12);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[52], y[53], y[54], y[55]))
            {
                output.second.emplace_back(1, 13);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[44], y[45], y[46], y[47]))
            {
                output.second.emplace_back(2, 11);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[48], y[49], y[50], y[51]))
            {
                output.second.emplace_back(2, 12);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[52], y[53], y[54], y[55]))
            {
                output.second.emplace_back(2, 13);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[56], y[57], y[58], y[59]))
            {
                output.second.emplace_back(0, 14);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[60], y[61], y[62], y[63]))
            {
                output.second.emplace_back(0, 15);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[56], y[57], y[58], y[59]))
            {
                output.second.emplace_back(1, 14);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[60], y[61], y[62], y[63]))
            {
                output.second.emplace_back(1, 15);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[56], y[57], y[58], y[59]))
            {
                output.second.emplace_back(2, 14);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[60], y[61], y[62], y[63]))
            {
                output.second.emplace_back(2, 15);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(0, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(0, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(1, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(1, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(2, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(2, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(0, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(1, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(2, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(0, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(0, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(1, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(1, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(2, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(2, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(0, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(0, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(1, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(1, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(2, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(2, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(0, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(0, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(0, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(1, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(1, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(1, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(2, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(2, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(2, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(0, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(0, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(0, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(0, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(1, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(1, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(1, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(1, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(2, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(2, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(2, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(2, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(0, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(0, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(1, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(1, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(2, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(2, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(0, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(0, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(0, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(1, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(1, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(1, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(2, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(2, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(2, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(0, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(0, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(0, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(0, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(1, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(1, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(1, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(1, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(2, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(2, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(2, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(2, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(0, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(1, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(2, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(0, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(0, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(0, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(1, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(1, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(1, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(2, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(2, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(2, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(0, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(0, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(1, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(1, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(2, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(2, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(0, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(0, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(1, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(1, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(2, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(2, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(0, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(1, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(2, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(0, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(0, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(1, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(1, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(2, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(2, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(0, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(0, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(1, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(1, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(2, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(2, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(0, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(0, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(0, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(1, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(1, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(1, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(2, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(2, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(2, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(0, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(0, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(0, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(0, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(1, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(1, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(1, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(1, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(2, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(2, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(2, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(2, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(0, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(0, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(1, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(1, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(2, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(2, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(0, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(0, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(0, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(1, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(1, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(1, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(2, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(2, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(2, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(0, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(0, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(0, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(0, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(1, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(1, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(1, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(1, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(2, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(2, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(2, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(2, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[40], y[41], y[42], y[43]))
            {
                output.second.emplace_back(3, 10);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[44], y[45], y[46], y[47]))
            {
                output.second.emplace_back(3, 11);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[48], y[49], y[50], y[51]))
            {
                output.second.emplace_back(3, 12);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[52], y[53], y[54], y[55]))
            {
                output.second.emplace_back(3, 13);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[56], y[57], y[58], y[59]))
            {
                output.second.emplace_back(3, 14);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[60], y[61], y[62], y[63]))
            {
                output.second.emplace_back(3, 15);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(3, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(3, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(3, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(3, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(3, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(3, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(3, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(3, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(3, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(3, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(3, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(3, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(3, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(3, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(3, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(3, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(3, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(3, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(3, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(3, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(3, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(3, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(3, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(3, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(3, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(3, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(3, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(3, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(3, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(3, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(3, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(3, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(3, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(3, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(3, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(3, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(3, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(3, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(3, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(3, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(3, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(3, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[44], y[45], y[46], y[47]))
            {
                output.second.emplace_back(4, 11);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[48], y[49], y[50], y[51]))
            {
                output.second.emplace_back(4, 12);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[52], y[53], y[54], y[55]))
            {
                output.second.emplace_back(4, 13);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[44], y[45], y[46], y[47]))
            {
                output.second.emplace_back(5, 11);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[48], y[49], y[50], y[51]))
            {
                output.second.emplace_back(5, 12);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[52], y[53], y[54], y[55]))
            {
                output.second.emplace_back(5, 13);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[56], y[57], y[58], y[59]))
            {
                output.second.emplace_back(4, 14);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[60], y[61], y[62], y[63]))
            {
                output.second.emplace_back(4, 15);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[56], y[57], y[58], y[59]))
            {
                output.second.emplace_back(5, 14);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[60], y[61], y[62], y[63]))
            {
                output.second.emplace_back(5, 15);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(4, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(4, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(5, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(5, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(4, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(5, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(4, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(4, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(5, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(5, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(4, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(4, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(5, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(5, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(4, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(4, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(4, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(5, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(5, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(5, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(4, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(4, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(4, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(4, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(5, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(5, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(5, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(5, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(4, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(4, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(5, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(5, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(4, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(4, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(4, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(5, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(5, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(5, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(4, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(4, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(4, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(4, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(5, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(5, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(5, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(5, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(4, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(5, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(4, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(4, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(5, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(5, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(4, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(4, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(5, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(5, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(4, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(5, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(4, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(4, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(5, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(5, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(4, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(4, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(5, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(5, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(4, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(4, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(4, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(5, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(5, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(5, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(4, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(4, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(4, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(4, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(5, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(5, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(5, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(5, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(4, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(4, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(5, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(5, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(4, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(4, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(4, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(5, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(5, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(5, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(4, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(4, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(4, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(4, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(5, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(5, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(5, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(5, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(6, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(6, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(6, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(6, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(6, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(6, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(6, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(6, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(6, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(6, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(6, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(6, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(6, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(6, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(6, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(6, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(6, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(6, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(6, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(6, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(6, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(6, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(6, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(6, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(6, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(6, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(6, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(6, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(6, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(6, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(6, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(6, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(6, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(6, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(6, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(6, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(6, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(6, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(6, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(6, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(6, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(6, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(6, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(6, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(6, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(6, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(6, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(6, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(6, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(6, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(6, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(6, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(7, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(7, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(8, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(8, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[64], y[65], y[66], y[67]))
            {
                output.second.emplace_back(9, 16);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(9, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(7, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(8, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(9, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(7, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(7, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(8, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(8, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(9, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(9, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(7, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(7, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(8, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(8, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(9, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(9, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(7, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(7, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(7, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(8, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(8, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(8, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(9, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(9, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(9, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(7, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(7, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(7, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(7, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(8, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(8, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(8, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(8, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(9, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(9, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(9, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(9, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(7, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(7, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(8, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(8, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(9, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(9, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(7, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(7, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(7, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(8, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(8, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(8, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(9, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(9, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(9, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(7, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(7, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(7, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(7, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(8, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(8, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(8, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(8, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(9, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(9, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(9, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(9, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(7, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(8, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(9, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(7, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(7, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(7, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(8, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(8, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(8, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(9, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(9, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(9, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(7, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(7, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(8, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(8, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(9, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(9, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(7, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(7, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(8, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(8, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(9, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(9, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(7, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(8, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(9, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(7, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(7, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(8, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(8, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(9, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(9, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(7, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(7, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(8, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(8, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(9, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(9, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(7, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(7, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(7, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(8, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(8, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(8, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(9, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(9, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(9, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(7, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(7, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(7, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(7, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(8, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(8, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(8, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(8, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(9, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(9, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(9, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(9, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(7, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(7, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(8, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(8, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(9, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(9, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(7, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(7, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(7, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(8, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(8, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(8, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(9, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(9, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(9, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(7, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(7, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(7, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(7, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(8, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(8, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(8, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(8, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(9, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(9, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(9, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(9, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(10, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(10, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(10, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(10, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(10, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(10, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(10, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(10, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(10, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(10, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(10, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(10, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(10, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(10, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(10, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(10, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(10, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(10, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(10, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(10, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(10, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(10, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(10, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(10, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(10, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(10, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(10, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(10, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(10, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(10, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(10, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(10, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(10, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(10, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(10, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(10, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(10, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(10, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(10, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(10, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(10, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(11, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(11, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(11, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(11, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(12, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(12, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(12, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(12, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(13, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(13, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(13, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(13, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(11, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(11, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(11, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(11, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(12, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(12, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(12, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(12, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(13, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(13, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(13, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(13, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(11, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(12, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(13, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(11, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(12, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(13, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(11, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(11, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(11, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(12, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(12, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(12, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(13, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(13, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(13, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(11, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(12, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(13, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(11, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(11, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(11, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(12, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(12, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(12, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(13, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(13, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(13, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(11, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(11, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(12, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(12, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(13, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(13, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(11, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(11, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(12, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(12, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(13, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(13, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(11, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(12, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(13, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(11, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(11, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(12, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(12, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(13, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(13, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(11, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(11, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(12, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(12, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(13, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(13, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(11, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(11, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(11, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(12, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(12, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(12, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(13, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(13, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(13, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(11, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(11, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(11, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(11, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(12, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(12, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(12, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(12, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(13, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(13, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(13, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(13, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(11, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(11, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(12, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(12, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(13, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(13, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(11, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(11, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(11, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(12, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(12, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(12, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(13, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(13, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(13, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(11, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(11, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(11, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(11, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(12, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(12, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(12, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(12, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(13, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(13, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(13, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(13, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(14, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(14, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(14, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(14, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(15, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(15, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(15, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(15, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(14, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(14, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(14, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(14, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(15, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(15, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(15, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(15, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(14, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(15, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(14, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(14, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(15, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(15, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(14, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(15, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(14, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(14, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(14, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(15, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(15, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(15, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(14, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(15, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(14, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(14, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(14, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(15, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(15, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(15, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(14, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(14, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(15, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(15, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(14, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(14, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(15, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(15, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(14, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(15, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(14, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(14, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(15, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(15, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(14, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(14, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(15, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(15, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(14, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(14, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(14, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(15, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(15, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(15, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(14, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(14, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(14, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(14, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(15, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(15, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(15, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(15, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(14, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(14, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(15, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(15, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(14, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(14, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(14, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(15, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(15, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(15, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(14, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(14, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(14, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(14, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(15, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(15, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(15, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(15, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(16, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(17, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(16, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(16, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(17, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(17, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(16, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(17, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(16, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(16, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(16, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(17, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(17, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(17, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(16, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(17, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(16, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(16, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(16, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(17, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(17, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(17, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(16, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(16, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(17, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(17, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(16, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(16, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(17, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(17, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(16, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(17, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(16, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(16, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(17, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(17, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(16, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(16, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(17, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(17, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(16, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(16, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(16, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(17, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(17, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(17, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(16, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(16, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(16, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(16, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(17, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(17, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(17, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(17, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(16, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(16, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(17, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(17, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(16, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(16, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(16, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(17, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(17, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(17, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(16, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(16, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(16, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(16, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(17, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(17, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(17, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(17, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(18, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(18, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(18, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(18, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(18, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(18, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(18, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(18, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(18, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(18, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(18, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(18, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(18, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(18, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(18, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(18, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(18, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(18, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(18, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(18, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(18, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(18, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(18, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(18, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(18, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(18, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(18, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(18, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(18, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(18, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(18, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(18, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(18, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(18, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(18, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(18, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(19, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(20, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(19, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(19, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(20, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(20, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(19, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(20, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(19, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(19, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(19, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(20, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(20, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(20, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(19, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(20, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(19, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(19, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(19, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(20, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(20, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(20, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(19, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(19, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(20, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(20, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(19, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(19, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(20, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(20, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(19, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(20, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(19, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(19, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(20, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(20, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(19, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(19, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(20, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(20, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(19, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(19, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(19, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(20, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(20, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(20, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(19, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(19, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(19, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(19, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(20, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(20, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(20, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(20, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(19, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(19, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(20, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(20, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(19, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(19, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(19, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(20, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(20, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(20, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(19, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(19, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(19, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(19, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(20, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(20, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(20, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(20, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(21, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(22, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(21, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(21, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(22, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(22, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(21, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(22, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(21, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(21, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(21, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(22, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(22, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(22, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(21, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(22, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(21, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(21, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(21, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(22, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(22, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(22, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(21, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(21, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(22, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(22, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(21, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(21, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(22, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(22, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(21, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(22, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(21, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(21, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(22, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(22, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(21, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(21, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(22, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(22, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(21, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(21, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(21, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(22, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(22, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(22, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(21, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(21, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(21, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(21, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(22, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(22, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(22, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(22, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(21, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(21, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(22, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(22, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(21, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(21, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(21, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(22, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(22, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(22, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(21, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(21, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(21, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(21, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(22, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(22, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(22, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(22, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(23, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(24, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(25, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(23, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(23, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(24, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(24, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(25, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(25, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(23, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(24, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(25, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(23, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(23, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(23, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(24, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(24, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(24, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(25, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(25, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(25, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(23, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(24, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(25, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(23, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(23, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(23, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(24, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(24, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(24, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(25, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(25, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(25, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(23, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(23, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(24, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(24, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(25, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(25, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(23, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(23, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(24, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(24, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(25, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(25, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(23, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(24, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(25, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(23, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(23, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(24, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(24, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(25, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(25, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(23, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(23, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(24, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(24, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(25, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(25, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(23, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(23, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(23, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(24, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(24, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(24, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(25, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(25, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(25, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(23, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(23, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(23, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(23, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(24, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(24, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(24, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(24, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(25, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(25, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(25, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(25, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(23, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(23, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(24, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(24, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(25, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(25, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(23, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(23, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(23, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(24, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(24, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(24, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(25, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(25, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(25, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(23, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(23, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(23, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(23, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(24, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(24, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(24, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(24, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(25, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(25, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(25, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(25, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(26, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(27, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(28, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(29, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(26, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(26, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(27, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(27, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(28, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(28, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(29, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(29, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(26, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(27, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(28, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(29, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(26, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(26, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(26, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(27, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(27, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(27, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(28, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(28, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(28, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(29, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(29, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(29, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(26, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(27, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(28, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(29, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(26, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(26, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(26, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(27, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(27, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(27, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(28, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(28, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(28, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(29, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(29, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(29, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(26, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(26, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(27, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(27, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(28, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(28, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(29, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(29, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(26, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(26, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(27, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(27, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(28, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(28, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(29, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(29, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(26, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(27, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(28, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(29, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(26, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(26, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(27, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(27, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(28, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(28, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(29, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(29, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(26, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(26, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(27, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(27, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(28, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(28, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(29, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(29, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(26, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(26, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(26, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(27, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(27, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(27, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(28, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(28, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(28, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(29, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(29, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(29, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(26, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(26, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(26, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(26, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(27, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(27, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(27, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(27, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(28, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(28, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(28, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(28, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(29, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(29, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(29, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(29, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(26, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(26, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(27, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(27, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(28, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(28, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(29, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(29, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(26, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(26, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(26, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(27, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(27, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(27, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(28, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(28, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(28, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(29, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(29, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(29, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(26, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(26, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(26, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(26, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(27, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(27, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(27, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(27, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(28, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(28, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(28, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(28, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(29, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(29, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(29, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(29, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(30, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(31, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(30, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(30, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(31, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(31, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(30, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(31, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(30, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(30, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(30, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(31, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(31, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(31, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(30, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(31, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(30, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(30, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(30, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(31, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(31, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(31, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(30, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(30, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(31, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(31, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(30, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(30, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(31, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(31, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(30, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(31, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(30, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(30, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(31, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(31, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(30, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(30, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(31, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(31, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(30, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(30, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(30, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(31, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(31, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(31, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(30, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(30, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(30, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(30, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(31, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(31, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(31, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(31, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(30, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(30, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(31, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(31, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(30, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(30, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(30, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(31, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(31, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(31, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(30, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(30, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(30, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(30, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(31, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(31, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(31, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(31, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(32, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(33, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(34, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(32, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(32, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(33, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(33, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(34, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(34, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(32, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(33, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(34, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(32, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(32, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(32, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(33, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(33, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(33, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(34, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(34, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(34, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(32, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(33, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(34, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(32, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(32, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(32, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(33, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(33, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(33, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(34, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(34, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(34, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(32, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(32, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(33, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(33, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(34, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(34, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(32, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(32, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(33, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(33, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(34, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(34, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(32, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(33, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(34, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(32, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(32, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(33, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(33, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(34, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(34, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(32, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(32, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(33, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(33, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(34, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(34, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(32, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(32, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(32, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(33, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(33, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(33, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(34, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(34, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(34, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(32, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(32, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(32, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(32, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(33, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(33, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(33, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(33, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(34, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(34, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(34, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(34, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(32, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(32, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(33, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(33, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(34, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(34, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(32, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(32, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(32, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(33, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(33, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(33, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(34, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(34, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(34, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(32, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(32, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(32, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(32, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(33, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(33, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(33, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(33, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(34, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(34, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(34, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(34, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(35, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(36, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(37, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(38, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(35, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(35, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(36, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(36, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(37, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(37, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(38, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(38, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(35, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(36, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(37, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(38, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(35, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(35, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(35, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(36, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(36, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(36, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(37, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(37, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(37, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(38, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(38, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(38, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(35, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(36, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(37, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(38, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(35, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(35, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(35, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(36, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(36, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(36, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(37, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(37, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(37, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(38, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(38, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(38, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(35, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(35, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(36, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(36, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(37, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(37, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(38, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(38, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(35, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(35, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(36, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(36, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(37, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(37, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(38, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(38, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(35, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(36, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(37, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(38, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(35, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(35, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(36, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(36, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(37, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(37, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(38, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(38, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(35, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(35, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(36, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(36, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(37, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(37, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(38, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(38, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(35, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(35, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(35, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(36, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(36, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(36, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(37, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(37, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(37, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(38, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(38, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(38, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(35, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(35, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(35, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(35, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(36, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(36, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(36, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(36, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(37, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(37, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(37, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(37, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(38, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(38, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(38, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(38, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(35, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(35, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(36, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(36, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(37, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(37, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(38, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(38, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(35, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(35, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(35, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(36, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(36, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(36, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(37, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(37, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(37, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(38, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(38, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(38, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(35, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(35, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(35, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(35, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(36, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(36, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(36, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(36, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(37, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(37, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(37, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(37, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(38, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(38, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(38, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(38, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(39, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(39, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(39, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(39, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(39, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(39, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(39, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(39, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(39, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(39, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(39, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(39, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(39, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(39, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(39, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(39, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(39, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(39, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(39, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(39, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(39, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(39, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(39, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(39, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(39, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(39, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(39, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(39, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(40, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(40, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(40, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(41, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(41, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(41, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(40, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(40, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(41, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(41, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(40, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(40, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(41, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(41, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(40, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(41, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(40, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(40, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(41, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(41, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(40, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(40, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(41, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(41, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(40, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(40, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(40, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(41, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(41, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(41, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(40, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(40, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(40, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(40, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(41, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(41, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(41, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(41, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(40, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(40, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(41, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(41, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(40, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(40, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(40, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(41, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(41, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(41, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(40, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(40, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(40, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(40, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(41, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(41, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(41, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(41, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(42, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(42, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(42, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(42, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(42, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(42, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(42, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(42, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(42, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(42, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(42, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(42, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(42, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(42, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(42, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(42, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(42, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(42, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(42, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(42, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(42, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(42, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(42, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(43, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(43, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(44, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(44, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(45, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(45, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(43, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(44, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(45, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(43, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(43, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(44, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(44, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(45, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(45, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(43, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(43, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(44, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(44, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(45, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(45, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(43, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(43, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(43, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(44, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(44, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(44, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[236], y[237], y[238], y[239]))
            {
                output.second.emplace_back(45, 59);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[240], y[241], y[242], y[243]))
            {
                output.second.emplace_back(45, 60);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[244], y[245], y[246], y[247]))
            {
                output.second.emplace_back(45, 61);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(43, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(43, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(43, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(43, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(44, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(44, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(44, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(44, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(45, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(45, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(45, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(45, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(43, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(43, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(44, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(44, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(45, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(45, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(43, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(43, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(43, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(44, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(44, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(44, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(45, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(45, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(45, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(43, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(43, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(43, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(43, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(44, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(44, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(44, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(44, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(45, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(45, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(45, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(45, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(46, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(46, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(46, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(46, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(46, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(46, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(46, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(46, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(47, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(47, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(47, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(47, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(48, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(48, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(48, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(48, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(49, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(49, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(49, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(49, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(47, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(47, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(47, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(47, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(48, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(48, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(48, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(48, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(49, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(49, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(49, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(49, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(50, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(50, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(50, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(50, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[248], y[249], y[250], y[251]))
            {
                output.second.emplace_back(51, 62);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[252], y[253], y[254], y[255]))
            {
                output.second.emplace_back(51, 63);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[256], y[257], y[258], y[259]))
            {
                output.second.emplace_back(51, 64);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(51, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(50, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(50, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(50, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(50, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(51, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(51, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(51, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(51, 74);
            }

            return output;
        }

        template <std::size_t rake>
        inline static bool fkcc(
            const vamp::collision::Environment<FloatVector<rake>> &environment,
            const ConfigurationBlock<rake> &x) noexcept
        {
            std::array<FloatVector<rake, 1>, 51> v;
            std::array<FloatVector<rake, 1>, 432> y;

            v[0] = cos(x[0]);
            v[1] = sin(x[0]);
            v[2] = 0.707105482511236 * v[0] + -0.707108079859474 * v[1];
            y[24] = 0.0640272398484633 + 0.069 * v[2];
            v[3] = 0.707108079859474 * v[0] + 0.707105482511236 * v[1];
            y[25] = 0.259027384507773 + 0.069 * v[3];
            v[4] = cos(x[1]);
            v[1] = -v[1];
            v[5] = 0.707105482511236 * v[1] + -0.707108079859474 * v[0];
            v[6] = sin(x[1]);
            v[7] = 4.89663865010925e-12 * v[6];
            v[8] = v[2] * v[4] + v[5] * v[7];
            v[9] = cos(x[2]);
            v[10] = sin(x[2]);
            v[11] = 4.89663865010925e-12 * v[9] + -4.89658313895802e-12 * v[10];
            v[12] = -v[6];
            v[13] = 4.89663865010925e-12 * v[4];
            v[2] = v[2] * v[12] + v[5] * v[13];
            v[14] = 5.55111512312578e-17 * v[9] + v[10];
            v[15] = v[8] * v[11] + v[2] * v[9] + v[5] * v[14];
            v[16] = v[8] + -4.89658313895802e-12 * v[2] + 4.89663865010925e-12 * v[5];
            y[36] = y[24] + 0.102 * v[8];
            y[28] = -0.02 * v[15] + 0.22 * v[16] + y[36];
            v[1] = 0.707108079859474 * v[1] + 0.707105482511236 * v[0];
            v[7] = v[3] * v[4] + v[1] * v[7];
            v[13] = v[3] * v[12] + v[1] * v[13];
            v[12] = v[7] * v[11] + v[13] * v[9] + v[1] * v[14];
            v[3] = v[7] + -4.89658313895802e-12 * v[13] + 4.89663865010925e-12 * v[1];
            y[37] = y[25] + 0.102 * v[7];
            y[29] = -0.02 * v[12] + 0.22 * v[3] + y[37];
            v[6] = -1. * v[6];
            v[4] = -1. * v[4];
            v[14] = v[6] * v[11] + v[4] * v[9] + 4.89663865010925e-12 * v[14];
            v[11] = v[6] + 2.39770700697438e-23 + -4.89658313895802e-12 * v[4];
            y[38] = 0.399976 + 0.102 * v[6];
            y[30] = -0.02 * v[14] + 0.22 * v[11] + y[38];
            y[32] = -0.01 * v[15] + 0.11 * v[16] + y[36];
            y[33] = -0.01 * v[12] + 0.11 * v[3] + y[37];
            y[34] = -0.01 * v[14] + 0.11 * v[11] + y[38];
            y[40] = y[36] + 0.069 * v[15] + 0.26242 * v[16];
            y[41] = y[37] + 0.069 * v[12] + 0.26242 * v[3];
            y[42] = y[38] + 0.069 * v[14] + 0.26242 * v[11];
            v[0] = cos(x[3]);
            v[17] = sin(x[3]);
            v[18] = 4.89663865010925e-12 * v[0] + v[17];
            v[10] = -v[10];
            v[19] = 4.89663865010925e-12 * v[10] + -4.89658313895802e-12 * v[9];
            v[9] = 5.55111512312578e-17 * v[10] + v[9];
            v[2] = v[8] * v[19] + v[2] * v[10] + v[5] * v[9];
            v[8] = 5.55111512312578e-17 * v[0] + 4.89663865010925e-12 * v[17];
            v[5] = v[0] + -4.89658313895802e-12 * v[17];
            v[20] = v[15] * v[18] + v[2] * v[8] + v[16] * v[5];
            y[44] = y[40] + 0.10359 * v[20];
            v[13] = v[7] * v[19] + v[13] * v[10] + v[1] * v[9];
            v[7] = v[12] * v[18] + v[13] * v[8] + v[3] * v[5];
            y[45] = y[41] + 0.10359 * v[7];
            v[9] = v[6] * v[19] + v[4] * v[10] + 4.89663865010925e-12 * v[9];
            v[5] = v[14] * v[18] + v[9] * v[8] + v[11] * v[5];
            y[46] = y[42] + 0.10359 * v[5];
            v[17] = -v[17];
            v[8] = 4.89663865010925e-12 * v[17] + v[0];
            v[18] = 5.55111512312578e-17 * v[17] + 4.89663865010925e-12 * v[0];
            v[17] = v[17] + -4.89658313895802e-12 * v[0];
            v[0] = v[15] * v[8] + v[2] * v[18] + v[16] * v[17];
            v[2] = -4.89658313895802e-12 * v[15] + v[2];
            v[19] = v[20] + -4.89658313895802e-12 * v[0] + 4.89663865010925e-12 * v[2];
            y[48] = 0.22 * v[19] + y[44];
            v[10] = v[12] * v[8] + v[13] * v[18] + v[3] * v[17];
            v[13] = -4.89658313895802e-12 * v[12] + v[13];
            v[4] = v[7] + -4.89658313895802e-12 * v[10] + 4.89663865010925e-12 * v[13];
            y[49] = 0.22 * v[4] + y[45];
            v[17] = v[14] * v[8] + v[9] * v[18] + v[11] * v[17];
            v[9] = -4.89658313895802e-12 * v[14] + v[9];
            v[18] = v[5] + -4.89658313895802e-12 * v[17] + 4.89663865010925e-12 * v[9];
            y[50] = 0.22 * v[18] + y[46];
            y[52] = 0.11 * v[19] + y[44];
            y[53] = 0.11 * v[4] + y[45];
            y[54] = 0.11 * v[18] + y[46];
            v[8] = cos(x[4]);
            v[6] = sin(x[4]);
            v[1] = 4.89663865010925e-12 * v[8] + -4.89658313895802e-12 * v[6];
            v[21] = 5.55111512312578e-17 * v[8] + v[6];
            v[22] = v[20] * v[1] + v[0] * v[8] + v[2] * v[21];
            v[6] = -v[6];
            v[23] = 4.89663865010925e-12 * v[6] + -4.89658313895802e-12 * v[8];
            v[24] = 5.55111512312578e-17 * v[6] + v[8];
            v[2] = v[20] * v[23] + v[0] * v[6] + v[2] * v[24];
            v[0] = -4.89658313895802e-12 * v[22] + v[2];
            y[328] = y[44] + 0.01 * v[22] + 0.2707 * v[19];
            y[56] = 0.03 * v[0] + y[328];
            v[20] = v[7] * v[1] + v[10] * v[8] + v[13] * v[21];
            v[13] = v[7] * v[23] + v[10] * v[6] + v[13] * v[24];
            v[10] = -4.89658313895802e-12 * v[20] + v[13];
            y[329] = y[45] + 0.01 * v[20] + 0.2707 * v[4];
            y[57] = 0.03 * v[10] + y[329];
            v[21] = v[5] * v[1] + v[17] * v[8] + v[9] * v[21];
            v[24] = v[5] * v[23] + v[17] * v[6] + v[9] * v[24];
            v[23] = -4.89658313895802e-12 * v[21] + v[24];
            y[330] = y[46] + 0.01 * v[21] + 0.2707 * v[18];
            y[58] = 0.03 * v[23] + y[330];
            y[60] = -0.03 * v[0] + y[328];
            y[61] = -0.03 * v[10] + y[329];
            y[62] = -0.03 * v[23] + y[330];
            v[6] = cos(x[5]);
            v[9] = sin(x[5]);
            v[17] = 4.89663865010925e-12 * v[6] + v[9];
            v[5] = 5.55111512312578e-17 * v[6] + 4.89663865010925e-12 * v[9];
            v[1] = v[6] + -4.89658313895802e-12 * v[9];
            v[8] = v[22] * v[17] + v[2] * v[5] + v[19] * v[1];
            v[9] = -v[9];
            v[7] = 4.89663865010925e-12 * v[9] + v[6];
            v[25] = 5.55111512312578e-17 * v[9] + 4.89663865010925e-12 * v[6];
            v[9] = v[9] + -4.89658313895802e-12 * v[6];
            v[2] = v[22] * v[7] + v[2] * v[25] + v[19] * v[9];
            v[22] = v[8] + -4.89658313895802e-12 * v[2] + 4.89663865010925e-12 * v[0];
            v[6] = y[328] + 0.115975 * v[8];
            y[64] = 0.02 * v[22] + v[6];
            v[26] = v[20] * v[17] + v[13] * v[5] + v[4] * v[1];
            v[13] = v[20] * v[7] + v[13] * v[25] + v[4] * v[9];
            v[20] = v[26] + -4.89658313895802e-12 * v[13] + 4.89663865010925e-12 * v[10];
            v[27] = y[329] + 0.115975 * v[26];
            y[65] = 0.02 * v[20] + v[27];
            v[1] = v[21] * v[17] + v[24] * v[5] + v[18] * v[1];
            v[9] = v[21] * v[7] + v[24] * v[25] + v[18] * v[9];
            v[25] = v[1] + -4.89658313895802e-12 * v[9] + 4.89663865010925e-12 * v[23];
            v[7] = y[330] + 0.115975 * v[1];
            y[66] = 0.02 * v[25] + v[7];
            y[68] = -0.04 * v[22] + v[6];
            y[69] = -0.04 * v[20] + v[27];
            y[70] = -0.04 * v[25] + v[7];
            v[24] = cos(x[6]);
            v[21] = sin(x[6]);
            v[5] = 4.89663865010925e-12 * v[24] + -4.89658313895802e-12 * v[21];
            v[17] = 5.55111512312578e-17 * v[24] + v[21];
            v[28] = v[8] * v[5] + v[2] * v[24] + v[0] * v[17];
            y[72] = 0.01 * v[28] + 0.09355 * v[22] + v[6];
            v[29] = v[26] * v[5] + v[13] * v[24] + v[10] * v[17];
            y[73] = 0.01 * v[29] + 0.09355 * v[20] + v[27];
            v[17] = v[1] * v[5] + v[9] * v[24] + v[23] * v[17];
            y[74] = 0.01 * v[17] + 0.09355 * v[25] + v[7];
            v[21] = -v[21];
            v[5] = 4.89663865010925e-12 * v[21] + -4.89658313895802e-12 * v[24];
            v[24] = 5.55111512312578e-17 * v[21] + v[24];
            v[2] = v[8] * v[5] + v[2] * v[21] + v[0] * v[24];
            y[76] = 0.02 * v[2] + 0.13855 * v[22] + v[6];
            v[13] = v[26] * v[5] + v[13] * v[21] + v[10] * v[24];
            y[77] = 0.02 * v[13] + 0.13855 * v[20] + v[27];
            v[24] = v[1] * v[5] + v[9] * v[21] + v[23] * v[24];
            y[78] = 0.02 * v[24] + 0.13855 * v[25] + v[7];
            y[80] = -0.02 * v[2] + 0.13855 * v[22] + v[6];
            y[81] = -0.02 * v[13] + 0.13855 * v[20] + v[27];
            y[82] = -0.02 * v[24] + 0.13855 * v[25] + v[7];
            y[84] = -0.005 * v[28] + 0.081333 * v[2] + 0.16655 * v[22] + v[6];
            y[85] = -0.005 * v[29] + 0.081333 * v[13] + 0.16655 * v[20] + v[27];
            y[86] = -0.005 * v[17] + 0.081333 * v[24] + 0.16655 * v[25] + v[7];
            y[88] = -0.005 * v[28] + 0.057333 * v[2] + 0.16655 * v[22] + v[6];
            y[89] = -0.005 * v[29] + 0.057333 * v[13] + 0.16655 * v[20] + v[27];
            y[90] = -0.005 * v[17] + 0.057333 * v[24] + 0.16655 * v[25] + v[7];
            y[92] = 0.086583 * v[2] + 0.18855 * v[22] + v[6];
            y[93] = 0.086583 * v[13] + 0.18855 * v[20] + v[27];
            y[94] = 0.086583 * v[24] + 0.18855 * v[25] + v[7];
            y[96] = 0.086583 * v[2] + 0.20855 * v[22] + v[6];
            y[97] = 0.086583 * v[13] + 0.20855 * v[20] + v[27];
            y[98] = 0.086583 * v[24] + 0.20855 * v[25] + v[7];
            y[100] = 0.086583 * v[2] + 0.22855 * v[22] + v[6];
            y[101] = 0.086583 * v[13] + 0.22855 * v[20] + v[27];
            y[102] = 0.086583 * v[24] + 0.22855 * v[25] + v[7];
            y[104] = 0.01 * v[28] + 0.082083 * v[2] + 0.26625 * v[22] + v[6];
            y[105] = 0.01 * v[29] + 0.082083 * v[13] + 0.26625 * v[20] + v[27];
            y[106] = 0.01 * v[17] + 0.082083 * v[24] + 0.26625 * v[25] + v[7];
            y[108] = -0.01 * v[28] + 0.082083 * v[2] + 0.26625 * v[22] + v[6];
            y[109] = -0.01 * v[29] + 0.082083 * v[13] + 0.26625 * v[20] + v[27];
            y[110] = -0.01 * v[17] + 0.082083 * v[24] + 0.26625 * v[25] + v[7];
            y[112] = -0.01 * v[28] + 0.082083 * v[2] + 0.24625 * v[22] + v[6];
            y[113] = -0.01 * v[29] + 0.082083 * v[13] + 0.24625 * v[20] + v[27];
            y[114] = -0.01 * v[17] + 0.082083 * v[24] + 0.24625 * v[25] + v[7];
            y[116] = 0.01 * v[28] + 0.082083 * v[2] + 0.24625 * v[22] + v[6];
            y[117] = 0.01 * v[29] + 0.082083 * v[13] + 0.24625 * v[20] + v[27];
            y[118] = 0.01 * v[17] + 0.082083 * v[24] + 0.24625 * v[25] + v[7];
            y[120] = 0.005 * v[28] + -0.059333 * v[2] + 0.16655 * v[22] + v[6];
            y[121] = 0.005 * v[29] + -0.059333 * v[13] + 0.16655 * v[20] + v[27];
            y[122] = 0.005 * v[17] + -0.059333 * v[24] + 0.16655 * v[25] + v[7];
            y[124] = 0.005 * v[28] + -0.079333 * v[2] + 0.16655 * v[22] + v[6];
            y[125] = 0.005 * v[29] + -0.079333 * v[13] + 0.16655 * v[20] + v[27];
            y[126] = 0.005 * v[17] + -0.079333 * v[24] + 0.16655 * v[25] + v[7];
            y[128] = -0.086583 * v[2] + 0.18855 * v[22] + v[6];
            y[129] = -0.086583 * v[13] + 0.18855 * v[20] + v[27];
            y[130] = -0.086583 * v[24] + 0.18855 * v[25] + v[7];
            y[132] = -0.086583 * v[2] + 0.20855 * v[22] + v[6];
            y[133] = -0.086583 * v[13] + 0.20855 * v[20] + v[27];
            y[134] = -0.086583 * v[24] + 0.20855 * v[25] + v[7];
            y[136] = -0.086583 * v[2] + 0.22855 * v[22] + v[6];
            y[137] = -0.086583 * v[13] + 0.22855 * v[20] + v[27];
            y[138] = -0.086583 * v[24] + 0.22855 * v[25] + v[7];
            y[140] = 0.01 * v[28] + -0.082083 * v[2] + 0.26625 * v[22] + v[6];
            y[141] = 0.01 * v[29] + -0.082083 * v[13] + 0.26625 * v[20] + v[27];
            y[142] = 0.01 * v[17] + -0.082083 * v[24] + 0.26625 * v[25] + v[7];
            y[144] = -0.01 * v[28] + -0.082083 * v[2] + 0.26625 * v[22] + v[6];
            y[145] = -0.01 * v[29] + -0.082083 * v[13] + 0.26625 * v[20] + v[27];
            y[146] = -0.01 * v[17] + -0.082083 * v[24] + 0.26625 * v[25] + v[7];
            y[148] = -0.01 * v[28] + -0.082083 * v[2] + 0.24625 * v[22] + v[6];
            y[149] = -0.01 * v[29] + -0.082083 * v[13] + 0.24625 * v[20] + v[27];
            y[150] = -0.01 * v[17] + -0.082083 * v[24] + 0.24625 * v[25] + v[7];
            y[152] = 0.01 * v[28] + -0.082083 * v[2] + 0.24625 * v[22] + v[6];
            y[153] = 0.01 * v[29] + -0.082083 * v[13] + 0.24625 * v[20] + v[27];
            y[154] = 0.01 * v[17] + -0.082083 * v[24] + 0.24625 * v[25] + v[7];
            v[5] = cos(x[7]);
            v[21] = sin(x[7]);
            v[9] = 0.707105482511236 * v[5] + 0.707108079859474 * v[21];
            y[168] = 0.0640272398484633 + 0.069 * v[9];
            v[1] = -0.707108079859474 * v[5] + 0.707105482511236 * v[21];
            y[169] = -0.259027384507773 + 0.069 * v[1];
            v[23] = cos(x[8]);
            v[21] = -v[21];
            v[26] = 0.707105482511236 * v[21] + 0.707108079859474 * v[5];
            v[10] = sin(x[8]);
            v[8] = 4.89663865010925e-12 * v[10];
            v[0] = v[9] * v[23] + v[26] * v[8];
            v[30] = cos(x[9]);
            v[31] = sin(x[9]);
            v[32] = 4.89663865010925e-12 * v[30] + -4.89658313895802e-12 * v[31];
            v[33] = -v[10];
            v[34] = 4.89663865010925e-12 * v[23];
            v[9] = v[9] * v[33] + v[26] * v[34];
            v[35] = 5.55111512312578e-17 * v[30] + v[31];
            v[36] = v[0] * v[32] + v[9] * v[30] + v[26] * v[35];
            v[37] = v[0] + -4.89658313895802e-12 * v[9] + 4.89663865010925e-12 * v[26];
            y[180] = y[168] + 0.102 * v[0];
            y[172] = -0.02 * v[36] + 0.22 * v[37] + y[180];
            v[21] = -0.707108079859474 * v[21] + 0.707105482511236 * v[5];
            v[8] = v[1] * v[23] + v[21] * v[8];
            v[34] = v[1] * v[33] + v[21] * v[34];
            v[33] = v[8] * v[32] + v[34] * v[30] + v[21] * v[35];
            v[1] = v[8] + -4.89658313895802e-12 * v[34] + 4.89663865010925e-12 * v[21];
            y[181] = y[169] + 0.102 * v[8];
            y[173] = -0.02 * v[33] + 0.22 * v[1] + y[181];
            v[10] = -1. * v[10];
            v[23] = -1. * v[23];
            v[35] = v[10] * v[32] + v[23] * v[30] + 4.89663865010925e-12 * v[35];
            v[32] = v[10] + 2.39770700697438e-23 + -4.89658313895802e-12 * v[23];
            y[182] = 0.399976 + 0.102 * v[10];
            y[174] = -0.02 * v[35] + 0.22 * v[32] + y[182];
            y[176] = -0.01 * v[36] + 0.11 * v[37] + y[180];
            y[177] = -0.01 * v[33] + 0.11 * v[1] + y[181];
            y[178] = -0.01 * v[35] + 0.11 * v[32] + y[182];
            y[184] = y[180] + 0.069 * v[36] + 0.26242 * v[37];
            y[185] = y[181] + 0.069 * v[33] + 0.26242 * v[1];
            y[186] = y[182] + 0.069 * v[35] + 0.26242 * v[32];
            v[5] = cos(x[10]);
            v[38] = sin(x[10]);
            v[39] = 4.89663865010925e-12 * v[5] + v[38];
            v[31] = -v[31];
            v[40] = 4.89663865010925e-12 * v[31] + -4.89658313895802e-12 * v[30];
            v[30] = 5.55111512312578e-17 * v[31] + v[30];
            v[9] = v[0] * v[40] + v[9] * v[31] + v[26] * v[30];
            v[0] = 5.55111512312578e-17 * v[5] + 4.89663865010925e-12 * v[38];
            v[26] = v[5] + -4.89658313895802e-12 * v[38];
            v[41] = v[36] * v[39] + v[9] * v[0] + v[37] * v[26];
            y[188] = y[184] + 0.10359 * v[41];
            v[34] = v[8] * v[40] + v[34] * v[31] + v[21] * v[30];
            v[8] = v[33] * v[39] + v[34] * v[0] + v[1] * v[26];
            y[189] = y[185] + 0.10359 * v[8];
            v[30] = v[10] * v[40] + v[23] * v[31] + 4.89663865010925e-12 * v[30];
            v[26] = v[35] * v[39] + v[30] * v[0] + v[32] * v[26];
            y[190] = y[186] + 0.10359 * v[26];
            v[38] = -v[38];
            v[0] = 4.89663865010925e-12 * v[38] + v[5];
            v[39] = 5.55111512312578e-17 * v[38] + 4.89663865010925e-12 * v[5];
            v[38] = v[38] + -4.89658313895802e-12 * v[5];
            v[5] = v[36] * v[0] + v[9] * v[39] + v[37] * v[38];
            v[9] = -4.89658313895802e-12 * v[36] + v[9];
            v[40] = v[41] + -4.89658313895802e-12 * v[5] + 4.89663865010925e-12 * v[9];
            y[192] = 0.22 * v[40] + y[188];
            v[31] = v[33] * v[0] + v[34] * v[39] + v[1] * v[38];
            v[34] = -4.89658313895802e-12 * v[33] + v[34];
            v[23] = v[8] + -4.89658313895802e-12 * v[31] + 4.89663865010925e-12 * v[34];
            y[193] = 0.22 * v[23] + y[189];
            v[38] = v[35] * v[0] + v[30] * v[39] + v[32] * v[38];
            v[30] = -4.89658313895802e-12 * v[35] + v[30];
            v[39] = v[26] + -4.89658313895802e-12 * v[38] + 4.89663865010925e-12 * v[30];
            y[194] = 0.22 * v[39] + y[190];
            y[196] = 0.11 * v[40] + y[188];
            y[197] = 0.11 * v[23] + y[189];
            y[198] = 0.11 * v[39] + y[190];
            v[0] = cos(x[11]);
            v[10] = sin(x[11]);
            v[21] = 4.89663865010925e-12 * v[0] + -4.89658313895802e-12 * v[10];
            v[42] = 5.55111512312578e-17 * v[0] + v[10];
            v[43] = v[41] * v[21] + v[5] * v[0] + v[9] * v[42];
            v[10] = -v[10];
            v[44] = 4.89663865010925e-12 * v[10] + -4.89658313895802e-12 * v[0];
            v[45] = 5.55111512312578e-17 * v[10] + v[0];
            v[9] = v[41] * v[44] + v[5] * v[10] + v[9] * v[45];
            v[5] = -4.89658313895802e-12 * v[43] + v[9];
            y[392] = y[188] + 0.01 * v[43] + 0.2707 * v[40];
            y[200] = 0.03 * v[5] + y[392];
            v[41] = v[8] * v[21] + v[31] * v[0] + v[34] * v[42];
            v[34] = v[8] * v[44] + v[31] * v[10] + v[34] * v[45];
            v[31] = -4.89658313895802e-12 * v[41] + v[34];
            y[393] = y[189] + 0.01 * v[41] + 0.2707 * v[23];
            y[201] = 0.03 * v[31] + y[393];
            v[42] = v[26] * v[21] + v[38] * v[0] + v[30] * v[42];
            v[45] = v[26] * v[44] + v[38] * v[10] + v[30] * v[45];
            v[44] = -4.89658313895802e-12 * v[42] + v[45];
            y[394] = y[190] + 0.01 * v[42] + 0.2707 * v[39];
            y[202] = 0.03 * v[44] + y[394];
            y[204] = -0.03 * v[5] + y[392];
            y[205] = -0.03 * v[31] + y[393];
            y[206] = -0.03 * v[44] + y[394];
            v[10] = cos(x[12]);
            v[30] = sin(x[12]);
            v[38] = 4.89663865010925e-12 * v[10] + v[30];
            v[26] = 5.55111512312578e-17 * v[10] + 4.89663865010925e-12 * v[30];
            v[21] = v[10] + -4.89658313895802e-12 * v[30];
            v[0] = v[43] * v[38] + v[9] * v[26] + v[40] * v[21];
            v[30] = -v[30];
            v[8] = 4.89663865010925e-12 * v[30] + v[10];
            v[46] = 5.55111512312578e-17 * v[30] + 4.89663865010925e-12 * v[10];
            v[30] = v[30] + -4.89658313895802e-12 * v[10];
            v[9] = v[43] * v[8] + v[9] * v[46] + v[40] * v[30];
            v[43] = v[0] + -4.89658313895802e-12 * v[9] + 4.89663865010925e-12 * v[5];
            v[10] = y[392] + 0.115975 * v[0];
            y[208] = 0.02 * v[43] + v[10];
            v[47] = v[41] * v[38] + v[34] * v[26] + v[23] * v[21];
            v[34] = v[41] * v[8] + v[34] * v[46] + v[23] * v[30];
            v[41] = v[47] + -4.89658313895802e-12 * v[34] + 4.89663865010925e-12 * v[31];
            v[48] = y[393] + 0.115975 * v[47];
            y[209] = 0.02 * v[41] + v[48];
            v[21] = v[42] * v[38] + v[45] * v[26] + v[39] * v[21];
            v[30] = v[42] * v[8] + v[45] * v[46] + v[39] * v[30];
            v[46] = v[21] + -4.89658313895802e-12 * v[30] + 4.89663865010925e-12 * v[44];
            v[8] = y[394] + 0.115975 * v[21];
            y[210] = 0.02 * v[46] + v[8];
            y[212] = -0.04 * v[43] + v[10];
            y[213] = -0.04 * v[41] + v[48];
            y[214] = -0.04 * v[46] + v[8];
            v[45] = cos(x[13]);
            v[42] = sin(x[13]);
            v[26] = 4.89663865010925e-12 * v[45] + -4.89658313895802e-12 * v[42];
            v[38] = 5.55111512312578e-17 * v[45] + v[42];
            v[49] = v[0] * v[26] + v[9] * v[45] + v[5] * v[38];
            y[216] = 0.01 * v[49] + 0.09355 * v[43] + v[10];
            v[50] = v[47] * v[26] + v[34] * v[45] + v[31] * v[38];
            y[217] = 0.01 * v[50] + 0.09355 * v[41] + v[48];
            v[38] = v[21] * v[26] + v[30] * v[45] + v[44] * v[38];
            y[218] = 0.01 * v[38] + 0.09355 * v[46] + v[8];
            v[42] = -v[42];
            v[26] = 4.89663865010925e-12 * v[42] + -4.89658313895802e-12 * v[45];
            v[45] = 5.55111512312578e-17 * v[42] + v[45];
            v[9] = v[0] * v[26] + v[9] * v[42] + v[5] * v[45];
            y[220] = 0.02 * v[9] + 0.13855 * v[43] + v[10];
            v[34] = v[47] * v[26] + v[34] * v[42] + v[31] * v[45];
            y[221] = 0.02 * v[34] + 0.13855 * v[41] + v[48];
            v[45] = v[21] * v[26] + v[30] * v[42] + v[44] * v[45];
            y[222] = 0.02 * v[45] + 0.13855 * v[46] + v[8];
            y[224] = -0.02 * v[9] + 0.13855 * v[43] + v[10];
            y[225] = -0.02 * v[34] + 0.13855 * v[41] + v[48];
            y[226] = -0.02 * v[45] + 0.13855 * v[46] + v[8];
            y[228] = -0.005 * v[49] + 0.081333 * v[9] + 0.16655 * v[43] + v[10];
            y[229] = -0.005 * v[50] + 0.081333 * v[34] + 0.16655 * v[41] + v[48];
            y[230] = -0.005 * v[38] + 0.081333 * v[45] + 0.16655 * v[46] + v[8];
            y[232] = -0.005 * v[49] + 0.057333 * v[9] + 0.16655 * v[43] + v[10];
            y[233] = -0.005 * v[50] + 0.057333 * v[34] + 0.16655 * v[41] + v[48];
            y[234] = -0.005 * v[38] + 0.057333 * v[45] + 0.16655 * v[46] + v[8];
            y[236] = 0.086583 * v[9] + 0.18855 * v[43] + v[10];
            y[237] = 0.086583 * v[34] + 0.18855 * v[41] + v[48];
            y[238] = 0.086583 * v[45] + 0.18855 * v[46] + v[8];
            y[240] = 0.086583 * v[9] + 0.20855 * v[43] + v[10];
            y[241] = 0.086583 * v[34] + 0.20855 * v[41] + v[48];
            y[242] = 0.086583 * v[45] + 0.20855 * v[46] + v[8];
            y[244] = 0.086583 * v[9] + 0.22855 * v[43] + v[10];
            y[245] = 0.086583 * v[34] + 0.22855 * v[41] + v[48];
            y[246] = 0.086583 * v[45] + 0.22855 * v[46] + v[8];
            y[248] = 0.01 * v[49] + 0.082083 * v[9] + 0.26625 * v[43] + v[10];
            y[249] = 0.01 * v[50] + 0.082083 * v[34] + 0.26625 * v[41] + v[48];
            y[250] = 0.01 * v[38] + 0.082083 * v[45] + 0.26625 * v[46] + v[8];
            y[252] = -0.01 * v[49] + 0.082083 * v[9] + 0.26625 * v[43] + v[10];
            y[253] = -0.01 * v[50] + 0.082083 * v[34] + 0.26625 * v[41] + v[48];
            y[254] = -0.01 * v[38] + 0.082083 * v[45] + 0.26625 * v[46] + v[8];
            y[256] = -0.01 * v[49] + 0.082083 * v[9] + 0.24625 * v[43] + v[10];
            y[257] = -0.01 * v[50] + 0.082083 * v[34] + 0.24625 * v[41] + v[48];
            y[258] = -0.01 * v[38] + 0.082083 * v[45] + 0.24625 * v[46] + v[8];
            y[260] = 0.01 * v[49] + 0.082083 * v[9] + 0.24625 * v[43] + v[10];
            y[261] = 0.01 * v[50] + 0.082083 * v[34] + 0.24625 * v[41] + v[48];
            y[262] = 0.01 * v[38] + 0.082083 * v[45] + 0.24625 * v[46] + v[8];
            y[264] = 0.005 * v[49] + -0.059333 * v[9] + 0.16655 * v[43] + v[10];
            y[265] = 0.005 * v[50] + -0.059333 * v[34] + 0.16655 * v[41] + v[48];
            y[266] = 0.005 * v[38] + -0.059333 * v[45] + 0.16655 * v[46] + v[8];
            y[268] = 0.005 * v[49] + -0.079333 * v[9] + 0.16655 * v[43] + v[10];
            y[269] = 0.005 * v[50] + -0.079333 * v[34] + 0.16655 * v[41] + v[48];
            y[270] = 0.005 * v[38] + -0.079333 * v[45] + 0.16655 * v[46] + v[8];
            y[272] = -0.086583 * v[9] + 0.18855 * v[43] + v[10];
            y[273] = -0.086583 * v[34] + 0.18855 * v[41] + v[48];
            y[274] = -0.086583 * v[45] + 0.18855 * v[46] + v[8];
            y[276] = -0.086583 * v[9] + 0.20855 * v[43] + v[10];
            y[277] = -0.086583 * v[34] + 0.20855 * v[41] + v[48];
            y[278] = -0.086583 * v[45] + 0.20855 * v[46] + v[8];
            y[280] = -0.086583 * v[9] + 0.22855 * v[43] + v[10];
            y[281] = -0.086583 * v[34] + 0.22855 * v[41] + v[48];
            y[282] = -0.086583 * v[45] + 0.22855 * v[46] + v[8];
            y[284] = 0.01 * v[49] + -0.082083 * v[9] + 0.26625 * v[43] + v[10];
            y[285] = 0.01 * v[50] + -0.082083 * v[34] + 0.26625 * v[41] + v[48];
            y[286] = 0.01 * v[38] + -0.082083 * v[45] + 0.26625 * v[46] + v[8];
            y[288] = -0.01 * v[49] + -0.082083 * v[9] + 0.26625 * v[43] + v[10];
            y[289] = -0.01 * v[50] + -0.082083 * v[34] + 0.26625 * v[41] + v[48];
            y[290] = -0.01 * v[38] + -0.082083 * v[45] + 0.26625 * v[46] + v[8];
            y[292] = -0.01 * v[49] + -0.082083 * v[9] + 0.24625 * v[43] + v[10];
            y[293] = -0.01 * v[50] + -0.082083 * v[34] + 0.24625 * v[41] + v[48];
            y[294] = -0.01 * v[38] + -0.082083 * v[45] + 0.24625 * v[46] + v[8];
            y[296] = 0.01 * v[49] + -0.082083 * v[9] + 0.24625 * v[43] + v[10];
            y[297] = 0.01 * v[50] + -0.082083 * v[34] + 0.24625 * v[41] + v[48];
            y[298] = 0.01 * v[38] + -0.082083 * v[45] + 0.24625 * v[46] + v[8];
            y[316] = -0.00999999977648258 * v[15] + 0.109999999403954 * v[16] + y[36];
            y[317] = -0.00999999977648258 * v[12] + 0.109999999403954 * v[3] + y[37];
            y[318] = -0.00999999977648258 * v[14] + 0.109999999403954 * v[11] + y[38];
            y[324] = 0.109999999403954 * v[19] + y[44];
            y[325] = 0.109999999403954 * v[4] + y[45];
            y[326] = 0.109999999403954 * v[18] + y[46];
            y[332] = -0.0149999987334013 * v[22] + v[6];
            y[333] = -0.0149999987334013 * v[20] + v[27];
            y[334] = -0.0149999987334013 * v[25] + v[7];
            y[336] = 0.00999999977648258 * v[28] + 0.0935499966144562 * v[22] + v[6];
            y[337] = 0.00999999977648258 * v[29] + 0.0935499966144562 * v[20] + v[27];
            y[338] = 0.00999999977648258 * v[17] + 0.0935499966144562 * v[25] + v[7];
            y[340] = 0.138549998402596 * v[22] + v[6];
            y[341] = 0.138549998402596 * v[20] + v[27];
            y[342] = 0.138549998402596 * v[25] + v[7];
            y[344] =
                -0.00499999988824129 * v[28] + 0.0693330019712448 * v[2] + 0.166549995541573 * v[22] + v[6];
            y[345] =
                -0.00499999988824129 * v[29] + 0.0693330019712448 * v[13] + 0.166549995541573 * v[20] + v[27];
            y[346] =
                -0.00499999988824129 * v[17] + 0.0693330019712448 * v[24] + 0.166549995541573 * v[25] + v[7];
            y[348] = 0.0865830034017563 * v[2] + 0.208550006151199 * v[22] + v[6];
            y[349] = 0.0865830034017563 * v[13] + 0.208550006151199 * v[20] + v[27];
            y[350] = 0.0865830034017563 * v[24] + 0.208550006151199 * v[25] + v[7];
            y[352] = 0.0820830017328262 * v[2] + 0.256249994039536 * v[22] + v[6];
            y[353] = 0.0820830017328262 * v[13] + 0.256249994039536 * v[20] + v[27];
            y[354] = 0.0820830017328262 * v[24] + 0.256249994039536 * v[25] + v[7];
            y[356] =
                0.00499999988824129 * v[28] + -0.0693330019712448 * v[2] + 0.166549995541573 * v[22] + v[6];
            y[357] =
                0.00499999988824129 * v[29] + -0.0693330019712448 * v[13] + 0.166549995541573 * v[20] + v[27];
            y[358] =
                0.00499999988824129 * v[17] + -0.0693330019712448 * v[24] + 0.166549995541573 * v[25] + v[7];
            y[360] = -0.0865830034017563 * v[2] + 0.208550006151199 * v[22] + v[6];
            y[361] = -0.0865830034017563 * v[13] + 0.208550006151199 * v[20] + v[27];
            y[362] = -0.0865830034017563 * v[24] + 0.208550006151199 * v[25] + v[7];
            y[364] = -0.0820830017328262 * v[2] + 0.256249994039536 * v[22] + v[6];
            y[365] = -0.0820830017328262 * v[13] + 0.256249994039536 * v[20] + v[27];
            y[366] = -0.0820830017328262 * v[24] + 0.256249994039536 * v[25] + v[7];
            y[380] = -0.00999999977648258 * v[36] + 0.109999999403954 * v[37] + y[180];
            y[381] = -0.00999999977648258 * v[33] + 0.109999999403954 * v[1] + y[181];
            y[382] = -0.00999999977648258 * v[35] + 0.109999999403954 * v[32] + y[182];
            y[388] = 0.109999999403954 * v[40] + y[188];
            y[389] = 0.109999999403954 * v[23] + y[189];
            y[390] = 0.109999999403954 * v[39] + y[190];
            y[396] = -0.0149999987334013 * v[43] + v[10];
            y[397] = -0.0149999987334013 * v[41] + v[48];
            y[398] = -0.0149999987334013 * v[46] + v[8];
            y[400] = 0.00999999977648258 * v[49] + 0.0935499966144562 * v[43] + v[10];
            y[401] = 0.00999999977648258 * v[50] + 0.0935499966144562 * v[41] + v[48];
            y[402] = 0.00999999977648258 * v[38] + 0.0935499966144562 * v[46] + v[8];
            y[404] = 0.138549998402596 * v[43] + v[10];
            y[405] = 0.138549998402596 * v[41] + v[48];
            y[406] = 0.138549998402596 * v[46] + v[8];
            y[408] =
                -0.00499999988824129 * v[49] + 0.0693330019712448 * v[9] + 0.166549995541573 * v[43] + v[10];
            y[409] =
                -0.00499999988824129 * v[50] + 0.0693330019712448 * v[34] + 0.166549995541573 * v[41] + v[48];
            y[410] =
                -0.00499999988824129 * v[38] + 0.0693330019712448 * v[45] + 0.166549995541573 * v[46] + v[8];
            y[412] = 0.0865830034017563 * v[9] + 0.208550006151199 * v[43] + v[10];
            y[413] = 0.0865830034017563 * v[34] + 0.208550006151199 * v[41] + v[48];
            y[414] = 0.0865830034017563 * v[45] + 0.208550006151199 * v[46] + v[8];
            y[416] = 0.0820830017328262 * v[9] + 0.256249994039536 * v[43] + v[10];
            y[417] = 0.0820830017328262 * v[34] + 0.256249994039536 * v[41] + v[48];
            y[418] = 0.0820830017328262 * v[45] + 0.256249994039536 * v[46] + v[8];
            y[420] =
                0.00499999988824129 * v[49] + -0.0693330019712448 * v[9] + 0.166549995541573 * v[43] + v[10];
            y[421] =
                0.00499999988824129 * v[50] + -0.0693330019712448 * v[34] + 0.166549995541573 * v[41] + v[48];
            y[422] =
                0.00499999988824129 * v[38] + -0.0693330019712448 * v[45] + 0.166549995541573 * v[46] + v[8];
            y[424] = -0.0865830034017563 * v[9] + 0.208550006151199 * v[43] + v[10];
            y[425] = -0.0865830034017563 * v[34] + 0.208550006151199 * v[41] + v[48];
            y[426] = -0.0865830034017563 * v[45] + 0.208550006151199 * v[46] + v[8];
            y[428] = -0.0820830017328262 * v[9] + 0.256249994039536 * v[43] + v[10];
            y[429] = -0.0820830017328262 * v[34] + 0.256249994039536 * v[41] + v[48];
            y[430] = -0.0820830017328262 * v[45] + 0.256249994039536 * v[46] + v[8];
            // variable duplicates: 10
            y[312] = y[24];
            y[313] = y[25];
            y[320] = y[40];
            y[321] = y[41];
            y[322] = y[42];
            y[376] = y[168];
            y[377] = y[169];
            y[384] = y[184];
            y[385] = y[185];
            y[386] = y[186];
            // dependent variables without operations
            y[0] = -0.025;
            y[1] = -0.1;
            y[2] = 0.1;
            y[3] = 0.25;
            y[4] = -0.025;
            y[5] = 0.1;
            y[6] = 0.1;
            y[7] = 0.25;
            y[8] = -0.065;
            y[9] = 0.;
            y[10] = 0.4;
            y[11] = 0.230000004172325;
            y[12] = 0.04;
            y[13] = 0.;
            y[14] = 0.686;
            y[15] = 0.200000002980232;
            y[16] = 0.0640272398484633;
            y[17] = 0.259027384507773;
            y[18] = 0.379626;
            y[19] = 0.100000001490116;
            y[20] = 0.0640272398484633;
            y[21] = 0.259027384507773;
            y[22] = 0.229626;
            y[23] = 0.100000001490116;
            y[26] = 0.399976;
            y[27] = 0.100000001490116;
            y[31] = 0.0799999982118607;
            y[35] = 0.0799999982118607;
            y[39] = 0.0799999982118607;
            y[43] = 0.100000001490116;
            y[47] = 0.0799999982118607;
            y[51] = 0.0799999982118607;
            y[55] = 0.0799999982118607;
            y[59] = 0.0700000002980232;
            y[63] = 0.0700000002980232;
            y[67] = 0.0700000002980232;
            y[71] = 0.0799999982118607;
            y[75] = 0.0500000007450581;
            y[79] = 0.0399999991059303;
            y[83] = 0.0399999991059303;
            y[87] = 0.0149999996647239;
            y[91] = 0.0149999996647239;
            y[95] = 0.0120000001043081;
            y[99] = 0.0120000001043081;
            y[103] = 0.0120000001043081;
            y[107] = 0.0140000004321337;
            y[111] = 0.0140000004321337;
            y[115] = 0.0140000004321337;
            y[119] = 0.0140000004321337;
            y[123] = 0.0149999996647239;
            y[127] = 0.0149999996647239;
            y[131] = 0.0120000001043081;
            y[135] = 0.0120000001043081;
            y[139] = 0.0120000001043081;
            y[143] = 0.0140000004321337;
            y[147] = 0.0140000004321337;
            y[151] = 0.0140000004321337;
            y[155] = 0.0140000004321337;
            y[156] = 0.;
            y[157] = 0.;
            y[158] = -0.6;
            y[159] = 0.5;
            y[160] = 0.0640272398484633;
            y[161] = -0.259027384507773;
            y[162] = 0.379626;
            y[163] = 0.100000001490116;
            y[164] = 0.0640272398484633;
            y[165] = -0.259027384507773;
            y[166] = 0.229626;
            y[167] = 0.100000001490116;
            y[170] = 0.399976;
            y[171] = 0.100000001490116;
            y[175] = 0.0799999982118607;
            y[179] = 0.0799999982118607;
            y[183] = 0.0799999982118607;
            y[187] = 0.100000001490116;
            y[191] = 0.0799999982118607;
            y[195] = 0.0799999982118607;
            y[199] = 0.0799999982118607;
            y[203] = 0.0700000002980232;
            y[207] = 0.0700000002980232;
            y[211] = 0.0700000002980232;
            y[215] = 0.0799999982118607;
            y[219] = 0.0500000007450581;
            y[223] = 0.0399999991059303;
            y[227] = 0.0399999991059303;
            y[231] = 0.0149999996647239;
            y[235] = 0.0149999996647239;
            y[239] = 0.0120000001043081;
            y[243] = 0.0120000001043081;
            y[247] = 0.0120000001043081;
            y[251] = 0.0140000004321337;
            y[255] = 0.0140000004321337;
            y[259] = 0.0140000004321337;
            y[263] = 0.0140000004321337;
            y[267] = 0.0149999996647239;
            y[271] = 0.0149999996647239;
            y[275] = 0.0120000001043081;
            y[279] = 0.0120000001043081;
            y[283] = 0.0120000001043081;
            y[287] = 0.0140000004321337;
            y[291] = 0.0140000004321337;
            y[295] = 0.0140000004321337;
            y[299] = 0.0140000004321337;
            y[300] = -0.0413404628634453;
            y[301] = 0.;
            y[302] = 0.222553476691246;
            y[303] = 0.409016877412796;
            y[304] = 0.0399999991059303;
            y[305] = 0.;
            y[306] = 0.685999989509583;
            y[307] = 0.200000002980232;
            y[308] = 0.0640272398484633;
            y[309] = 0.259027384507773;
            y[310] = 0.304625997019768;
            y[311] = 0.174999997019768;
            y[314] = 0.399976;
            y[315] = 0.100000001490116;
            y[319] = 0.190453603863716;
            y[323] = 0.100000001490116;
            y[327] = 0.189999997615814;
            y[331] = 0.100000001490116;
            y[335] = 0.104999996721745;
            y[339] = 0.0500000007450581;
            y[343] = 0.0599999986588955;
            y[347] = 0.0269999988377094;
            y[351] = 0.0320000015199184;
            y[355] = 0.0281421355903149;
            y[359] = 0.025000000372529;
            y[363] = 0.0320000015199184;
            y[367] = 0.0281421355903149;
            y[368] = 0.;
            y[369] = 0.;
            y[370] = -0.600000023841858;
            y[371] = 0.5;
            y[372] = 0.0640272398484633;
            y[373] = -0.259027384507773;
            y[374] = 0.304625997019768;
            y[375] = 0.174999997019768;
            y[378] = 0.399976;
            y[379] = 0.100000001490116;
            y[383] = 0.190453603863716;
            y[387] = 0.100000001490116;
            y[391] = 0.189999997615814;
            y[395] = 0.100000001490116;
            y[399] = 0.104999996721745;
            y[403] = 0.0500000007450581;
            y[407] = 0.0599999986588955;
            y[411] = 0.0269999988377094;
            y[415] = 0.0320000015199184;
            y[419] = 0.0281421355903149;
            y[423] = 0.025000000372529;
            y[427] = 0.0320000015199184;
            y[431] = 0.0281421355903149;

            //
            // environment vs. robot collisions
            //

            // r_gripper_r_finger_tip
            if (sphere_environment_in_collision(environment, y[428], y[429], y[430], y[431]))
            {
                if (sphere_environment_in_collision(environment, y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // r_gripper_r_finger_2
            if (sphere_environment_in_collision(environment, y[424], y[425], y[426], y[427]))
            {
                if (sphere_environment_in_collision(environment, y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // r_gripper_r_finger
            if (sphere_environment_in_collision(environment, y[420], y[421], y[422], y[423]))
            {
                if (sphere_environment_in_collision(environment, y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // r_gripper_l_finger_tip
            if (sphere_environment_in_collision(environment, y[416], y[417], y[418], y[419]))
            {
                if (sphere_environment_in_collision(environment, y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // r_gripper_l_finger_2
            if (sphere_environment_in_collision(environment, y[412], y[413], y[414], y[415]))
            {
                if (sphere_environment_in_collision(environment, y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // r_gripper_l_finger
            if (sphere_environment_in_collision(environment, y[408], y[409], y[410], y[411]))
            {
                if (sphere_environment_in_collision(environment, y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // right_gripper_base
            if (sphere_environment_in_collision(environment, y[404], y[405], y[406], y[407]))
            {
                if (sphere_environment_in_collision(environment, y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // right_hand
            if (sphere_environment_in_collision(environment, y[400], y[401], y[402], y[403]))
            {
                if (sphere_environment_in_collision(environment, y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // right_wrist
            if (sphere_environment_in_collision(environment, y[396], y[397], y[398], y[399]))
            {
                if (sphere_environment_in_collision(environment, y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // right_lower_forearm
            if (sphere_environment_in_collision(environment, y[392], y[393], y[394], y[395]))
            {
                if (sphere_environment_in_collision(environment, y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // right_upper_forearm
            if (sphere_environment_in_collision(environment, y[388], y[389], y[390], y[391]))
            {
                if (sphere_environment_in_collision(environment, y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // right_lower_elbow
            if (sphere_environment_in_collision(environment, y[384], y[385], y[386], y[387]))
            {
                if (sphere_environment_in_collision(environment, y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // right_upper_elbow
            if (sphere_environment_in_collision(environment, y[380], y[381], y[382], y[383]))
            {
                if (sphere_environment_in_collision(environment, y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // right_lower_shoulder
            if (sphere_environment_in_collision(environment, y[376], y[377], y[378], y[379]))
            {
                if (sphere_environment_in_collision(environment, y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // right_upper_shoulder
            if (sphere_environment_in_collision(environment, y[372], y[373], y[374], y[375]))
            {
                if (sphere_environment_in_collision(environment, y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // pedestal
            if (sphere_environment_in_collision(environment, y[368], y[369], y[370], y[371]))
            {
                if (sphere_environment_in_collision(environment, y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // l_gripper_r_finger_tip
            if (sphere_environment_in_collision(environment, y[364], y[365], y[366], y[367]))
            {
                if (sphere_environment_in_collision(environment, y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[152], y[153], y[154], y[155]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // l_gripper_r_finger_2
            if (sphere_environment_in_collision(environment, y[360], y[361], y[362], y[363]))
            {
                if (sphere_environment_in_collision(environment, y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[136], y[137], y[138], y[139]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // l_gripper_r_finger
            if (sphere_environment_in_collision(environment, y[356], y[357], y[358], y[359]))
            {
                if (sphere_environment_in_collision(environment, y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // l_gripper_l_finger_tip
            if (sphere_environment_in_collision(environment, y[352], y[353], y[354], y[355]))
            {
                if (sphere_environment_in_collision(environment, y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // l_gripper_l_finger_2
            if (sphere_environment_in_collision(environment, y[348], y[349], y[350], y[351]))
            {
                if (sphere_environment_in_collision(environment, y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // l_gripper_l_finger
            if (sphere_environment_in_collision(environment, y[344], y[345], y[346], y[347]))
            {
                if (sphere_environment_in_collision(environment, y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // left_gripper_base
            if (sphere_environment_in_collision(environment, y[340], y[341], y[342], y[343]))
            {
                if (sphere_environment_in_collision(environment, y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[80], y[81], y[82], y[83]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // left_hand
            if (sphere_environment_in_collision(environment, y[336], y[337], y[338], y[339]))
            {
                if (sphere_environment_in_collision(environment, y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // left_wrist
            if (sphere_environment_in_collision(environment, y[332], y[333], y[334], y[335]))
            {
                if (sphere_environment_in_collision(environment, y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[68], y[69], y[70], y[71]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // left_lower_forearm
            if (sphere_environment_in_collision(environment, y[328], y[329], y[330], y[331]))
            {
                if (sphere_environment_in_collision(environment, y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // left_upper_forearm
            if (sphere_environment_in_collision(environment, y[324], y[325], y[326], y[327]))
            {
                if (sphere_environment_in_collision(environment, y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[52], y[53], y[54], y[55]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // left_lower_elbow
            if (sphere_environment_in_collision(environment, y[320], y[321], y[322], y[323]))
            {
                if (sphere_environment_in_collision(environment, y[40], y[41], y[42], y[43]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // left_upper_elbow
            if (sphere_environment_in_collision(environment, y[316], y[317], y[318], y[319]))
            {
                if (sphere_environment_in_collision(environment, y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[32], y[33], y[34], y[35]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[36], y[37], y[38], y[39]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // left_lower_shoulder
            if (sphere_environment_in_collision(environment, y[312], y[313], y[314], y[315]))
            {
                if (sphere_environment_in_collision(environment, y[24], y[25], y[26], y[27]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // left_upper_shoulder
            if (sphere_environment_in_collision(environment, y[308], y[309], y[310], y[311]))
            {
                if (sphere_environment_in_collision(environment, y[16], y[17], y[18], y[19]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[20], y[21], y[22], y[23]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // head
            if (sphere_environment_in_collision(environment, y[304], y[305], y[306], y[307]))
            {
                if (sphere_environment_in_collision(environment, y[12], y[13], y[14], y[15]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // torso
            if (sphere_environment_in_collision(environment, y[300], y[301], y[302], y[303]))
            {
                if (sphere_environment_in_collision(environment, y[0], y[1], y[2], y[3]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[4], y[5], y[6], y[7]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[8], y[9], y[10], y[11]))
                {
                    return false;
                }
            }

            //
            // robot self-collisions
            //

            // torso vs. left_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[320], y[321], y[322], y[323]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[40], y[41], y[42], y[43]))
                {
                    return false;
                }
            }

            // torso vs. left_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[324], y[325], y[326], y[327]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[52], y[53], y[54], y[55]))
                {
                    return false;
                }
            }

            // torso vs. left_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[328], y[329], y[330], y[331]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[60], y[61], y[62], y[63]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[60], y[61], y[62], y[63]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            // torso vs. left_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[332], y[333], y[334], y[335]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }
            }

            // torso vs. left_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[336], y[337], y[338], y[339]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // torso vs. left_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[340], y[341], y[342], y[343]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }
            }

            // torso vs. l_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[344], y[345], y[346], y[347]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // torso vs. l_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[348], y[349], y[350], y[351]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // torso vs. l_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[352], y[353], y[354], y[355]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // torso vs. l_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[356], y[357], y[358], y[359]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            // torso vs. l_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[360], y[361], y[362], y[363]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }
            }

            // torso vs. l_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[364], y[365], y[366], y[367]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }
            }

            // torso vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // torso vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // torso vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // torso vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // torso vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // torso vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // torso vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // torso vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // torso vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // torso vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // torso vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // torso vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // head vs. left_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[320], y[321], y[322], y[323]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[40], y[41], y[42], y[43]))
                {
                    return false;
                }
            }

            // head vs. left_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[324], y[325], y[326], y[327]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[52], y[53], y[54], y[55]))
                {
                    return false;
                }
            }

            // head vs. left_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[328], y[329], y[330], y[331]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            // head vs. left_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[332], y[333], y[334], y[335]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }
            }

            // head vs. left_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[336], y[337], y[338], y[339]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // head vs. left_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[340], y[341], y[342], y[343]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }
            }

            // head vs. l_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[352], y[353], y[354], y[355]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // head vs. l_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[356], y[357], y[358], y[359]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            // head vs. l_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[360], y[361], y[362], y[363]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }
            }

            // head vs. l_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[364], y[365], y[366], y[367]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }
            }

            // head vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // head vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // head vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // head vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // head vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // head vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // head vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // head vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // head vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // head vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. left_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[324], y[325], y[326], y[327]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[52], y[53], y[54], y[55]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. left_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[328], y[329], y[330], y[331]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[60], y[61], y[62], y[63]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[60], y[61], y[62], y[63]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. left_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[332], y[333], y[334], y[335]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. left_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[336], y[337], y[338], y[339]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. left_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[340], y[341], y[342], y[343]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. l_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[344], y[345], y[346], y[347]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. l_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[348], y[349], y[350], y[351]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. l_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[352], y[353], y[354], y[355]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. l_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[356], y[357], y[358], y[359]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. l_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[360], y[361], y[362], y[363]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. l_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[364], y[365], y[366], y[367]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // left_upper_shoulder vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. left_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[332], y[333], y[334], y[335]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. left_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[336], y[337], y[338], y[339]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. left_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[340], y[341], y[342], y[343]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. l_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[344], y[345], y[346], y[347]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. l_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[348], y[349], y[350], y[351]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. l_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[352], y[353], y[354], y[355]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. l_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[356], y[357], y[358], y[359]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. l_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[360], y[361], y[362], y[363]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. l_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[364], y[365], y[366], y[367]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // left_lower_shoulder vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. left_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[332], y[333], y[334], y[335]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. left_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[336], y[337], y[338], y[339]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. left_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[340], y[341], y[342], y[343]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. l_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[344], y[345], y[346], y[347]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. l_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[348], y[349], y[350], y[351]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. l_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[352], y[353], y[354], y[355]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. l_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[356], y[357], y[358], y[359]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. l_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[360], y[361], y[362], y[363]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. l_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[364], y[365], y[366], y[367]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // left_upper_elbow vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. l_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[352], y[353], y[354], y[355]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. l_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[364], y[365], y[366], y[367]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // left_lower_elbow vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. l_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[352], y[353], y[354], y[355]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. l_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[364], y[365], y[366], y[367]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // left_upper_forearm vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. l_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[352], y[353], y[354], y[355]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. l_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[364], y[365], y[366], y[367]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. right_upper_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[372], y[373], y[374], y[375]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // left_lower_forearm vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // left_wrist vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // left_wrist vs. right_upper_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[372], y[373], y[374], y[375]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            // left_wrist vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // left_wrist vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // left_wrist vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // left_wrist vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // left_wrist vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // left_wrist vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // left_wrist vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // left_wrist vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // left_wrist vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // left_wrist vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // left_wrist vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // left_wrist vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // left_wrist vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // left_wrist vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // left_hand vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // left_hand vs. right_upper_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[372], y[373], y[374], y[375]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            // left_hand vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // left_hand vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // left_hand vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // left_hand vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // left_hand vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // left_hand vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // left_hand vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // left_hand vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // left_hand vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // left_hand vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // left_hand vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // left_hand vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // left_hand vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // left_hand vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. right_upper_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[372], y[373], y[374], y[375]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // left_gripper_base vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. right_upper_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[372], y[373], y[374], y[375]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. right_upper_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[372], y[373], y[374], y[375]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_2 vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. right_upper_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[372], y[373], y[374], y[375]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // l_gripper_l_finger_tip vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. right_upper_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[372], y[373], y[374], y[375]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. right_upper_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[372], y[373], y[374], y[375]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_2 vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. pedestal
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[368], y[369], y[370], y[371]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. right_upper_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[372], y[373], y[374], y[375]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. right_lower_shoulder
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[376], y[377], y[378], y[379]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. right_upper_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[380], y[381], y[382], y[383]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. right_lower_elbow
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[384], y[385], y[386], y[387]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // l_gripper_r_finger_tip vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // pedestal vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // pedestal vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // pedestal vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            // pedestal vs. right_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[400], y[401], y[402], y[403]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // pedestal vs. right_gripper_base
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[404], y[405], y[406], y[407]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // pedestal vs. r_gripper_l_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[408], y[409], y[410], y[411]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // pedestal vs. r_gripper_l_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[412], y[413], y[414], y[415]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[244], y[245], y[246], y[247]))
                {
                    return false;
                }
            }

            // pedestal vs. r_gripper_l_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[416], y[417], y[418], y[419]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[256], y[257], y[258], y[259]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }
            }

            // pedestal vs. r_gripper_r_finger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[420], y[421], y[422], y[423]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }
            }

            // pedestal vs. r_gripper_r_finger_2
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[424], y[425], y[426], y[427]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }
            }

            // pedestal vs. r_gripper_r_finger_tip
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[428], y[429], y[430], y[431]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }
            }

            // right_upper_shoulder vs. right_upper_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[372], y[373], y[374], y[375], y[388], y[389], y[390], y[391]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }
            }

            // right_upper_shoulder vs. right_lower_forearm
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[372], y[373], y[374], y[375], y[392], y[393], y[394], y[395]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }
            }

            // right_upper_shoulder vs. right_wrist
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[372], y[373], y[374], y[375], y[396], y[397], y[398], y[399]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[212], y[213], y[214], y[215]))
            