#pragma once

#include <vamp/vector.hh>
#include <vamp/vector/math.hh>
#include <vamp/collision/environment.hh>
#include <vamp/collision/validity.hh>

// NOLINTBEGIN(*-magic-numbers)
namespace vamp::robots
{
    struct Fetch
    {
        static constexpr char *name = "fetch";
        static constexpr std::size_t dimension = 8;
        static constexpr std::size_t n_spheres = 111;
        static constexpr float min_radius = 0.012000000104308128;
        static constexpr float max_radius = 0.23999999463558197;
        static constexpr std::size_t resolution = 32;

        static constexpr std::array<std::string_view, dimension> joint_names = {
            "torso_lift_joint",
            "shoulder_pan_joint",
            "shoulder_lift_joint",
            "upperarm_roll_joint",
            "elbow_flex_joint",
            "forearm_roll_joint",
            "wrist_flex_joint",
            "wrist_roll_joint"};
        static constexpr char *end_effector = "gripper_link";

        using Configuration = FloatVector<dimension>;
        using ConfigurationArray = std::array<FloatT, dimension>;

        struct alignas(FloatVectorAlignment) ConfigurationBuffer
          : std::array<float, Configuration::num_scalars_rounded>
        {
        };

        template <std::size_t rake>
        using ConfigurationBlock = FloatVector<rake, dimension>;

        template <std::size_t rake>
        struct Spheres
        {
            FloatVector<rake, n_spheres> x;
            FloatVector<rake, n_spheres> y;
            FloatVector<rake, n_spheres> z;
            FloatVector<rake, n_spheres> r;
        };

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> s_m{
            0.3861500024795532,
            3.211199998855591,
            2.739000082015991,
            6.283180236816406,
            4.501999855041504,
            6.283180236816406,
            4.320000171661377,
            6.283180236816406};

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> s_a{
            0.0,
            -1.6055999994277954,
            -1.2209999561309814,
            -3.141590118408203,
            -2.250999927520752,
            -3.141590118408203,
            -2.1600000858306885,
            -3.141590118408203};

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> d_m{
            2.589667320251465,
            0.3114100694656372,
            0.3650967478752136,
            0.15915507078170776,
            0.22212350368499756,
            0.15915507078170776,
            0.23148147761821747,
            0.15915507078170776};

        inline static void scale_configuration(Configuration &q) noexcept
        {
            q = q * Configuration(s_m) + Configuration(s_a);
        }

        inline static void descale_configuration(Configuration &q) noexcept
        {
            q = (q - Configuration(s_a)) * Configuration(d_m);
        }

        template <std::size_t rake>
        inline static void scale_configuration_block(ConfigurationBlock<rake> &q) noexcept
        {
            q[0] = 0.0 + (q[0] * 0.3861500024795532);
            q[1] = -1.6055999994277954 + (q[1] * 3.211199998855591);
            q[2] = -1.2209999561309814 + (q[2] * 2.739000082015991);
            q[3] = -3.141590118408203 + (q[3] * 6.283180236816406);
            q[4] = -2.250999927520752 + (q[4] * 4.501999855041504);
            q[5] = -3.141590118408203 + (q[5] * 6.283180236816406);
            q[6] = -2.1600000858306885 + (q[6] * 4.320000171661377);
            q[7] = -3.141590118408203 + (q[7] * 6.283180236816406);
        }

        template <std::size_t rake>
        inline static void descale_configuration_block(ConfigurationBlock<rake> &q) noexcept
        {
            q[0] = 2.589667320251465 * (q[0] - 0.0);
            q[1] = 0.3114100694656372 * (q[1] - -1.6055999994277954);
            q[2] = 0.3650967478752136 * (q[2] - -1.2209999561309814);
            q[3] = 0.15915507078170776 * (q[3] - -3.141590118408203);
            q[4] = 0.22212350368499756 * (q[4] - -2.250999927520752);
            q[5] = 0.15915507078170776 * (q[5] - -3.141590118408203);
            q[6] = 0.23148147761821747 * (q[6] - -2.1600000858306885);
            q[7] = 0.15915507078170776 * (q[7] - -3.141590118408203);
        }

        inline static auto space_measure() noexcept -> float
        {
            return 16384.87636281249;
        }

        template <std::size_t rake>
        inline static void sphere_fk(const ConfigurationBlock<rake> &x, Spheres<rake> &out) noexcept
        {
            std::array<FloatVector<rake, 1>, 17> v;
            std::array<FloatVector<rake, 1>, 444> y;

            v[0] = 0.37743 + x[0];
            y[82] = 0.15 + v[0];
            y[86] = 0.15 + v[0];
            y[90] = 0.3 + v[0];
            y[94] = 0.45 + v[0];
            y[98] = 0.45 + v[0];
            y[102] = 0.3 + v[0];
            y[106] = 0.663001417713939 + v[0];
            y[110] = 0.661001417713939 + v[0];
            y[114] = 0.661001417713939 + v[0];
            y[118] = 0.661001417713939 + v[0];
            y[122] = 0.661001417713939 + v[0];
            y[126] = 0.661001417713939 + v[0];
            y[130] = 0.633001417713939 + v[0];
            y[134] = 0.633001417713939 + v[0];
            y[138] = 0.633001417713939 + v[0];
            y[142] = 0.633001417713939 + v[0];
            y[146] = 0.688001417713939 + v[0];
            y[150] = 0.688001417713939 + v[0];
            y[154] = 0.688001417713939 + v[0];
            y[158] = 0.688001417713939 + v[0];
            y[162] = 0.678001417713939 + v[0];
            y[166] = 0.660501417713939 + v[0];
            y[170] = 0.643001417713939 + v[0];
            y[174] = 0.633001417713939 + v[0];
            y[178] = 0.633001417713939 + v[0];
            y[182] = 0.633001417713939 + v[0];
            y[186] = 0.633001417713939 + v[0];
            y[190] = 0.688001417713939 + v[0];
            y[194] = 0.688001417713939 + v[0];
            y[198] = 0.688001417713939 + v[0];
            y[202] = 0.688001417713939 + v[0];
            y[206] = 0.678001417713939 + v[0];
            y[210] = 0.660501417713939 + v[0];
            y[214] = 0.643001417713939 + v[0];
            y[218] = 0.34858 + v[0];
            v[1] = cos(x[1]);
            v[2] = sin(x[1]);
            v[3] = -v[2];
            y[220] = 0.03265 + 0.025 * v[1] + -0.015 * v[3];
            y[221] = 0.025 * v[2] + -0.015 * v[1];
            y[222] = 0.035 + y[218];
            y[224] = 0.03265 + 0.05 * v[1] + -0.03 * v[3];
            y[225] = 0.05 * v[2] + -0.03 * v[1];
            y[226] = 0.06 + y[218];
            y[228] = 0.03265 + 0.12 * v[1] + -0.03 * v[3];
            y[229] = 0.12 * v[2] + -0.03 * v[1];
            y[230] = 0.06 + y[218];
            v[4] = cos(x[2]);
            v[5] = v[1] * v[4];
            v[6] = sin(x[2]);
            v[7] = v[1] * v[6];
            v[8] = 0.03265 + 0.117 * v[1];
            y[232] = 0.025 * v[5] + 0.04 * v[3] + 0.025 * v[7] + v[8];
            v[9] = v[2] * v[4];
            v[10] = v[2] * v[6];
            v[2] = 0.117 * v[2];
            y[233] = 0.025 * v[9] + 0.04 * v[1] + 0.025 * v[10] + v[2];
            v[6] = -v[6];
            v[11] = 0.0599999999999999 + y[218];
            y[234] = 0.025 * v[6] + 0.025 * v[4] + v[11];
            y[236] = -0.025 * v[5] + 0.04 * v[3] + -0.025 * v[7] + v[8];
            y[237] = -0.025 * v[9] + 0.04 * v[1] + -0.025 * v[10] + v[2];
            y[238] = -0.025 * v[6] + -0.025 * v[4] + v[11];
            y[240] = 0.025 * v[5] + 0.04 * v[3] + -0.025 * v[7] + v[8];
            y[241] = 0.025 * v[9] + 0.04 * v[1] + -0.025 * v[10] + v[2];
            y[242] = 0.025 * v[6] + -0.025 * v[4] + v[11];
            y[244] = -0.025 * v[5] + 0.04 * v[3] + 0.025 * v[7] + v[8];
            y[245] = -0.025 * v[9] + 0.04 * v[1] + 0.025 * v[10] + v[2];
            y[246] = -0.025 * v[6] + 0.025 * v[4] + v[11];
            y[248] = 0.08 * v[5] + v[8];
            y[249] = 0.08 * v[9] + v[2];
            y[250] = 0.08 * v[6] + v[11];
            y[252] = 0.11 * v[5] + v[8];
            y[253] = 0.11 * v[9] + v[2];
            y[254] = 0.11 * v[6] + v[11];
            y[256] = 0.14 * v[5] + v[8];
            y[257] = 0.14 * v[9] + v[2];
            y[258] = 0.14 * v[6] + v[11];
            v[8] = v[8] + 0.219 * v[5];
            y[260] = -0.02 * v[5] + v[8];
            v[2] = v[2] + 0.219 * v[9];
            y[261] = -0.02 * v[9] + v[2];
            v[11] = v[11] + 0.219 * v[6];
            y[262] = -0.02 * v[6] + v[11];
            y[264] = 0.03 * v[5] + v[8];
            y[265] = 0.03 * v[9] + v[2];
            y[266] = 0.03 * v[6] + v[11];
            y[268] = 0.08 * v[5] + v[8];
            y[269] = 0.08 * v[9] + v[2];
            y[270] = 0.08 * v[6] + v[11];
            v[12] = cos(x[3]);
            v[13] = sin(x[3]);
            v[14] = v[3] * v[12] + v[7] * v[13];
            v[15] = -v[13];
            v[7] = v[3] * v[15] + v[7] * v[12];
            y[272] = 0.11 * v[5] + -0.045 * v[14] + 0.02 * v[7] + v[8];
            v[3] = v[1] * v[12] + v[10] * v[13];
            v[15] = v[1] * v[15] + v[10] * v[12];
            y[273] = 0.11 * v[9] + -0.045 * v[3] + 0.02 * v[15] + v[2];
            v[13] = v[4] * v[13];
            v[12] = v[4] * v[12];
            y[274] = 0.11 * v[6] + -0.045 * v[13] + 0.02 * v[12] + v[11];
            y[276] = 0.11 * v[5] + -0.045 * v[14] + -0.02 * v[7] + v[8];
            y[277] = 0.11 * v[9] + -0.045 * v[3] + -0.02 * v[15] + v[2];
            y[278] = 0.11 * v[6] + -0.045 * v[13] + -0.02 * v[12] + v[11];
            y[280] = 0.155 * v[5] + -0.045 * v[14] + 0.02 * v[7] + v[8];
            y[281] = 0.155 * v[9] + -0.045 * v[3] + 0.02 * v[15] + v[2];
            y[282] = 0.155 * v[6] + -0.045 * v[13] + 0.02 * v[12] + v[11];
            y[284] = 0.155 * v[5] + -0.045 * v[14] + -0.02 * v[7] + v[8];
            y[285] = 0.155 * v[9] + -0.045 * v[3] + -0.02 * v[15] + v[2];
            y[286] = 0.155 * v[6] + -0.045 * v[13] + -0.02 * v[12] + v[11];
            y[288] = 0.13 * v[5] + v[8];
            y[289] = 0.13 * v[9] + v[2];
            y[290] = 0.13 * v[6] + v[11];
            v[4] = cos(x[4]);
            v[10] = sin(x[4]);
            v[1] = -v[10];
            v[16] = v[5] * v[4] + v[7] * v[1];
            v[7] = v[5] * v[10] + v[7] * v[4];
            v[8] = v[8] + 0.133 * v[5];
            y[292] = 0.02 * v[16] + 0.045 * v[14] + 0.02 * v[7] + v[8];
            v[5] = v[9] * v[4] + v[15] * v[1];
            v[15] = v[9] * v[10] + v[15] * v[4];
            v[2] = v[2] + 0.133 * v[9];
            y[293] = 0.02 * v[5] + 0.045 * v[3] + 0.02 * v[15] + v[2];
            v[1] = v[6] * v[4] + v[12] * v[1];
            v[10] = v[6] * v[10] + v[12] * v[4];
            v[11] = v[11] + 0.133 * v[6];
            y[294] = 0.02 * v[1] + 0.045 * v[13] + 0.02 * v[10] + v[11];
            y[296] = 0.02 * v[16] + 0.045 * v[14] + -0.02 * v[7] + v[8];
            y[297] = 0.02 * v[5] + 0.045 * v[3] + -0.02 * v[15] + v[2];
            y[298] = 0.02 * v[1] + 0.045 * v[13] + -0.02 * v[10] + v[11];
            y[300] = -0.02 * v[16] + 0.045 * v[14] + 0.02 * v[7] + v[8];
            y[301] = -0.02 * v[5] + 0.045 * v[3] + 0.02 * v[15] + v[2];
            y[302] = -0.02 * v[1] + 0.045 * v[13] + 0.02 * v[10] + v[11];
            y[304] = -0.02 * v[16] + 0.045 * v[14] + -0.02 * v[7] + v[8];
            y[305] = -0.02 * v[5] + 0.045 * v[3] + -0.02 * v[15] + v[2];
            y[306] = -0.02 * v[1] + 0.045 * v[13] + -0.02 * v[10] + v[11];
            y[308] = 0.08 * v[16] + v[8];
            y[309] = 0.08 * v[5] + v[2];
            y[310] = 0.08 * v[1] + v[11];
            y[312] = 0.14 * v[16] + v[8];
            y[313] = 0.14 * v[5] + v[2];
            y[314] = 0.14 * v[1] + v[11];
            y[316] = v[8] + 0.197 * v[16];
            y[317] = v[2] + 0.197 * v[5];
            y[318] = v[11] + 0.197 * v[1];
            v[11] = cos(x[5]);
            v[2] = sin(x[5]);
            v[8] = v[14] * v[11] + v[7] * v[2];
            v[6] = -v[2];
            v[7] = v[14] * v[6] + v[7] * v[11];
            y[320] = 0.05 * v[16] + -0.06 * v[8] + 0.02 * v[7] + y[316];
            v[14] = v[3] * v[11] + v[15] * v[2];
            v[15] = v[3] * v[6] + v[15] * v[11];
            y[321] = 0.05 * v[5] + -0.06 * v[14] + 0.02 * v[15] + y[317];
            v[2] = v[13] * v[11] + v[10] * v[2];
            v[6] = v[13] * v[6] + v[10] * v[11];
            y[322] = 0.05 * v[1] + -0.06 * v[2] + 0.02 * v[6] + y[318];
            y[324] = 0.05 * v[16] + -0.06 * v[8] + -0.02 * v[7] + y[316];
            y[325] = 0.05 * v[5] + -0.06 * v[14] + -0.02 * v[15] + y[317];
            y[326] = 0.05 * v[1] + -0.06 * v[2] + -0.02 * v[6] + y[318];
            y[328] = 0.1 * v[16] + -0.06 * v[8] + 0.02 * v[7] + y[316];
            y[329] = 0.1 * v[5] + -0.06 * v[14] + 0.02 * v[15] + y[317];
            y[330] = 0.1 * v[1] + -0.06 * v[2] + 0.02 * v[6] + y[318];
            y[332] = 0.1 * v[16] + -0.06 * v[8] + -0.02 * v[7] + y[316];
            y[333] = 0.1 * v[5] + -0.06 * v[14] + -0.02 * v[15] + y[317];
            y[334] = 0.1 * v[1] + -0.06 * v[2] + -0.02 * v[6] + y[318];
            y[336] = 0.15 * v[16] + -0.06 * v[8] + 0.02 * v[7] + y[316];
            y[337] = 0.15 * v[5] + -0.06 * v[14] + 0.02 * v[15] + y[317];
            y[338] = 0.15 * v[1] + -0.06 * v[2] + 0.02 * v[6] + y[318];
            y[340] = 0.15 * v[16] + -0.06 * v[8] + -0.02 * v[7] + y[316];
            y[341] = 0.15 * v[5] + -0.06 * v[14] + -0.02 * v[15] + y[317];
            y[342] = 0.15 * v[1] + -0.06 * v[2] + -0.02 * v[6] + y[318];
            y[344] = y[316] + 0.1245 * v[16];
            y[345] = y[317] + 0.1245 * v[5];
            y[346] = y[318] + 0.1245 * v[1];
            v[11] = cos(x[6]);
            v[10] = sin(x[6]);
            v[13] = -v[10];
            v[3] = v[16] * v[11] + v[7] * v[13];
            y[348] = 0.06 * v[3] + y[344];
            v[4] = v[5] * v[11] + v[15] * v[13];
            y[349] = 0.06 * v[4] + y[345];
            v[13] = v[1] * v[11] + v[6] * v[13];
            y[350] = 0.06 * v[13] + y[346];
            v[7] = v[16] * v[10] + v[7] * v[11];
            y[352] = 0.02 * v[3] + 0.045 * v[8] + 0.02 * v[7] + y[344];
            v[15] = v[5] * v[10] + v[15] * v[11];
            y[353] = 0.02 * v[4] + 0.045 * v[14] + 0.02 * v[15] + y[345];
            v[10] = v[1] * v[10] + v[6] * v[11];
            y[354] = 0.02 * v[13] + 0.045 * v[2] + 0.02 * v[10] + y[346];
            y[356] = 0.02 * v[3] + 0.045 * v[8] + -0.02 * v[7] + y[344];
            y[357] = 0.02 * v[4] + 0.045 * v[14] + -0.02 * v[15] + y[345];
            y[358] = 0.02 * v[13] + 0.045 * v[2] + -0.02 * v[10] + y[346];
            y[360] = -0.02 * v[3] + 0.045 * v[8] + 0.02 * v[7] + y[344];
            y[361] = -0.02 * v[4] + 0.045 * v[14] + 0.02 * v[15] + y[345];
            y[362] = -0.02 * v[13] + 0.045 * v[2] + 0.02 * v[10] + y[346];
            y[364] = -0.02 * v[3] + 0.045 * v[8] + -0.02 * v[7] + y[344];
            y[365] = -0.02 * v[4] + 0.045 * v[14] + -0.02 * v[15] + y[345];
            y[366] = -0.02 * v[13] + 0.045 * v[2] + -0.02 * v[10] + y[346];
            y[372] = y[344] + 0.1385 * v[3];
            y[368] = -0.03 * v[3] + y[372];
            y[373] = y[345] + 0.1385 * v[4];
            y[369] = -0.03 * v[4] + y[373];
            y[374] = y[346] + 0.1385 * v[13];
            y[370] = -0.03 * v[13] + y[374];
            v[11] = cos(x[7]);
            v[6] = sin(x[7]);
            v[1] = v[8] * v[11] + v[7] * v[6];
            y[376] = 0.09645 * v[3] + 0.02 * v[1] + y[372];
            v[5] = v[14] * v[11] + v[15] * v[6];
            y[377] = 0.09645 * v[4] + 0.02 * v[5] + y[373];
            v[16] = v[2] * v[11] + v[10] * v[6];
            y[378] = 0.09645 * v[13] + 0.02 * v[16] + y[374];
            y[380] = 0.09645 * v[3] + -0.02 * v[1] + y[372];
            y[381] = 0.09645 * v[4] + -0.02 * v[5] + y[373];
            y[382] = 0.09645 * v[13] + -0.02 * v[16] + y[374];
            y[384] = 0.06645 * v[3] + 0.02 * v[1] + y[372];
            y[385] = 0.06645 * v[4] + 0.02 * v[5] + y[373];
            y[386] = 0.06645 * v[13] + 0.02 * v[16] + y[374];
            y[388] = 0.06645 * v[3] + -0.02 * v[1] + y[372];
            y[389] = 0.06645 * v[4] + -0.02 * v[5] + y[373];
            y[390] = 0.06645 * v[13] + -0.02 * v[16] + y[374];
            v[6] = -v[6];
            v[7] = v[8] * v[6] + v[7] * v[11];
            y[392] = 0.18345 * v[3] + -0.056925 * v[1] + -0.005 * v[7] + y[372];
            v[15] = v[14] * v[6] + v[15] * v[11];
            y[393] = 0.18345 * v[4] + -0.056925 * v[5] + -0.005 * v[15] + y[373];
            v[6] = v[2] * v[6] + v[10] * v[11];
            y[394] = 0.18345 * v[13] + -0.056925 * v[16] + -0.005 * v[6] + y[374];
            y[396] = 0.18345 * v[3] + -0.056925 * v[1] + 0.005 * v[7] + y[372];
            y[397] = 0.18345 * v[4] + -0.056925 * v[5] + 0.005 * v[15] + y[373];
            y[398] = 0.18345 * v[13] + -0.056925 * v[16] + 0.005 * v[6] + y[374];
            y[400] = 0.16645 * v[3] + -0.056925 * v[1] + -0.005 * v[7] + y[372];
            y[401] = 0.16645 * v[4] + -0.056925 * v[5] + -0.005 * v[15] + y[373];
            y[402] = 0.16645 * v[13] + -0.056925 * v[16] + -0.005 * v[6] + y[374];
            y[404] = 0.16645 * v[3] + -0.056925 * v[1] + 0.005 * v[7] + y[372];
            y[405] = 0.16645 * v[4] + -0.056925 * v[5] + 0.005 * v[15] + y[373];
            y[406] = 0.16645 * v[13] + -0.056925 * v[16] + 0.005 * v[6] + y[374];
            y[408] = 0.14945 * v[3] + -0.056925 * v[1] + -0.005 * v[7] + y[372];
            y[409] = 0.14945 * v[4] + -0.056925 * v[5] + -0.005 * v[15] + y[373];
            y[410] = 0.14945 * v[13] + -0.056925 * v[16] + -0.005 * v[6] + y[374];
            y[412] = 0.14945 * v[3] + -0.056925 * v[1] + 0.005 * v[7] + y[372];
            y[413] = 0.14945 * v[4] + -0.056925 * v[5] + 0.005 * v[15] + y[373];
            y[414] = 0.14945 * v[13] + -0.056925 * v[16] + 0.005 * v[6] + y[374];
            y[416] = 0.18345 * v[3] + 0.056925 * v[1] + -0.005 * v[7] + y[372];
            y[417] = 0.18345 * v[4] + 0.056925 * v[5] + -0.005 * v[15] + y[373];
            y[418] = 0.18345 * v[13] + 0.056925 * v[16] + -0.005 * v[6] + y[374];
            y[420] = 0.18345 * v[3] + 0.056925 * v[1] + 0.005 * v[7] + y[372];
            y[421] = 0.18345 * v[4] + 0.056925 * v[5] + 0.005 * v[15] + y[373];
            y[422] = 0.18345 * v[13] + 0.056925 * v[16] + 0.005 * v[6] + y[374];
            y[424] = 0.16645 * v[3] + 0.056925 * v[1] + -0.005 * v[7] + y[372];
            y[425] = 0.16645 * v[4] + 0.056925 * v[5] + -0.005 * v[15] + y[373];
            y[426] = 0.16645 * v[13] + 0.056925 * v[16] + -0.005 * v[6] + y[374];
            y[428] = 0.16645 * v[3] + 0.056925 * v[1] + 0.005 * v[7] + y[372];
            y[429] = 0.16645 * v[4] + 0.056925 * v[5] + 0.005 * v[15] + y[373];
            y[430] = 0.16645 * v[13] + 0.056925 * v[16] + 0.005 * v[6] + y[374];
            y[432] = 0.14945 * v[3] + 0.056925 * v[1] + -0.005 * v[7] + y[372];
            y[433] = 0.14945 * v[4] + 0.056925 * v[5] + -0.005 * v[15] + y[373];
            y[434] = 0.14945 * v[13] + 0.056925 * v[16] + -0.005 * v[6] + y[374];
            y[436] = 0.14945 * v[3] + 0.056925 * v[1] + 0.005 * v[7] + y[372];
            y[437] = 0.14945 * v[4] + 0.056925 * v[5] + 0.005 * v[15] + y[373];
            y[438] = 0.14945 * v[13] + 0.056925 * v[16] + 0.005 * v[6] + y[374];
            y[442] = 0.24 + v[0];
            // dependent variables without operations
            y[0] = -0.12;
            y[1] = 0.;
            y[2] = 0.182;
            y[3] = 0.239999994635582;
            y[4] = 0.225;
            y[5] = 0.;
            y[6] = 0.31;
            y[7] = 0.0659999996423721;
            y[8] = 0.08;
            y[9] = -0.06;
            y[10] = 0.16;
            y[11] = 0.219999998807907;
            y[12] = 0.215;
            y[13] = -0.07;
            y[14] = 0.31;
            y[15] = 0.0659999996423721;
            y[16] = 0.185;
            y[17] = -0.135;
            y[18] = 0.31;
            y[19] = 0.0659999996423721;
            y[20] = 0.13;
            y[21] = -0.185;
            y[22] = 0.31;
            y[23] = 0.0659999996423721;
            y[24] = 0.065;
            y[25] = -0.2;
            y[26] = 0.31;
            y[27] = 0.0659999996423721;
            y[28] = 0.01;
            y[29] = -0.2;
            y[30] = 0.31;
            y[31] = 0.0659999996423721;
            y[32] = 0.08;
            y[33] = 0.06;
            y[34] = 0.16;
            y[35] = 0.219999998807907;
            y[36] = 0.215;
            y[37] = 0.07;
            y[38] = 0.31;
            y[39] = 0.0659999996423721;
            y[40] = 0.185;
            y[41] = 0.135;
            y[42] = 0.31;
            y[43] = 0.0659999996423721;
            y[44] = 0.13;
            y[45] = 0.185;
            y[46] = 0.31;
            y[47] = 0.0659999996423721;
            y[48] = 0.065;
            y[49] = 0.2;
            y[50] = 0.31;
            y[51] = 0.0659999996423721;
            y[52] = 0.01;
            y[53] = 0.2;
            y[54] = 0.31;
            y[55] = 0.0659999996423721;
            y[56] = -0.186875;
            y[57] = -0.07;
            y[58] = 0.727425;
            y[59] = 0.119999997317791;
            y[60] = -0.186875;
            y[61] = 0.07;
            y[62] = 0.727425;
            y[63] = 0.119999997317791;
            y[64] = -0.186875;
            y[65] = -0.07;
            y[66] = 0.577425;
            y[67] = 0.119999997317791;
            y[68] = -0.186875;
            y[69] = 0.07;
            y[70] = 0.577425;
            y[71] = 0.119999997317791;
            y[72] = -0.186875;
            y[73] = 0.07;
            y[74] = 0.447425;
            y[75] = 0.119999997317791;
            y[76] = -0.186875;
            y[77] = -0.07;
            y[78] = 0.447425;
            y[79] = 0.119999997317791;
            y[80] = -0.186875;
            y[81] = -0.05;
            y[83] = 0.150000005960464;
            y[84] = -0.186875;
            y[85] = 0.05;
            y[87] = 0.150000005960464;
            y[88] = -0.186875;
            y[89] = 0.05;
            y[91] = 0.150000005960464;
            y[92] = -0.186875;
            y[93] = 0.05;
            y[95] = 0.150000005960464;
            y[96] = -0.186875;
            y[97] = -0.05;
            y[99] = 0.150000005960464;
            y[100] = -0.186875;
            y[101] = -0.05;
            y[103] = 0.150000005960464;
            y[104] = -0.03375;
            y[105] = 0.;
            y[107] = 0.150000005960464;
            y[108] = 0.11125;
            y[109] = 0.;
            y[111] = 0.0500000007450581;
            y[112] = 0.11125;
            y[113] = -0.0425;
            y[115] = 0.0500000007450581;
            y[116] = 0.11125;
            y[117] = 0.0425;
            y[119] = 0.0500000007450581;
            y[120] = 0.11125;
            y[121] = 0.085;
            y[123] = 0.0500000007450581;
            y[124] = 0.11125;
            y[125] = -0.085;
            y[127] = 0.0500000007450581;
            y[128] = 0.02875;
            y[129] = -0.115;
            y[131] = 0.0299999993294477;
            y[132] = 0.05425;
            y[133] = -0.115;
            y[135] = 0.0299999993294477;
            y[136] = 0.07975;
            y[137] = -0.115;
            y[139] = 0.0299999993294477;
            y[140] = 0.10525;
            y[141] = -0.115;
            y[143] = 0.0299999993294477;
            y[144] = 0.02875;
            y[145] = -0.115;
            y[147] = 0.0299999993294477;
            y[148] = 0.05425;
            y[149] = -0.115;
            y[151] = 0.0299999993294477;
            y[152] = 0.07975;
            y[153] = -0.115;
            y[155] = 0.0299999993294477;
            y[156] = 0.10525;
            y[157] = -0.115;
            y[159] = 0.0299999993294477;
            y[160] = 0.12625;
            y[161] = -0.115;
            y[163] = 0.0299999993294477;
            y[164] = 0.13425;
            y[165] = -0.115;
            y[167] = 0.0299999993294477;
            y[168] = 0.12625;
            y[169] = -0.115;
            y[171] = 0.0299999993294477;
            y[172] = 0.02875;
            y[173] = 0.115;
            y[175] = 0.0299999993294477;
            y[176] = 0.05425;
            y[177] = 0.115;
            y[179] = 0.0299999993294477;
            y[180] = 0.07975;
            y[181] = 0.115;
            y[183] = 0.0299999993294477;
            y[184] = 0.10525;
            y[185] = 0.115;
            y[187] = 0.0299999993294477;
            y[188] = 0.02875;
            y[189] = 0.115;
            y[191] = 0.0299999993294477;
            y[192] = 0.05425;
            y[193] = 0.115;
            y[195] = 0.0299999993294477;
            y[196] = 0.07975;
            y[197] = 0.115;
            y[199] = 0.0299999993294477;
            y[200] = 0.10525;
            y[201] = 0.115;
            y[203] = 0.0299999993294477;
            y[204] = 0.12625;
            y[205] = 0.115;
            y[207] = 0.0299999993294477;
            y[208] = 0.13425;
            y[209] = 0.115;
            y[211] = 0.0299999993294477;
            y[212] = 0.12625;
            y[213] = 0.115;
            y[215] = 0.0299999993294477;
            y[216] = 0.03265;
            y[217] = 0.;
            y[219] = 0.0549999997019768;
            y[223] = 0.0549999997019768;
            y[227] = 0.0549999997019768;
            y[231] = 0.0549999997019768;
            y[235] = 0.0399999991059303;
            y[239] = 0.0399999991059303;
            y[243] = 0.0399999991059303;
            y[247] = 0.0399999991059303;
            y[251] = 0.0549999997019768;
            y[255] = 0.0549999997019768;
            y[259] = 0.0549999997019768;
            y[263] = 0.0549999997019768;
            y[267] = 0.0549999997019768;
            y[271] = 0.0549999997019768;
            y[275] = 0.0299999993294477;
            y[279] = 0.0299999993294477;
            y[283] = 0.0299999993294477;
            y[287] = 0.0299999993294477;
            y[291] = 0.0549999997019768;
            y[295] = 0.0299999993294477;
            y[299] = 0.0299999993294477;
            y[303] = 0.0299999993294477;
            y[307] = 0.0299999993294477;
            y[311] = 0.0549999997019768;
            y[315] = 0.0549999997019768;
            y[319] = 0.0549999997019768;
            y[323] = 0.0299999993294477;
            y[327] = 0.0299999993294477;
            y[331] = 0.0299999993294477;
            y[335] = 0.0299999993294477;
            y[339] = 0.0299999993294477;
            y[343] = 0.0299999993294477;
            y[347] = 0.0549999997019768;
            y[351] = 0.0549999997019768;
            y[355] = 0.0299999993294477;
            y[359] = 0.0299999993294477;
            y[363] = 0.0299999993294477;
            y[367] = 0.0299999993294477;
            y[371] = 0.0549999997019768;
            y[375] = 0.0549999997019768;
            y[379] = 0.0500000007450581;
            y[383] = 0.0500000007450581;
            y[387] = 0.0500000007450581;
            y[391] = 0.0500000007450581;
            y[395] = 0.0120000001043081;
            y[399] = 0.0120000001043081;
            y[403] = 0.0120000001043081;
            y[407] = 0.0120000001043081;
            y[411] = 0.0120000001043081;
            y[415] = 0.0120000001043081;
            y[419] = 0.0120000001043081;
            y[423] = 0.0120000001043081;
            y[427] = 0.0120000001043081;
            y[431] = 0.0120000001043081;
            y[435] = 0.0120000001043081;
            y[439] = 0.0120000001043081;
            y[440] = 0.013125;
            y[441] = 0.;
            y[443] = 0.0700000002980232;

            for (auto i = 0U; i < 111; ++i)
            {
                out.x[i] = y[i * 4 + 0];
                out.y[i] = y[i * 4 + 1];
                out.z[i] = y[i * 4 + 2];
                out.r[i] = y[i * 4 + 3];
            }
        }

        using Debug = std::
            pair<std::vector<std::vector<std::string>>, std::vector<std::pair<std::size_t, std::size_t>>>;

        template <std::size_t rake>
        inline static auto fkcc_debug(
            const vamp::collision::Environment<FloatVector<rake>> &environment,
            const ConfigurationBlock<rake> &x) noexcept -> Debug
        {
            std::array<FloatVector<rake, 1>, 50> v;
            std::array<FloatVector<rake, 1>, 504> y;

            v[0] = 0.37743 + x[0];
            y[82] = 0.15 + v[0];
            y[86] = 0.15 + v[0];
            y[90] = 0.3 + v[0];
            y[94] = 0.45 + v[0];
            y[98] = 0.45 + v[0];
            y[102] = 0.3 + v[0];
            y[106] = 0.663001417713939 + v[0];
            y[110] = 0.661001417713939 + v[0];
            y[114] = 0.661001417713939 + v[0];
            y[118] = 0.661001417713939 + v[0];
            y[122] = 0.661001417713939 + v[0];
            y[126] = 0.661001417713939 + v[0];
            y[130] = 0.633001417713939 + v[0];
            y[134] = 0.633001417713939 + v[0];
            y[138] = 0.633001417713939 + v[0];
            y[142] = 0.633001417713939 + v[0];
            y[146] = 0.688001417713939 + v[0];
            y[150] = 0.688001417713939 + v[0];
            y[154] = 0.688001417713939 + v[0];
            y[158] = 0.688001417713939 + v[0];
            y[162] = 0.678001417713939 + v[0];
            y[166] = 0.660501417713939 + v[0];
            y[170] = 0.643001417713939 + v[0];
            y[174] = 0.633001417713939 + v[0];
            y[178] = 0.633001417713939 + v[0];
            y[182] = 0.633001417713939 + v[0];
            y[186] = 0.633001417713939 + v[0];
            y[190] = 0.688001417713939 + v[0];
            y[194] = 0.688001417713939 + v[0];
            y[198] = 0.688001417713939 + v[0];
            y[202] = 0.688001417713939 + v[0];
            y[206] = 0.678001417713939 + v[0];
            y[210] = 0.660501417713939 + v[0];
            y[214] = 0.643001417713939 + v[0];
            y[218] = 0.34858 + v[0];
            v[1] = cos(x[1]);
            v[2] = sin(x[1]);
            v[3] = -v[2];
            y[220] = 0.03265 + 0.025 * v[1] + -0.015 * v[3];
            y[221] = 0.025 * v[2] + -0.015 * v[1];
            y[222] = 0.035 + y[218];
            y[224] = 0.03265 + 0.05 * v[1] + -0.03 * v[3];
            y[225] = 0.05 * v[2] + -0.03 * v[1];
            y[226] = 0.06 + y[218];
            y[228] = 0.03265 + 0.12 * v[1] + -0.03 * v[3];
            y[229] = 0.12 * v[2] + -0.03 * v[1];
            y[230] = 0.06 + y[218];
            v[4] = cos(x[2]);
            v[5] = v[1] * v[4];
            v[6] = sin(x[2]);
            v[7] = v[1] * v[6];
            v[8] = 0.03265 + 0.117 * v[1];
            y[232] = 0.025 * v[5] + 0.04 * v[3] + 0.025 * v[7] + v[8];
            v[9] = v[2] * v[4];
            v[10] = v[2] * v[6];
            v[11] = 0.117 * v[2];
            y[233] = 0.025 * v[9] + 0.04 * v[1] + 0.025 * v[10] + v[11];
            v[6] = -v[6];
            v[12] = 0.0599999999999999 + y[218];
            y[234] = 0.025 * v[6] + 0.025 * v[4] + v[12];
            y[236] = -0.025 * v[5] + 0.04 * v[3] + -0.025 * v[7] + v[8];
            y[237] = -0.025 * v[9] + 0.04 * v[1] + -0.025 * v[10] + v[11];
            y[238] = -0.025 * v[6] + -0.025 * v[4] + v[12];
            y[240] = 0.025 * v[5] + 0.04 * v[3] + -0.025 * v[7] + v[8];
            y[241] = 0.025 * v[9] + 0.04 * v[1] + -0.025 * v[10] + v[11];
            y[242] = 0.025 * v[6] + -0.025 * v[4] + v[12];
            y[244] = -0.025 * v[5] + 0.04 * v[3] + 0.025 * v[7] + v[8];
            y[245] = -0.025 * v[9] + 0.04 * v[1] + 0.025 * v[10] + v[11];
            y[246] = -0.025 * v[6] + 0.025 * v[4] + v[12];
            y[248] = 0.08 * v[5] + v[8];
            y[249] = 0.08 * v[9] + v[11];
            y[250] = 0.08 * v[6] + v[12];
            y[252] = 0.11 * v[5] + v[8];
            y[253] = 0.11 * v[9] + v[11];
            y[254] = 0.11 * v[6] + v[12];
            y[256] = 0.14 * v[5] + v[8];
            y[257] = 0.14 * v[9] + v[11];
            y[258] = 0.14 * v[6] + v[12];
            v[13] = v[8] + 0.219 * v[5];
            y[260] = -0.02 * v[5] + v[13];
            v[14] = v[11] + 0.219 * v[9];
            y[261] = -0.02 * v[9] + v[14];
            v[15] = v[12] + 0.219 * v[6];
            y[262] = -0.02 * v[6] + v[15];
            y[264] = 0.03 * v[5] + v[13];
            y[265] = 0.03 * v[9] + v[14];
            y[266] = 0.03 * v[6] + v[15];
            y[268] = 0.08 * v[5] + v[13];
            y[269] = 0.08 * v[9] + v[14];
            y[270] = 0.08 * v[6] + v[15];
            v[16] = cos(x[3]);
            v[17] = sin(x[3]);
            v[18] = v[3] * v[16] + v[7] * v[17];
            v[19] = -v[17];
            v[20] = v[3] * v[19] + v[7] * v[16];
            y[272] = 0.11 * v[5] + -0.045 * v[18] + 0.02 * v[20] + v[13];
            v[21] = v[1] * v[16] + v[10] * v[17];
            v[19] = v[1] * v[19] + v[10] * v[16];
            y[273] = 0.11 * v[9] + -0.045 * v[21] + 0.02 * v[19] + v[14];
            v[17] = v[4] * v[17];
            v[16] = v[4] * v[16];
            y[274] = 0.11 * v[6] + -0.045 * v[17] + 0.02 * v[16] + v[15];
            y[276] = 0.11 * v[5] + -0.045 * v[18] + -0.02 * v[20] + v[13];
            y[277] = 0.11 * v[9] + -0.045 * v[21] + -0.02 * v[19] + v[14];
            y[278] = 0.11 * v[6] + -0.045 * v[17] + -0.02 * v[16] + v[15];
            y[280] = 0.155 * v[5] + -0.045 * v[18] + 0.02 * v[20] + v[13];
            y[281] = 0.155 * v[9] + -0.045 * v[21] + 0.02 * v[19] + v[14];
            y[282] = 0.155 * v[6] + -0.045 * v[17] + 0.02 * v[16] + v[15];
            y[284] = 0.155 * v[5] + -0.045 * v[18] + -0.02 * v[20] + v[13];
            y[285] = 0.155 * v[9] + -0.045 * v[21] + -0.02 * v[19] + v[14];
            y[286] = 0.155 * v[6] + -0.045 * v[17] + -0.02 * v[16] + v[15];
            y[288] = 0.13 * v[5] + v[13];
            y[289] = 0.13 * v[9] + v[14];
            y[290] = 0.13 * v[6] + v[15];
            v[22] = cos(x[4]);
            v[23] = sin(x[4]);
            v[24] = -v[23];
            v[25] = v[5] * v[22] + v[20] * v[24];
            v[26] = v[5] * v[23] + v[20] * v[22];
            v[27] = v[13] + 0.133 * v[5];
            y[292] = 0.02 * v[25] + 0.045 * v[18] + 0.02 * v[26] + v[27];
            v[28] = v[9] * v[22] + v[19] * v[24];
            v[29] = v[9] * v[23] + v[19] * v[22];
            v[30] = v[14] + 0.133 * v[9];
            y[293] = 0.02 * v[28] + 0.045 * v[21] + 0.02 * v[29] + v[30];
            v[24] = v[6] * v[22] + v[16] * v[24];
            v[23] = v[6] * v[23] + v[16] * v[22];
            v[22] = v[15] + 0.133 * v[6];
            y[294] = 0.02 * v[24] + 0.045 * v[17] + 0.02 * v[23] + v[22];
            y[296] = 0.02 * v[25] + 0.045 * v[18] + -0.02 * v[26] + v[27];
            y[297] = 0.02 * v[28] + 0.045 * v[21] + -0.02 * v[29] + v[30];
            y[298] = 0.02 * v[24] + 0.045 * v[17] + -0.02 * v[23] + v[22];
            y[300] = -0.02 * v[25] + 0.045 * v[18] + 0.02 * v[26] + v[27];
            y[301] = -0.02 * v[28] + 0.045 * v[21] + 0.02 * v[29] + v[30];
            y[302] = -0.02 * v[24] + 0.045 * v[17] + 0.02 * v[23] + v[22];
            y[304] = -0.02 * v[25] + 0.045 * v[18] + -0.02 * v[26] + v[27];
            y[305] = -0.02 * v[28] + 0.045 * v[21] + -0.02 * v[29] + v[30];
            y[306] = -0.02 * v[24] + 0.045 * v[17] + -0.02 * v[23] + v[22];
            y[308] = 0.08 * v[25] + v[27];
            y[309] = 0.08 * v[28] + v[30];
            y[310] = 0.08 * v[24] + v[22];
            y[312] = 0.14 * v[25] + v[27];
            y[313] = 0.14 * v[28] + v[30];
            y[314] = 0.14 * v[24] + v[22];
            y[316] = v[27] + 0.197 * v[25];
            y[317] = v[30] + 0.197 * v[28];
            y[318] = v[22] + 0.197 * v[24];
            v[31] = cos(x[5]);
            v[32] = sin(x[5]);
            v[33] = v[18] * v[31] + v[26] * v[32];
            v[34] = -v[32];
            v[35] = v[18] * v[34] + v[26] * v[31];
            y[320] = 0.05 * v[25] + -0.06 * v[33] + 0.02 * v[35] + y[316];
            v[36] = v[21] * v[31] + v[29] * v[32];
            v[37] = v[21] * v[34] + v[29] * v[31];
            y[321] = 0.05 * v[28] + -0.06 * v[36] + 0.02 * v[37] + y[317];
            v[32] = v[17] * v[31] + v[23] * v[32];
            v[34] = v[17] * v[34] + v[23] * v[31];
            y[322] = 0.05 * v[24] + -0.06 * v[32] + 0.02 * v[34] + y[318];
            y[324] = 0.05 * v[25] + -0.06 * v[33] + -0.02 * v[35] + y[316];
            y[325] = 0.05 * v[28] + -0.06 * v[36] + -0.02 * v[37] + y[317];
            y[326] = 0.05 * v[24] + -0.06 * v[32] + -0.02 * v[34] + y[318];
            y[328] = 0.1 * v[25] + -0.06 * v[33] + 0.02 * v[35] + y[316];
            y[329] = 0.1 * v[28] + -0.06 * v[36] + 0.02 * v[37] + y[317];
            y[330] = 0.1 * v[24] + -0.06 * v[32] + 0.02 * v[34] + y[318];
            y[332] = 0.1 * v[25] + -0.06 * v[33] + -0.02 * v[35] + y[316];
            y[333] = 0.1 * v[28] + -0.06 * v[36] + -0.02 * v[37] + y[317];
            y[334] = 0.1 * v[24] + -0.06 * v[32] + -0.02 * v[34] + y[318];
            y[336] = 0.15 * v[25] + -0.06 * v[33] + 0.02 * v[35] + y[316];
            y[337] = 0.15 * v[28] + -0.06 * v[36] + 0.02 * v[37] + y[317];
            y[338] = 0.15 * v[24] + -0.06 * v[32] + 0.02 * v[34] + y[318];
            y[340] = 0.15 * v[25] + -0.06 * v[33] + -0.02 * v[35] + y[316];
            y[341] = 0.15 * v[28] + -0.06 * v[36] + -0.02 * v[37] + y[317];
            y[342] = 0.15 * v[24] + -0.06 * v[32] + -0.02 * v[34] + y[318];
            y[344] = y[316] + 0.1245 * v[25];
            y[345] = y[317] + 0.1245 * v[28];
            y[346] = y[318] + 0.1245 * v[24];
            v[31] = cos(x[6]);
            v[38] = sin(x[6]);
            v[39] = -v[38];
            v[40] = v[25] * v[31] + v[35] * v[39];
            y[348] = 0.06 * v[40] + y[344];
            v[41] = v[28] * v[31] + v[37] * v[39];
            y[349] = 0.06 * v[41] + y[345];
            v[39] = v[24] * v[31] + v[34] * v[39];
            y[350] = 0.06 * v[39] + y[346];
            v[42] = v[25] * v[38] + v[35] * v[31];
            y[352] = 0.02 * v[40] + 0.045 * v[33] + 0.02 * v[42] + y[344];
            v[43] = v[28] * v[38] + v[37] * v[31];
            y[353] = 0.02 * v[41] + 0.045 * v[36] + 0.02 * v[43] + y[345];
            v[38] = v[24] * v[38] + v[34] * v[31];
            y[354] = 0.02 * v[39] + 0.045 * v[32] + 0.02 * v[38] + y[346];
            y[356] = 0.02 * v[40] + 0.045 * v[33] + -0.02 * v[42] + y[344];
            y[357] = 0.02 * v[41] + 0.045 * v[36] + -0.02 * v[43] + y[345];
            y[358] = 0.02 * v[39] + 0.045 * v[32] + -0.02 * v[38] + y[346];
            y[360] = -0.02 * v[40] + 0.045 * v[33] + 0.02 * v[42] + y[344];
            y[361] = -0.02 * v[41] + 0.045 * v[36] + 0.02 * v[43] + y[345];
            y[362] = -0.02 * v[39] + 0.045 * v[32] + 0.02 * v[38] + y[346];
            y[364] = -0.02 * v[40] + 0.045 * v[33] + -0.02 * v[42] + y[344];
            y[365] = -0.02 * v[41] + 0.045 * v[36] + -0.02 * v[43] + y[345];
            y[366] = -0.02 * v[39] + 0.045 * v[32] + -0.02 * v[38] + y[346];
            y[372] = y[344] + 0.1385 * v[40];
            y[368] = -0.03 * v[40] + y[372];
            y[373] = y[345] + 0.1385 * v[41];
            y[369] = -0.03 * v[41] + y[373];
            y[374] = y[346] + 0.1385 * v[39];
            y[370] = -0.03 * v[39] + y[374];
            v[31] = cos(x[7]);
            v[44] = sin(x[7]);
            v[45] = v[33] * v[31] + v[42] * v[44];
            y[376] = 0.09645 * v[40] + 0.02 * v[45] + y[372];
            v[46] = v[36] * v[31] + v[43] * v[44];
            y[377] = 0.09645 * v[41] + 0.02 * v[46] + y[373];
            v[47] = v[32] * v[31] + v[38] * v[44];
            y[378] = 0.09645 * v[39] + 0.02 * v[47] + y[374];
            y[380] = 0.09645 * v[40] + -0.02 * v[45] + y[372];
            y[381] = 0.09645 * v[41] + -0.02 * v[46] + y[373];
            y[382] = 0.09645 * v[39] + -0.02 * v[47] + y[374];
            y[384] = 0.06645 * v[40] + 0.02 * v[45] + y[372];
            y[385] = 0.06645 * v[41] + 0.02 * v[46] + y[373];
            y[386] = 0.06645 * v[39] + 0.02 * v[47] + y[374];
            y[388] = 0.06645 * v[40] + -0.02 * v[45] + y[372];
            y[389] = 0.06645 * v[41] + -0.02 * v[46] + y[373];
            y[390] = 0.06645 * v[39] + -0.02 * v[47] + y[374];
            v[44] = -v[44];
            v[48] = v[33] * v[44] + v[42] * v[31];
            y[392] = 0.18345 * v[40] + -0.056925 * v[45] + -0.005 * v[48] + y[372];
            v[49] = v[36] * v[44] + v[43] * v[31];
            y[393] = 0.18345 * v[41] + -0.056925 * v[46] + -0.005 * v[49] + y[373];
            v[44] = v[32] * v[44] + v[38] * v[31];
            y[394] = 0.18345 * v[39] + -0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[396] = 0.18345 * v[40] + -0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[397] = 0.18345 * v[41] + -0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[398] = 0.18345 * v[39] + -0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[400] = 0.16645 * v[40] + -0.056925 * v[45] + -0.005 * v[48] + y[372];
            y[401] = 0.16645 * v[41] + -0.056925 * v[46] + -0.005 * v[49] + y[373];
            y[402] = 0.16645 * v[39] + -0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[404] = 0.16645 * v[40] + -0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[405] = 0.16645 * v[41] + -0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[406] = 0.16645 * v[39] + -0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[408] = 0.14945 * v[40] + -0.056925 * v[45] + -0.005 * v[48] + y[372];
            y[409] = 0.14945 * v[41] + -0.056925 * v[46] + -0.005 * v[49] + y[373];
            y[410] = 0.14945 * v[39] + -0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[412] = 0.14945 * v[40] + -0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[413] = 0.14945 * v[41] + -0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[414] = 0.14945 * v[39] + -0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[416] = 0.18345 * v[40] + 0.056925 * v[45] + -0.005 * v[48] + y[372];
            y[417] = 0.18345 * v[41] + 0.056925 * v[46] + -0.005 * v[49] + y[373];
            y[418] = 0.18345 * v[39] + 0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[420] = 0.18345 * v[40] + 0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[421] = 0.18345 * v[41] + 0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[422] = 0.18345 * v[39] + 0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[424] = 0.16645 * v[40] + 0.056925 * v[45] + -0.005 * v[48] + y[372];
            y[425] = 0.16645 * v[41] + 0.056925 * v[46] + -0.005 * v[49] + y[373];
            y[426] = 0.16645 * v[39] + 0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[428] = 0.16645 * v[40] + 0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[429] = 0.16645 * v[41] + 0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[430] = 0.16645 * v[39] + 0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[432] = 0.14945 * v[40] + 0.056925 * v[45] + -0.005 * v[48] + y[372];
            y[433] = 0.14945 * v[41] + 0.056925 * v[46] + -0.005 * v[49] + y[373];
            y[434] = 0.14945 * v[39] + 0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[436] = 0.14945 * v[40] + 0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[437] = 0.14945 * v[41] + 0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[438] = 0.14945 * v[39] + 0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[442] = 0.24 + v[0];
            y[454] = 0.300000011920929 + v[0];
            y[458] = 0.662302553653717 + v[0];
            y[460] = 0.03265 + 0.0599999986588955 * v[1] + -0.0149999996647239 * v[3];
            y[461] = 0.0599999986588955 * v[2] + -0.0149999996647239 * v[1];
            y[462] = 0.0299999993294477 + y[218];
            y[464] =
                0.0631452798843384 * v[5] + 0.018631448969245 * v[3] + 1.73472347597681e-18 * v[7] + v[8];
            y[465] =
                0.0631452798843384 * v[9] + 0.018631448969245 * v[1] + 1.73472347597681e-18 * v[10] + v[11];
            y[466] = 0.0631452798843384 * v[6] + 1.73472347597681e-18 * v[4] + v[12];
            y[468] = 0.056335523724556 * v[5] + -0.0196291357278824 * v[18] + -1.73472347597681e-18 * v[20] +
                     v[13];
            y[469] = 0.056335523724556 * v[9] + -0.0196291357278824 * v[21] + -1.73472347597681e-18 * v[19] +
                     v[14];
            y[470] = 0.056335523724556 * v[6] + -0.0196291357278824 * v[17] + -1.73472347597681e-18 * v[16] +
                     v[15];
            y[472] = 0.0710262209177017 * v[25] + 0.0193988755345345 * v[18] + 1.73472347597681e-18 * v[26] +
                     v[27];
            y[473] = 0.0710262209177017 * v[28] + 0.0193988755345345 * v[21] + 1.73472347597681e-18 * v[29] +
                     v[30];
            y[474] = 0.0710262209177017 * v[24] + 0.0193988755345345 * v[17] + 1.73472347597681e-18 * v[23] +
                     v[22];
            y[476] = 0.0643894299864769 * v[25] + -0.0257557742297649 * v[33] +
                     -1.73472347597681e-18 * v[35] + y[316];
            y[477] = 0.0643894299864769 * v[28] + -0.0257557742297649 * v[36] +
                     -1.73472347597681e-18 * v[37] + y[317];
            y[478] = 0.0643894299864769 * v[24] + -0.0257557742297649 * v[32] +
                     -1.73472347597681e-18 * v[34] + y[318];
            y[480] = 0.0294021144509315 * v[40] + 0.0172113105654716 * v[33] + 4.33680868994202e-18 * v[42] +
                     y[344];
            y[481] = 0.0294021144509315 * v[41] + 0.0172113105654716 * v[36] + 4.33680868994202e-18 * v[43] +
                     y[345];
            y[482] = 0.0294021144509315 * v[39] + 0.0172113105654716 * v[32] + 4.33680868994202e-18 * v[38] +
                     y[346];
            y[484] = -0.0149999996647239 * v[40] + y[372];
            y[485] = -0.0149999996647239 * v[41] + y[373];
            y[486] = -0.0149999996647239 * v[39] + y[374];
            y[488] = 0.0814500004053116 * v[40] + y[372];
            y[489] = 0.0814500004053116 * v[41] + y[373];
            y[490] = 0.0814500004053116 * v[39] + y[374];
            y[492] = 0.166449993848801 * v[40] + -0.0569249987602234 * v[45] + y[372];
            y[493] = 0.166449993848801 * v[41] + -0.0569249987602234 * v[46] + y[373];
            y[494] = 0.166449993848801 * v[39] + -0.0569249987602234 * v[47] + y[374];
            y[496] = 0.166449993848801 * v[40] + 0.0569249987602234 * v[45] + y[372];
            y[497] = 0.166449993848801 * v[41] + 0.0569249987602234 * v[46] + y[373];
            y[498] = 0.166449993848801 * v[39] + 0.0569249987602234 * v[47] + y[374];
            y[502] = 0.239999994635582 + v[0];
            // dependent variables without operations
            y[0] = -0.12;
            y[1] = 0.;
            y[2] = 0.182;
            y[3] = 0.239999994635582;
            y[4] = 0.225;
            y[5] = 0.;
            y[6] = 0.31;
            y[7] = 0.0659999996423721;
            y[8] = 0.08;
            y[9] = -0.06;
            y[10] = 0.16;
            y[11] = 0.219999998807907;
            y[12] = 0.215;
            y[13] = -0.07;
            y[14] = 0.31;
            y[15] = 0.0659999996423721;
            y[16] = 0.185;
            y[17] = -0.135;
            y[18] = 0.31;
            y[19] = 0.0659999996423721;
            y[20] = 0.13;
            y[21] = -0.185;
            y[22] = 0.31;
            y[23] = 0.0659999996423721;
            y[24] = 0.065;
            y[25] = -0.2;
            y[26] = 0.31;
            y[27] = 0.0659999996423721;
            y[28] = 0.01;
            y[29] = -0.2;
            y[30] = 0.31;
            y[31] = 0.0659999996423721;
            y[32] = 0.08;
            y[33] = 0.06;
            y[34] = 0.16;
            y[35] = 0.219999998807907;
            y[36] = 0.215;
            y[37] = 0.07;
            y[38] = 0.31;
            y[39] = 0.0659999996423721;
            y[40] = 0.185;
            y[41] = 0.135;
            y[42] = 0.31;
            y[43] = 0.0659999996423721;
            y[44] = 0.13;
            y[45] = 0.185;
            y[46] = 0.31;
            y[47] = 0.0659999996423721;
            y[48] = 0.065;
            y[49] = 0.2;
            y[50] = 0.31;
            y[51] = 0.0659999996423721;
            y[52] = 0.01;
            y[53] = 0.2;
            y[54] = 0.31;
            y[55] = 0.0659999996423721;
            y[56] = -0.186875;
            y[57] = -0.07;
            y[58] = 0.727425;
            y[59] = 0.119999997317791;
            y[60] = -0.186875;
            y[61] = 0.07;
            y[62] = 0.727425;
            y[63] = 0.119999997317791;
            y[64] = -0.186875;
            y[65] = -0.07;
            y[66] = 0.577425;
            y[67] = 0.119999997317791;
            y[68] = -0.186875;
            y[69] = 0.07;
            y[70] = 0.577425;
            y[71] = 0.119999997317791;
            y[72] = -0.186875;
            y[73] = 0.07;
            y[74] = 0.447425;
            y[75] = 0.119999997317791;
            y[76] = -0.186875;
            y[77] = -0.07;
            y[78] = 0.447425;
            y[79] = 0.119999997317791;
            y[80] = -0.186875;
            y[81] = -0.05;
            y[83] = 0.150000005960464;
            y[84] = -0.186875;
            y[85] = 0.05;
            y[87] = 0.150000005960464;
            y[88] = -0.186875;
            y[89] = 0.05;
            y[91] = 0.150000005960464;
            y[92] = -0.186875;
            y[93] = 0.05;
            y[95] = 0.150000005960464;
            y[96] = -0.186875;
            y[97] = -0.05;
            y[99] = 0.150000005960464;
            y[100] = -0.186875;
            y[101] = -0.05;
            y[103] = 0.150000005960464;
            y[104] = -0.03375;
            y[105] = 0.;
            y[107] = 0.150000005960464;
            y[108] = 0.11125;
            y[109] = 0.;
            y[111] = 0.0500000007450581;
            y[112] = 0.11125;
            y[113] = -0.0425;
            y[115] = 0.0500000007450581;
            y[116] = 0.11125;
            y[117] = 0.0425;
            y[119] = 0.0500000007450581;
            y[120] = 0.11125;
            y[121] = 0.085;
            y[123] = 0.0500000007450581;
            y[124] = 0.11125;
            y[125] = -0.085;
            y[127] = 0.0500000007450581;
            y[128] = 0.02875;
            y[129] = -0.115;
            y[131] = 0.0299999993294477;
            y[132] = 0.05425;
            y[133] = -0.115;
            y[135] = 0.0299999993294477;
            y[136] = 0.07975;
            y[137] = -0.115;
            y[139] = 0.0299999993294477;
            y[140] = 0.10525;
            y[141] = -0.115;
            y[143] = 0.0299999993294477;
            y[144] = 0.02875;
            y[145] = -0.115;
            y[147] = 0.0299999993294477;
            y[148] = 0.05425;
            y[149] = -0.115;
            y[151] = 0.0299999993294477;
            y[152] = 0.07975;
            y[153] = -0.115;
            y[155] = 0.0299999993294477;
            y[156] = 0.10525;
            y[157] = -0.115;
            y[159] = 0.0299999993294477;
            y[160] = 0.12625;
            y[161] = -0.115;
            y[163] = 0.0299999993294477;
            y[164] = 0.13425;
            y[165] = -0.115;
            y[167] = 0.0299999993294477;
            y[168] = 0.12625;
            y[169] = -0.115;
            y[171] = 0.0299999993294477;
            y[172] = 0.02875;
            y[173] = 0.115;
            y[175] = 0.0299999993294477;
            y[176] = 0.05425;
            y[177] = 0.115;
            y[179] = 0.0299999993294477;
            y[180] = 0.07975;
            y[181] = 0.115;
            y[183] = 0.0299999993294477;
            y[184] = 0.10525;
            y[185] = 0.115;
            y[187] = 0.0299999993294477;
            y[188] = 0.02875;
            y[189] = 0.115;
            y[191] = 0.0299999993294477;
            y[192] = 0.05425;
            y[193] = 0.115;
            y[195] = 0.0299999993294477;
            y[196] = 0.07975;
            y[197] = 0.115;
            y[199] = 0.0299999993294477;
            y[200] = 0.10525;
            y[201] = 0.115;
            y[203] = 0.0299999993294477;
            y[204] = 0.12625;
            y[205] = 0.115;
            y[207] = 0.0299999993294477;
            y[208] = 0.13425;
            y[209] = 0.115;
            y[211] = 0.0299999993294477;
            y[212] = 0.12625;
            y[213] = 0.115;
            y[215] = 0.0299999993294477;
            y[216] = 0.03265;
            y[217] = 0.;
            y[219] = 0.0549999997019768;
            y[223] = 0.0549999997019768;
            y[227] = 0.0549999997019768;
            y[231] = 0.0549999997019768;
            y[235] = 0.0399999991059303;
            y[239] = 0.0399999991059303;
            y[243] = 0.0399999991059303;
            y[247] = 0.0399999991059303;
            y[251] = 0.0549999997019768;
            y[255] = 0.0549999997019768;
            y[259] = 0.0549999997019768;
            y[263] = 0.0549999997019768;
            y[267] = 0.0549999997019768;
            y[271] = 0.0549999997019768;
            y[275] = 0.0299999993294477;
            y[279] = 0.0299999993294477;
            y[283] = 0.0299999993294477;
            y[287] = 0.0299999993294477;
            y[291] = 0.0549999997019768;
            y[295] = 0.0299999993294477;
            y[299] = 0.0299999993294477;
            y[303] = 0.0299999993294477;
            y[307] = 0.0299999993294477;
            y[311] = 0.0549999997019768;
            y[315] = 0.0549999997019768;
            y[319] = 0.0549999997019768;
            y[323] = 0.0299999993294477;
            y[327] = 0.0299999993294477;
            y[331] = 0.0299999993294477;
            y[335] = 0.0299999993294477;
            y[339] = 0.0299999993294477;
            y[343] = 0.0299999993294477;
            y[347] = 0.0549999997019768;
            y[351] = 0.0549999997019768;
            y[355] = 0.0299999993294477;
            y[359] = 0.0299999993294477;
            y[363] = 0.0299999993294477;
            y[367] = 0.0299999993294477;
            y[371] = 0.0549999997019768;
            y[375] = 0.0549999997019768;
            y[379] = 0.0500000007450581;
            y[383] = 0.0500000007450581;
            y[387] = 0.0500000007450581;
            y[391] = 0.0500000007450581;
            y[395] = 0.0120000001043081;
            y[399] = 0.0120000001043081;
            y[403] = 0.0120000001043081;
            y[407] = 0.0120000001043081;
            y[411] = 0.0120000001043081;
            y[415] = 0.0120000001043081;
            y[419] = 0.0120000001043081;
            y[423] = 0.0120000001043081;
            y[427] = 0.0120000001043081;
            y[431] = 0.0120000001043081;
            y[435] = 0.0120000001043081;
            y[439] = 0.0120000001043081;
            y[440] = 0.013125;
            y[441] = 0.;
            y[443] = 0.0700000002980232;
            y[444] = -0.0201118849217892;
            y[445] = -5.20417042793042e-17;
            y[446] = 0.188239961862564;
            y[447] = 0.340082824230194;
            y[448] = -0.186875000596046;
            y[449] = 2.77555756156289e-17;
            y[450] = 0.587424993515015;
            y[451] = 0.276524752378464;
            y[452] = -0.186875001490116;
            y[453] = 0.;
            y[455] = 0.308113902807236;
            y[456] = 0.0132126799225807;
            y[457] = -2.42861286636753e-17;
            y[459] = 0.196967884898186;
            y[463] = 0.123738631606102;
            y[467] = 0.134080842137337;
            y[471] = 0.133818879723549;
            y[475] = 0.12664982676506;
            y[479] = 0.124349541962147;
            y[483] = 0.0901064053177834;
            y[487] = 0.0700000002980232;
            y[491] = 0.0750000029802322;
            y[495] = 0.0297200456261635;
            y[499] = 0.0297200456261635;
            y[500] = 0.0131250014901161;
            y[501] = 0.;
            y[503] = 0.0700000002980232;

            Debug output;

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[0], y[1], y[2], y[3]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[4], y[5], y[6], y[7]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[8], y[9], y[10], y[11]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[12], y[13], y[14], y[15]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[16], y[17], y[18], y[19]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[20], y[21], y[22], y[23]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[24], y[25], y[26], y[27]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[28], y[29], y[30], y[31]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[32], y[33], y[34], y[35]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[36], y[37], y[38], y[39]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[40], y[41], y[42], y[43]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[44], y[45], y[46], y[47]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[48], y[49], y[50], y[51]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[52], y[53], y[54], y[55]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[56], y[57], y[58], y[59]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[60], y[61], y[62], y[63]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[64], y[65], y[66], y[67]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[68], y[69], y[70], y[71]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[72], y[73], y[74], y[75]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[76], y[77], y[78], y[79]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[80], y[81], y[82], y[83]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[84], y[85], y[86], y[87]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[88], y[89], y[90], y[91]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[92], y[93], y[94], y[95]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[96], y[97], y[98], y[99]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[100], y[101], y[102], y[103]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[104], y[105], y[106], y[107]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[108], y[109], y[110], y[111]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[112], y[113], y[114], y[115]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[116], y[117], y[118], y[119]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[120], y[121], y[122], y[123]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[124], y[125], y[126], y[127]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[128], y[129], y[130], y[131]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[132], y[133], y[134], y[135]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[136], y[137], y[138], y[139]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[140], y[141], y[142], y[143]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[144], y[145], y[146], y[147]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[148], y[149], y[150], y[151]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[152], y[153], y[154], y[155]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[156], y[157], y[158], y[159]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[160], y[161], y[162], y[163]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[164], y[165], y[166], y[167]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[168], y[169], y[170], y[171]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[172], y[173], y[174], y[175]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[176], y[177], y[178], y[179]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[180], y[181], y[182], y[183]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[184], y[185], y[186], y[187]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[188], y[189], y[190], y[191]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[192], y[193], y[194], y[195]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[196], y[197], y[198], y[199]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[200], y[201], y[202], y[203]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[204], y[205], y[206], y[207]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[208], y[209], y[210], y[211]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[212], y[213], y[214], y[215]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[216], y[217], y[218], y[219]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[220], y[221], y[222], y[223]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[224], y[225], y[226], y[227]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[228], y[229], y[230], y[231]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[232], y[233], y[234], y[235]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[236], y[237], y[238], y[239]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[240], y[241], y[242], y[243]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[244], y[245], y[246], y[247]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[248], y[249], y[250], y[251]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[252], y[253], y[254], y[255]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[256], y[257], y[258], y[259]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[260], y[261], y[262], y[263]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[264], y[265], y[266], y[267]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[268], y[269], y[270], y[271]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[272], y[273], y[274], y[275]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[276], y[277], y[278], y[279]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[280], y[281], y[282], y[283]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[284], y[285], y[286], y[287]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[288], y[289], y[290], y[291]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[292], y[293], y[294], y[295]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[296], y[297], y[298], y[299]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[300], y[301], y[302], y[303]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[304], y[305], y[306], y[307]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[308], y[309], y[310], y[311]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[312], y[313], y[314], y[315]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[316], y[317], y[318], y[319]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[320], y[321], y[322], y[323]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[324], y[325], y[326], y[327]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[328], y[329], y[330], y[331]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[332], y[333], y[334], y[335]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[336], y[337], y[338], y[339]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[340], y[341], y[342], y[343]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[344], y[345], y[346], y[347]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[348], y[349], y[350], y[351]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[352], y[353], y[354], y[355]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[356], y[357], y[358], y[359]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[360], y[361], y[362], y[363]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[364], y[365], y[366], y[367]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[368], y[369], y[370], y[371]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[372], y[373], y[374], y[375]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[376], y[377], y[378], y[379]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[380], y[381], y[382], y[383]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[384], y[385], y[386], y[387]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[388], y[389], y[390], y[391]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[392], y[393], y[394], y[395]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[396], y[397], y[398], y[399]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[400], y[401], y[402], y[403]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[404], y[405], y[406], y[407]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[408], y[409], y[410], y[411]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[412], y[413], y[414], y[415]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[416], y[417], y[418], y[419]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[420], y[421], y[422], y[423]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[424], y[425], y[426], y[427]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[428], y[429], y[430], y[431]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[432], y[433], y[434], y[435]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[436], y[437], y[438], y[439]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[440], y[441], y[442], y[443]));

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(0, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(0, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(0, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(0, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(0, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(0, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(1, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(1, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(1, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(1, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(1, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(1, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(2, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(2, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(2, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(2, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(2, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(2, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(3, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(3, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(3, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(3, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(3, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(3, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(4, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(4, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(4, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(4, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(4, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(4, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(5, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(5, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(5, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(5, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(5, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(5, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(6, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(6, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(6, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(6, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(6, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(6, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(7, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(7, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(7, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(7, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(7, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(7, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(8, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(8, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(8, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(8, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(8, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(8, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(9, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(9, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(9, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(9, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(9, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(9, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(10, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(10, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(10, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(10, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(10, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(10, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(11, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(11, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(11, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(11, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(11, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(11, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(12, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(12, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(12, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(12, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(12, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(12, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(13, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(13, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(13, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(13, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(13, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(13, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(0, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(0, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(0, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(0, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(0, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(0, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(0, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(1, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(1, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(1, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(1, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(1, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(1, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(1, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(2, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(2, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(2, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(2, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(2, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(2, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(2, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(3, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(3, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(3, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(3, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(3, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(3, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(3, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(4, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(4, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(4, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(4, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(4, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(4, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(4, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(5, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(5, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(5, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(5, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(5, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(5, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(5, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(6, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(6, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(6, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(6, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(6, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(6, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(6, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(7, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(7, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(7, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(7, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(7, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(7, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(7, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(8, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(8, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(8, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(8, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(8, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(8, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(8, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(9, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(9, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(9, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(9, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(9, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(9, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(9, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(10, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(10, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(10, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(10, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(10, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(10, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(10, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(11, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(11, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(11, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(11, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(11, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(11, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(11, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(12, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(12, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(12, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(12, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(12, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(12, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(12, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(13, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(13, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(13, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(13, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(13, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(13, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(13, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(0, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(0, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(0, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(0, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(0, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(0, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(1, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(1, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(1, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(1, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(1, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(1, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(2, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(2, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(2, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(2, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(2, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(2, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(3, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(3, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(3, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(3, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(3, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(3, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(4, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(4, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(4, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(4, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(4, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(4, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(5, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(5, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(5, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(5, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(5, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(5, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(6, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(6, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(6, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(6, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(6, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(6, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(7, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(7, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(7, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(7, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(7, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(7, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(8, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(8, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(8, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(8, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(8, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(8, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(9, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(9, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(9, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(9, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(9, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(9, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(10, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(10, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(10, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(10, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(10, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(10, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(11, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(11, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(11, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(11, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(11, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(11, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(12, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(12, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(12, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(12, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(12, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(12, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(13, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(13, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(13, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(13, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(13, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(13, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(0, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(0, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(1, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(1, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(2, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(2, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(3, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(3, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(4, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(4, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(5, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(5, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(6, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(6, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(7, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(7, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(8, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(8, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(9, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(9, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(10, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(10, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(11, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(11, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(12, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(12, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(13, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(13, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(0, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(0, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(0, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(0, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(1, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(1, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(1, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(1, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(2, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(2, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(2, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(2, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(3, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(3, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(3, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(3, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(4, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(4, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(4, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(4, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(5, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(5, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(5, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(5, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(6, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(6, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(6, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(6, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(7, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(7, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(7, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(7, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(8, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(8, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(8, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(8, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(9, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(9, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(9, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(9, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(10, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(10, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(10, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(10, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(11, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(11, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(11, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(11, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(12, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(12, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(12, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(12, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(13, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(13, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(13, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(13, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(0, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(0, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(0, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(0, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(0, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(0, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(1, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(1, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(1, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(1, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(1, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(1, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(2, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(2, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(2, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(2, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(2, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(2, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(3, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(3, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(3, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(3, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(3, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(3, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(4, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(4, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(4, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(4, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(4, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(4, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(5, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(5, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(5, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(5, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(5, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(5, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(6, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(6, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(6, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(6, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(6, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(6, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(7, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(7, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(7, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(7, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(7, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(7, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(8, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(8, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(8, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(8, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(8, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(8, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(9, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(9, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(9, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(9, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(9, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(9, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(10, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(10, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(10, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(10, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(10, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(10, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(11, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(11, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(11, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(11, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(11, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(11, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(12, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(12, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(12, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(12, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(12, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(12, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(13, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(13, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(13, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(13, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(13, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(13, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(0, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(0, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(0, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(0, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(0, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(0, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(1, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(1, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(1, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(1, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(1, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(1, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(2, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(2, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(2, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(2, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(2, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(2, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(3, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(3, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(3, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(3, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(3, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(3, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(4, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(4, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(4, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(4, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(4, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(4, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(5, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(5, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(5, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(5, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(5, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(5, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(6, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(6, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(6, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(6, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(6, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(6, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(7, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(7, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(7, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(7, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(7, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(7, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(8, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(8, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(8, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(8, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(8, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(8, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(9, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(9, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(9, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(9, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(9, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[36], y[37], y[38], y[39], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(9, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(10, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(10, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(10, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(10, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(10, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[40], y[41], y[42], y[43], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(10, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(11, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(11, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(11, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(11, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(11, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[44], y[45], y[46], y[47], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(11, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(12, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(12, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(12, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(12, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(12, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[48], y[49], y[50], y[51], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(12, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(13, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(13, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(13, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(13, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(13, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[52], y[53], y[54], y[55], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(13, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(14, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(14, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(14, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(14, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(14, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(14, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(14, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(14, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(15, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(15, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(15, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(15, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(15, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(15, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(15, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(15, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(16, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(16, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(16, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(16, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(16, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(16, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(16, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(16, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(17, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(17, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(17, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(17, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(17, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(17, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(17, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(17, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(18, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(18, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(18, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(18, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(18, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(18, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(18, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(18, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(19, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(19, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(19, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(19, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(19, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(19, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(19, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(19, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(14, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(14, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(14, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(14, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(14, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(14, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(15, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(15, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(15, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(15, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(15, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(15, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(16, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(16, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(16, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(16, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(16, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(16, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(17, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(17, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(17, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(17, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(17, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(17, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(18, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(18, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(18, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(18, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(18, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(18, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(19, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(19, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(19, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(19, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(19, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(19, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(14, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(14, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(14, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(14, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(14, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(14, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(14, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(15, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(15, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(15, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(15, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(15, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(15, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(15, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(16, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(16, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(16, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(16, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(16, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(16, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(16, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(17, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(17, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(17, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(17, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(17, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(17, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(17, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(18, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(18, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(18, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(18, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(18, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(18, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(18, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(19, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(19, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(19, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(19, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(19, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(19, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(19, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(14, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(14, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(14, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(14, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(14, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(14, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(15, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(15, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(15, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(15, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(15, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(15, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(16, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(16, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(16, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(16, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(16, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(16, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(17, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(17, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(17, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(17, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(17, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(17, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(18, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(18, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(18, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(18, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(18, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(18, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(19, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(19, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(19, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(19, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(19, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(19, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(14, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(14, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(15, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(15, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(16, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(16, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(17, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(17, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(18, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(18, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(19, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(19, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(14, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(14, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(14, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(14, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(15, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(15, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(15, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(15, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(16, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(16, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(16, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(16, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(17, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(17, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(17, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(17, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(18, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(18, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(18, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(18, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(19, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(19, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(19, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(19, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(14, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(14, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(14, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(14, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(14, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(14, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(15, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(15, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(15, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(15, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(15, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(15, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(16, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(16, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(16, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(16, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(16, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(16, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(17, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(17, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(17, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(17, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(17, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(17, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(18, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(18, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(18, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(18, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(18, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(18, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(19, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(19, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(19, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(19, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(19, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(19, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(14, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(14, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(14, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(14, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(14, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[56], y[57], y[58], y[59], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(14, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(15, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(15, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(15, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(15, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(15, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[60], y[61], y[62], y[63], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(15, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(16, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(16, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(16, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(16, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(16, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[64], y[65], y[66], y[67], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(16, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(17, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(17, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(17, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(17, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(17, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(17, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(18, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(18, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(18, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(18, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(18, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(18, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(19, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(19, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(19, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(19, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(19, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(19, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(20, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(20, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(20, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(20, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(20, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(20, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(20, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(20, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(21, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(21, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(21, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(21, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(21, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(21, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(21, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(21, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(22, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(22, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(22, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(22, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(22, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(22, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(22, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(22, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(23, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(23, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(23, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(23, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(23, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(23, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(23, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(23, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(24, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(24, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(24, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(24, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(24, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(24, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(24, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(24, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(25, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(25, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(25, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(25, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(25, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(25, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(25, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(25, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(20, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(20, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(20, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(20, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(20, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(20, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(21, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(21, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(21, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(21, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(21, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(21, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(22, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(22, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(22, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(22, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(22, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(22, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(23, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(23, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(23, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(23, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(23, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(23, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(24, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(24, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(24, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(24, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(24, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(24, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(25, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(25, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(25, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(25, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(25, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(25, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(20, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(20, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(20, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(20, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(20, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(20, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(20, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(21, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(21, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(21, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(21, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(21, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(21, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(21, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(22, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(22, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(22, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(22, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(22, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(22, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(22, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(23, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(23, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(23, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(23, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(23, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(23, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(23, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(24, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(24, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(24, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(24, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(24, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(24, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(24, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(25, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(25, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(25, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(25, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(25, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(25, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(25, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(20, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(20, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(20, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(20, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(20, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(20, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(21, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(21, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(21, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(21, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(21, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(21, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(22, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(22, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(22, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(22, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(22, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(22, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(23, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(23, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(23, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(23, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(23, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(23, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(24, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(24, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(24, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(24, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(24, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(24, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(25, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(25, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(25, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(25, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(25, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(25, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(20, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(20, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(21, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(21, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(22, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(22, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(23, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(23, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(24, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(24, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(25, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(25, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(20, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(20, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(20, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(20, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(21, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(21, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(21, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(21, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(22, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(22, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(22, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(22, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(23, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(23, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(23, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(23, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(24, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(24, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(24, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(24, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(25, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(25, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(25, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(25, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(20, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(20, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(20, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(20, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(20, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(20, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(21, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(21, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(21, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(21, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(21, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(21, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(22, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(22, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(22, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(22, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(22, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(22, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(23, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(23, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(23, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(23, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(23, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(23, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(24, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(24, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(24, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(24, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(24, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(24, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(25, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(25, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(25, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(25, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(25, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(25, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(20, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(20, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(20, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(20, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(20, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(20, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(21, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(21, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(21, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(21, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(21, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(21, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(22, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(22, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(22, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(22, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(22, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(22, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(23, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(23, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(23, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(23, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(23, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(23, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(24, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(24, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(24, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(24, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(24, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(24, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(25, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(25, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(25, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(25, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(25, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(25, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(26, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(26, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(26, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(26, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(26, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(26, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(26, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(26, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(27, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(27, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(27, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(27, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(27, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(27, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(27, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(27, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(28, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(28, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(28, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(28, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(28, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(28, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(28, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(28, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(29, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(29, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(29, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(29, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(29, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(29, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(29, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(29, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(30, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(30, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(30, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(30, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(30, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(30, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(30, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(30, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(31, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(31, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(31, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(31, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(31, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(31, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(31, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(31, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(32, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(32, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(32, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(32, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(32, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(32, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(32, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(32, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(33, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(33, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(33, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(33, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(33, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(33, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(33, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(33, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(34, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(34, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(34, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(34, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(34, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(34, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(34, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(34, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(35, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(35, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(35, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(35, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(35, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(35, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(35, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(35, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(36, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(36, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(36, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(36, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(36, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(36, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(36, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(36, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(37, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(37, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(37, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(37, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(37, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(37, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(37, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(37, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(38, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(38, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(38, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(38, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(38, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(38, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(38, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(38, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(39, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(39, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(39, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(39, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(39, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(39, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(39, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(39, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(40, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(40, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(40, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(40, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(40, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(40, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(40, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(40, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(41, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(41, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(41, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(41, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(41, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(41, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(41, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(41, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(42, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(42, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(42, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(42, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(42, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(42, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(42, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(42, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(43, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(43, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(43, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(43, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(43, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(43, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(43, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(43, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(44, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(44, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(44, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(44, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(44, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(44, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(44, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(44, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(45, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(45, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(45, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(45, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(45, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(45, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(45, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(45, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(46, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(46, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(46, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(46, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(46, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(46, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(46, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(46, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(47, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(47, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(47, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(47, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(47, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(47, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(47, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(47, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(48, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(48, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(48, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(48, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(48, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(48, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(48, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(48, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(49, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(49, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(49, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(49, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(49, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(49, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(49, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(49, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(50, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(50, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(50, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(50, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(50, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(50, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(50, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(50, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(51, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(51, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(51, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(51, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(51, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(51, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(51, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(51, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(52, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(52, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(52, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(52, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(52, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(52, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(52, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(52, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(53, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(53, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(53, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(53, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(53, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(53, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(53, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(53, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(26, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(26, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(26, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(26, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(26, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(26, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(27, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(27, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(27, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(27, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(27, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(27, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(28, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(28, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(28, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(28, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(28, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(28, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(29, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(29, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(29, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(29, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(29, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(29, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(30, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(30, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(30, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(30, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(30, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(30, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(31, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(31, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(31, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(31, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(31, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(31, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(32, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(32, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(32, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(32, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(32, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(32, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(33, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(33, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(33, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(33, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(33, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(33, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(34, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(34, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(34, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(34, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(34, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(34, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(35, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(35, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(35, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(35, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(35, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(35, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(36, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(36, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(36, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(36, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(36, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(36, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(37, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(37, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(37, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(37, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(37, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(37, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(38, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(38, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(38, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(38, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(38, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(38, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(39, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(39, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(39, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(39, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(39, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(39, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(40, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(40, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(40, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(40, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(40, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(40, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(41, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(41, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(41, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(41, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(41, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(41, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(42, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(42, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(42, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(42, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(42, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(42, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(43, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(43, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(43, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(43, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(43, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(43, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(44, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(44, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(44, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(44, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(44, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(44, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(45, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(45, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(45, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(45, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(45, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(45, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(46, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(46, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(46, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(46, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(46, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(46, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(47, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(47, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(47, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(47, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(47, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(47, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(48, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(48, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(48, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(48, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(48, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(48, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(49, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(49, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(49, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(49, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(49, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(49, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(50, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(50, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(50, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(50, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(50, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(50, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(51, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(51, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(51, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(51, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(51, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(51, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(52, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(52, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(52, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(52, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(52, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(52, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[292], y[293], y[294], y[295]))
            {
                output.second.emplace_back(53, 73);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[296], y[297], y[298], y[299]))
            {
                output.second.emplace_back(53, 74);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[300], y[301], y[302], y[303]))
            {
                output.second.emplace_back(53, 75);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[304], y[305], y[306], y[307]))
            {
                output.second.emplace_back(53, 76);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[308], y[309], y[310], y[311]))
            {
                output.second.emplace_back(53, 77);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[312], y[313], y[314], y[315]))
            {
                output.second.emplace_back(53, 78);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(26, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(26, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(26, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(26, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(26, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(26, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(26, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(27, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(27, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(27, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(27, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(27, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(27, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(27, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(28, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(28, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(28, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(28, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(28, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(28, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(28, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(29, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(29, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(29, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(29, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(29, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(29, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(29, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(30, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(30, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(30, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(30, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(30, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(30, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(30, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(31, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(31, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(31, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(31, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(31, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(31, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(31, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(32, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(32, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(32, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(32, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(32, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(32, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(32, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(33, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(33, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(33, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(33, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(33, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(33, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(33, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(34, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(34, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(34, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(34, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(34, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(34, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(34, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(35, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(35, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(35, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(35, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(35, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(35, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(35, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(36, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(36, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(36, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(36, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(36, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(36, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(36, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(37, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(37, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(37, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(37, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(37, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(37, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(37, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(38, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(38, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(38, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(38, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(38, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(38, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(38, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(39, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(39, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(39, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(39, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(39, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(39, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(39, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(40, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(40, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(40, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(40, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(40, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(40, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(40, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(41, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(41, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(41, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(41, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(41, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(41, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(41, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(42, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(42, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(42, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(42, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(42, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(42, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(42, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(43, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(43, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(43, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(43, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(43, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(43, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(43, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(44, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(44, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(44, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(44, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(44, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(44, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(44, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(45, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(45, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(45, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(45, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(45, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(45, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(45, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(46, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(46, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(46, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(46, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(46, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(46, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(46, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(47, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(47, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(47, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(47, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(47, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(47, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(47, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(48, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(48, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(48, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(48, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(48, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(48, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(48, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(49, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(49, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(49, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(49, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(49, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(49, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(49, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(50, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(50, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(50, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(50, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(50, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(50, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(50, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(51, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(51, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(51, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(51, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(51, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(51, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(51, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(52, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(52, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(52, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(52, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(52, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(52, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(52, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[316], y[317], y[318], y[319]))
            {
                output.second.emplace_back(53, 79);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[320], y[321], y[322], y[323]))
            {
                output.second.emplace_back(53, 80);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[324], y[325], y[326], y[327]))
            {
                output.second.emplace_back(53, 81);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[328], y[329], y[330], y[331]))
            {
                output.second.emplace_back(53, 82);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[332], y[333], y[334], y[335]))
            {
                output.second.emplace_back(53, 83);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[336], y[337], y[338], y[339]))
            {
                output.second.emplace_back(53, 84);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[340], y[341], y[342], y[343]))
            {
                output.second.emplace_back(53, 85);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(26, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(26, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(26, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(26, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(26, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(26, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(27, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(27, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(27, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(27, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(27, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(27, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(28, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(28, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(28, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(28, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(28, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(28, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(29, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(29, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(29, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(29, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(29, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(29, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(30, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(30, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(30, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(30, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(30, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(30, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(31, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(31, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(31, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(31, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(31, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(31, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(32, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(32, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(32, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(32, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(32, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(32, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(33, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(33, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(33, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(33, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(33, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(33, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(34, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(34, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(34, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(34, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(34, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(34, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(35, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(35, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(35, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(35, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(35, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(35, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(36, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(36, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(36, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(36, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(36, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(36, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(37, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(37, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(37, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(37, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(37, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(37, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(38, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(38, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(38, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(38, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(38, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(38, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(39, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(39, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(39, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(39, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(39, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(39, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(40, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(40, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(40, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(40, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(40, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(40, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(41, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(41, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(41, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(41, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(41, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(41, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(42, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(42, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(42, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(42, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(42, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(42, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(43, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(43, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(43, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(43, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(43, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(43, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(44, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(44, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(44, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(44, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(44, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(44, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(45, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(45, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(45, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(45, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(45, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(45, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(46, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(46, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(46, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(46, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(46, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(46, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(47, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(47, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(47, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(47, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(47, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(47, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(48, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(48, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(48, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(48, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(48, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(48, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(49, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(49, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(49, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(49, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(49, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(49, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(50, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(50, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(50, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(50, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(50, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(50, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(51, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(51, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(51, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(51, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(51, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(51, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(52, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(52, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(52, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(52, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(52, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(52, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[344], y[345], y[346], y[347]))
            {
                output.second.emplace_back(53, 86);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[348], y[349], y[350], y[351]))
            {
                output.second.emplace_back(53, 87);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[352], y[353], y[354], y[355]))
            {
                output.second.emplace_back(53, 88);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[356], y[357], y[358], y[359]))
            {
                output.second.emplace_back(53, 89);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[360], y[361], y[362], y[363]))
            {
                output.second.emplace_back(53, 90);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[364], y[365], y[366], y[367]))
            {
                output.second.emplace_back(53, 91);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(26, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(26, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(27, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(27, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(28, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(28, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(29, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(29, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(30, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(30, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(31, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(31, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(32, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(32, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(33, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(33, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(34, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(34, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(35, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(35, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(36, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(36, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(37, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(37, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(38, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(38, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(39, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(39, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(40, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(40, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(41, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(41, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(42, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(42, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(43, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(43, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(44, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(44, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(45, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(45, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(46, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(46, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(47, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(47, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(48, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(48, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(49, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(49, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(50, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(50, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(51, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(51, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(52, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(52, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(53, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(53, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(26, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(26, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(26, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(26, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(27, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(27, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(27, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(27, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(28, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(28, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(28, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(28, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(29, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(29, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(29, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(29, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(30, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(30, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(30, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(30, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(31, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(31, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(31, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(31, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(32, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(32, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(32, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(32, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(33, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(33, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(33, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(33, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(34, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(34, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(34, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(34, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(35, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(35, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(35, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(35, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(36, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(36, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(36, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(36, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(37, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(37, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(37, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(37, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(38, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(38, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(38, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(38, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(39, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(39, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(39, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(39, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(40, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(40, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(40, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(40, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(41, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(41, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(41, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(41, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(42, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(42, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(42, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(42, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(43, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(43, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(43, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(43, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(44, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(44, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(44, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(44, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(45, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(45, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(45, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(45, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(46, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(46, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(46, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(46, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(47, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(47, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(47, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(47, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(48, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(48, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(48, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(48, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(49, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(49, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(49, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(49, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(50, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(50, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(50, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(50, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(51, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(51, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(51, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(51, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(52, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(52, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(52, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(52, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(53, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(53, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(53, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(53, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(26, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(26, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(26, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(26, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(26, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(26, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(27, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(27, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(27, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(27, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(27, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(27, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(28, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(28, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(28, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(28, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(28, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(28, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(29, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(29, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(29, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(29, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(29, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(29, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(30, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(30, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(30, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(30, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(30, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(30, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(31, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(31, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(31, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(31, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(31, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(31, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(32, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(32, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(32, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(32, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(32, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(32, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(33, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(33, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(33, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(33, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(33, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(33, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(34, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(34, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(34, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(34, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(34, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(34, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(35, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(35, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(35, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(35, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(35, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(35, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(36, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(36, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(36, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(36, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(36, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(36, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(37, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(37, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(37, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(37, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(37, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(37, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(38, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(38, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(38, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(38, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(38, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(38, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(39, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(39, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(39, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(39, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(39, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(39, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(40, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(40, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(40, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(40, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(40, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(40, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(41, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(41, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(41, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(41, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(41, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(41, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(42, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(42, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(42, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(42, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(42, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(42, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(43, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(43, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(43, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(43, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(43, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(43, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(44, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(44, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(44, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(44, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(44, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(44, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(45, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(45, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(45, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(45, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(45, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(45, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(46, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(46, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(46, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(46, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(46, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(46, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(47, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(47, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(47, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(47, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(47, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(47, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(48, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(48, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(48, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(48, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(48, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(48, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(49, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(49, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(49, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(49, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(49, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(49, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(50, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(50, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(50, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(50, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(50, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(50, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(51, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(51, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(51, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(51, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(51, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(51, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(52, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(52, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(52, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(52, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(52, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(52, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(53, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(53, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(53, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(53, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(53, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(53, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(26, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(26, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(26, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(26, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(26, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(26, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(27, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(27, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(27, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(27, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(27, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(27, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(28, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(28, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(28, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(28, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(28, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(28, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(29, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(29, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(29, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(29, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(29, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[116], y[117], y[118], y[119], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(29, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(30, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(30, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(30, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(30, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(30, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[120], y[121], y[122], y[123], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(30, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(31, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(31, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(31, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(31, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(31, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[124], y[125], y[126], y[127], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(31, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(32, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(32, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(32, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(32, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(32, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[128], y[129], y[130], y[131], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(32, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(33, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(33, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(33, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(33, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(33, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[132], y[133], y[134], y[135], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(33, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(34, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(34, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(34, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(34, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(34, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[136], y[137], y[138], y[139], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(34, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(35, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(35, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(35, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(35, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(35, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[140], y[141], y[142], y[143], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(35, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(36, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(36, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(36, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(36, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(36, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[144], y[145], y[146], y[147], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(36, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(37, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(37, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(37, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(37, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(37, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[148], y[149], y[150], y[151], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(37, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(38, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(38, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(38, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(38, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(38, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[152], y[153], y[154], y[155], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(38, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(39, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(39, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(39, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(39, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(39, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[156], y[157], y[158], y[159], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(39, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(40, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(40, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(40, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(40, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(40, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[160], y[161], y[162], y[163], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(40, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(41, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(41, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(41, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(41, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(41, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[164], y[165], y[166], y[167], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(41, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(42, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(42, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(42, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(42, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(42, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[168], y[169], y[170], y[171], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(42, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(43, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(43, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(43, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(43, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(43, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[172], y[173], y[174], y[175], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(43, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(44, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(44, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(44, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(44, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(44, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[176], y[177], y[178], y[179], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(44, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(45, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(45, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(45, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(45, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(45, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[180], y[181], y[182], y[183], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(45, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(46, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(46, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(46, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(46, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(46, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[184], y[185], y[186], y[187], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(46, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(47, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(47, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(47, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(47, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(47, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[188], y[189], y[190], y[191], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(47, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(48, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(48, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(48, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(48, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(48, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[192], y[193], y[194], y[195], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(48, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(49, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(49, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(49, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(49, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(49, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[196], y[197], y[198], y[199], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(49, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(50, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(50, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(50, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(50, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(50, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[200], y[201], y[202], y[203], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(50, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(51, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(51, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(51, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(51, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(51, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[204], y[205], y[206], y[207], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(51, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(52, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(52, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(52, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(52, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(52, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[208], y[209], y[210], y[211], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(52, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(53, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(53, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(53, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(53, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(53, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[212], y[213], y[214], y[215], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(53, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(54, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(54, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(54, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(54, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(54, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(54, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(54, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(54, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(55, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(55, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(55, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(55, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(55, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(55, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(55, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(55, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(56, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(56, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(56, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(56, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(56, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(56, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(56, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(56, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[260], y[261], y[262], y[263]))
            {
                output.second.emplace_back(57, 65);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[264], y[265], y[266], y[267]))
            {
                output.second.emplace_back(57, 66);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[268], y[269], y[270], y[271]))
            {
                output.second.emplace_back(57, 67);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[272], y[273], y[274], y[275]))
            {
                output.second.emplace_back(57, 68);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[276], y[277], y[278], y[279]))
            {
                output.second.emplace_back(57, 69);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[280], y[281], y[282], y[283]))
            {
                output.second.emplace_back(57, 70);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[284], y[285], y[286], y[287]))
            {
                output.second.emplace_back(57, 71);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[288], y[289], y[290], y[291]))
            {
                output.second.emplace_back(57, 72);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(54, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(54, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(55, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(55, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(56, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(56, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[368], y[369], y[370], y[371]))
            {
                output.second.emplace_back(57, 92);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[372], y[373], y[374], y[375]))
            {
                output.second.emplace_back(57, 93);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(54, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(54, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(54, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(54, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(55, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(55, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(55, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(55, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(56, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(56, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(56, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(56, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(57, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(57, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(57, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(57, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(54, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(54, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(54, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(54, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(54, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(54, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(55, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(55, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(55, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(55, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(55, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(55, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(56, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(56, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(56, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(56, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(56, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(56, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(57, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(57, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(57, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(57, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(57, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(57, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(54, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(54, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(54, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(54, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(54, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[216], y[217], y[218], y[219], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(54, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(55, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(55, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(55, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(55, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(55, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[220], y[221], y[222], y[223], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(55, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(56, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(56, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(56, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(56, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(56, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[224], y[225], y[226], y[227], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(56, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(57, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(57, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(57, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(57, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(57, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[228], y[229], y[230], y[231], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(57, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(58, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(58, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(58, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(58, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(59, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(59, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(59, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(59, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(60, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(60, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(60, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(60, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(61, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(61, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(61, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(61, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(62, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(62, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(62, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(62, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(63, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(63, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(63, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(63, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[376], y[377], y[378], y[379]))
            {
                output.second.emplace_back(64, 94);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[380], y[381], y[382], y[383]))
            {
                output.second.emplace_back(64, 95);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[384], y[385], y[386], y[387]))
            {
                output.second.emplace_back(64, 96);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[388], y[389], y[390], y[391]))
            {
                output.second.emplace_back(64, 97);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(58, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(58, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(58, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(58, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(58, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(58, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(59, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(59, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(59, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(59, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(59, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(59, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(60, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(60, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(60, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(60, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(60, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(60, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(61, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(61, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(61, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(61, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(61, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(61, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(62, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(62, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(62, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(62, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(62, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(62, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(63, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(63, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(63, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(63, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(63, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(63, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[392], y[393], y[394], y[395]))
            {
                output.second.emplace_back(64, 98);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[396], y[397], y[398], y[399]))
            {
                output.second.emplace_back(64, 99);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[400], y[401], y[402], y[403]))
            {
                output.second.emplace_back(64, 100);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[404], y[405], y[406], y[407]))
            {
                output.second.emplace_back(64, 101);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[408], y[409], y[410], y[411]))
            {
                output.second.emplace_back(64, 102);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[412], y[413], y[414], y[415]))
            {
                output.second.emplace_back(64, 103);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(58, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(58, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(58, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(58, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(58, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(58, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(59, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(59, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(59, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(59, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(59, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(59, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(60, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(60, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(60, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(60, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(60, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(60, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(61, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(61, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(61, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(61, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(61, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(61, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(62, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(62, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(62, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(62, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(62, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(62, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(63, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(63, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(63, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(63, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(63, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(63, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[416], y[417], y[418], y[419]))
            {
                output.second.emplace_back(64, 104);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[420], y[421], y[422], y[423]))
            {
                output.second.emplace_back(64, 105);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[424], y[425], y[426], y[427]))
            {
                output.second.emplace_back(64, 106);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[428], y[429], y[430], y[431]))
            {
                output.second.emplace_back(64, 107);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[432], y[433], y[434], y[435]))
            {
                output.second.emplace_back(64, 108);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[436], y[437], y[438], y[439]))
            {
                output.second.emplace_back(64, 109);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[232], y[233], y[234], y[235], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(58, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(59, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(60, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(61, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[248], y[249], y[250], y[251], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(62, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[252], y[253], y[254], y[255], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(63, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(64, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[260], y[261], y[262], y[263], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(65, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[264], y[265], y[266], y[267], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(66, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[268], y[269], y[270], y[271], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(67, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[272], y[273], y[274], y[275], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(68, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[276], y[277], y[278], y[279], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(69, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[280], y[281], y[282], y[283], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(70, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[284], y[285], y[286], y[287], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(71, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[288], y[289], y[290], y[291], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(72, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[292], y[293], y[294], y[295], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(73, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[296], y[297], y[298], y[299], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(74, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[300], y[301], y[302], y[303], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(75, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[304], y[305], y[306], y[307], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(76, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[308], y[309], y[310], y[311], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(77, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[312], y[313], y[314], y[315], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(78, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[316], y[317], y[318], y[319], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(79, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[320], y[321], y[322], y[323], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(80, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[324], y[325], y[326], y[327], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(81, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[328], y[329], y[330], y[331], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(82, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[332], y[333], y[334], y[335], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(83, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[336], y[337], y[338], y[339], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(84, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[340], y[341], y[342], y[343], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(85, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[344], y[345], y[346], y[347], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(86, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[348], y[349], y[350], y[351], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(87, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[352], y[353], y[354], y[355], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(88, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[356], y[357], y[358], y[359], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(89, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[360], y[361], y[362], y[363], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(90, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[364], y[365], y[366], y[367], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(91, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[368], y[369], y[370], y[371], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(92, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[372], y[373], y[374], y[375], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(93, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[376], y[377], y[378], y[379], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(94, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[380], y[381], y[382], y[383], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(95, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[384], y[385], y[386], y[387], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(96, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[388], y[389], y[390], y[391], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(97, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[392], y[393], y[394], y[395], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(98, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[396], y[397], y[398], y[399], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(99, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[400], y[401], y[402], y[403], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(100, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[404], y[405], y[406], y[407], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(101, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[408], y[409], y[410], y[411], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(102, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[412], y[413], y[414], y[415], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(103, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[416], y[417], y[418], y[419], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(104, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[420], y[421], y[422], y[423], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(105, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[424], y[425], y[426], y[427], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(106, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[428], y[429], y[430], y[431], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(107, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[432], y[433], y[434], y[435], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(108, 110);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[436], y[437], y[438], y[439], y[440], y[441], y[442], y[443]))
            {
                output.second.emplace_back(109, 110);
            }

            return output;
        }

        template <std::size_t rake>
        inline static bool fkcc(
            const vamp::collision::Environment<FloatVector<rake>> &environment,
            const ConfigurationBlock<rake> &x) noexcept
        {
            std::array<FloatVector<rake, 1>, 50> v;
            std::array<FloatVector<rake, 1>, 504> y;

            v[0] = 0.37743 + x[0];
            y[82] = 0.15 + v[0];
            y[86] = 0.15 + v[0];
            y[90] = 0.3 + v[0];
            y[94] = 0.45 + v[0];
            y[98] = 0.45 + v[0];
            y[102] = 0.3 + v[0];
            y[106] = 0.663001417713939 + v[0];
            y[110] = 0.661001417713939 + v[0];
            y[114] = 0.661001417713939 + v[0];
            y[118] = 0.661001417713939 + v[0];
            y[122] = 0.661001417713939 + v[0];
            y[126] = 0.661001417713939 + v[0];
            y[130] = 0.633001417713939 + v[0];
            y[134] = 0.633001417713939 + v[0];
            y[138] = 0.633001417713939 + v[0];
            y[142] = 0.633001417713939 + v[0];
            y[146] = 0.688001417713939 + v[0];
            y[150] = 0.688001417713939 + v[0];
            y[154] = 0.688001417713939 + v[0];
            y[158] = 0.688001417713939 + v[0];
            y[162] = 0.678001417713939 + v[0];
            y[166] = 0.660501417713939 + v[0];
            y[170] = 0.643001417713939 + v[0];
            y[174] = 0.633001417713939 + v[0];
            y[178] = 0.633001417713939 + v[0];
            y[182] = 0.633001417713939 + v[0];
            y[186] = 0.633001417713939 + v[0];
            y[190] = 0.688001417713939 + v[0];
            y[194] = 0.688001417713939 + v[0];
            y[198] = 0.688001417713939 + v[0];
            y[202] = 0.688001417713939 + v[0];
            y[206] = 0.678001417713939 + v[0];
            y[210] = 0.660501417713939 + v[0];
            y[214] = 0.643001417713939 + v[0];
            y[218] = 0.34858 + v[0];
            v[1] = cos(x[1]);
            v[2] = sin(x[1]);
            v[3] = -v[2];
            y[220] = 0.03265 + 0.025 * v[1] + -0.015 * v[3];
            y[221] = 0.025 * v[2] + -0.015 * v[1];
            y[222] = 0.035 + y[218];
            y[224] = 0.03265 + 0.05 * v[1] + -0.03 * v[3];
            y[225] = 0.05 * v[2] + -0.03 * v[1];
            y[226] = 0.06 + y[218];
            y[228] = 0.03265 + 0.12 * v[1] + -0.03 * v[3];
            y[229] = 0.12 * v[2] + -0.03 * v[1];
            y[230] = 0.06 + y[218];
            v[4] = cos(x[2]);
            v[5] = v[1] * v[4];
            v[6] = sin(x[2]);
            v[7] = v[1] * v[6];
            v[8] = 0.03265 + 0.117 * v[1];
            y[232] = 0.025 * v[5] + 0.04 * v[3] + 0.025 * v[7] + v[8];
            v[9] = v[2] * v[4];
            v[10] = v[2] * v[6];
            v[11] = 0.117 * v[2];
            y[233] = 0.025 * v[9] + 0.04 * v[1] + 0.025 * v[10] + v[11];
            v[6] = -v[6];
            v[12] = 0.0599999999999999 + y[218];
            y[234] = 0.025 * v[6] + 0.025 * v[4] + v[12];
            y[236] = -0.025 * v[5] + 0.04 * v[3] + -0.025 * v[7] + v[8];
            y[237] = -0.025 * v[9] + 0.04 * v[1] + -0.025 * v[10] + v[11];
            y[238] = -0.025 * v[6] + -0.025 * v[4] + v[12];
            y[240] = 0.025 * v[5] + 0.04 * v[3] + -0.025 * v[7] + v[8];
            y[241] = 0.025 * v[9] + 0.04 * v[1] + -0.025 * v[10] + v[11];
            y[242] = 0.025 * v[6] + -0.025 * v[4] + v[12];
            y[244] = -0.025 * v[5] + 0.04 * v[3] + 0.025 * v[7] + v[8];
            y[245] = -0.025 * v[9] + 0.04 * v[1] + 0.025 * v[10] + v[11];
            y[246] = -0.025 * v[6] + 0.025 * v[4] + v[12];
            y[248] = 0.08 * v[5] + v[8];
            y[249] = 0.08 * v[9] + v[11];
            y[250] = 0.08 * v[6] + v[12];
            y[252] = 0.11 * v[5] + v[8];
            y[253] = 0.11 * v[9] + v[11];
            y[254] = 0.11 * v[6] + v[12];
            y[256] = 0.14 * v[5] + v[8];
            y[257] = 0.14 * v[9] + v[11];
            y[258] = 0.14 * v[6] + v[12];
            v[13] = v[8] + 0.219 * v[5];
            y[260] = -0.02 * v[5] + v[13];
            v[14] = v[11] + 0.219 * v[9];
            y[261] = -0.02 * v[9] + v[14];
            v[15] = v[12] + 0.219 * v[6];
            y[262] = -0.02 * v[6] + v[15];
            y[264] = 0.03 * v[5] + v[13];
            y[265] = 0.03 * v[9] + v[14];
            y[266] = 0.03 * v[6] + v[15];
            y[268] = 0.08 * v[5] + v[13];
            y[269] = 0.08 * v[9] + v[14];
            y[270] = 0.08 * v[6] + v[15];
            v[16] = cos(x[3]);
            v[17] = sin(x[3]);
            v[18] = v[3] * v[16] + v[7] * v[17];
            v[19] = -v[17];
            v[20] = v[3] * v[19] + v[7] * v[16];
            y[272] = 0.11 * v[5] + -0.045 * v[18] + 0.02 * v[20] + v[13];
            v[21] = v[1] * v[16] + v[10] * v[17];
            v[19] = v[1] * v[19] + v[10] * v[16];
            y[273] = 0.11 * v[9] + -0.045 * v[21] + 0.02 * v[19] + v[14];
            v[17] = v[4] * v[17];
            v[16] = v[4] * v[16];
            y[274] = 0.11 * v[6] + -0.045 * v[17] + 0.02 * v[16] + v[15];
            y[276] = 0.11 * v[5] + -0.045 * v[18] + -0.02 * v[20] + v[13];
            y[277] = 0.11 * v[9] + -0.045 * v[21] + -0.02 * v[19] + v[14];
            y[278] = 0.11 * v[6] + -0.045 * v[17] + -0.02 * v[16] + v[15];
            y[280] = 0.155 * v[5] + -0.045 * v[18] + 0.02 * v[20] + v[13];
            y[281] = 0.155 * v[9] + -0.045 * v[21] + 0.02 * v[19] + v[14];
            y[282] = 0.155 * v[6] + -0.045 * v[17] + 0.02 * v[16] + v[15];
            y[284] = 0.155 * v[5] + -0.045 * v[18] + -0.02 * v[20] + v[13];
            y[285] = 0.155 * v[9] + -0.045 * v[21] + -0.02 * v[19] + v[14];
            y[286] = 0.155 * v[6] + -0.045 * v[17] + -0.02 * v[16] + v[15];
            y[288] = 0.13 * v[5] + v[13];
            y[289] = 0.13 * v[9] + v[14];
            y[290] = 0.13 * v[6] + v[15];
            v[22] = cos(x[4]);
            v[23] = sin(x[4]);
            v[24] = -v[23];
            v[25] = v[5] * v[22] + v[20] * v[24];
            v[26] = v[5] * v[23] + v[20] * v[22];
            v[27] = v[13] + 0.133 * v[5];
            y[292] = 0.02 * v[25] + 0.045 * v[18] + 0.02 * v[26] + v[27];
            v[28] = v[9] * v[22] + v[19] * v[24];
            v[29] = v[9] * v[23] + v[19] * v[22];
            v[30] = v[14] + 0.133 * v[9];
            y[293] = 0.02 * v[28] + 0.045 * v[21] + 0.02 * v[29] + v[30];
            v[24] = v[6] * v[22] + v[16] * v[24];
            v[23] = v[6] * v[23] + v[16] * v[22];
            v[22] = v[15] + 0.133 * v[6];
            y[294] = 0.02 * v[24] + 0.045 * v[17] + 0.02 * v[23] + v[22];
            y[296] = 0.02 * v[25] + 0.045 * v[18] + -0.02 * v[26] + v[27];
            y[297] = 0.02 * v[28] + 0.045 * v[21] + -0.02 * v[29] + v[30];
            y[298] = 0.02 * v[24] + 0.045 * v[17] + -0.02 * v[23] + v[22];
            y[300] = -0.02 * v[25] + 0.045 * v[18] + 0.02 * v[26] + v[27];
            y[301] = -0.02 * v[28] + 0.045 * v[21] + 0.02 * v[29] + v[30];
            y[302] = -0.02 * v[24] + 0.045 * v[17] + 0.02 * v[23] + v[22];
            y[304] = -0.02 * v[25] + 0.045 * v[18] + -0.02 * v[26] + v[27];
            y[305] = -0.02 * v[28] + 0.045 * v[21] + -0.02 * v[29] + v[30];
            y[306] = -0.02 * v[24] + 0.045 * v[17] + -0.02 * v[23] + v[22];
            y[308] = 0.08 * v[25] + v[27];
            y[309] = 0.08 * v[28] + v[30];
            y[310] = 0.08 * v[24] + v[22];
            y[312] = 0.14 * v[25] + v[27];
            y[313] = 0.14 * v[28] + v[30];
            y[314] = 0.14 * v[24] + v[22];
            y[316] = v[27] + 0.197 * v[25];
            y[317] = v[30] + 0.197 * v[28];
            y[318] = v[22] + 0.197 * v[24];
            v[31] = cos(x[5]);
            v[32] = sin(x[5]);
            v[33] = v[18] * v[31] + v[26] * v[32];
            v[34] = -v[32];
            v[35] = v[18] * v[34] + v[26] * v[31];
            y[320] = 0.05 * v[25] + -0.06 * v[33] + 0.02 * v[35] + y[316];
            v[36] = v[21] * v[31] + v[29] * v[32];
            v[37] = v[21] * v[34] + v[29] * v[31];
            y[321] = 0.05 * v[28] + -0.06 * v[36] + 0.02 * v[37] + y[317];
            v[32] = v[17] * v[31] + v[23] * v[32];
            v[34] = v[17] * v[34] + v[23] * v[31];
            y[322] = 0.05 * v[24] + -0.06 * v[32] + 0.02 * v[34] + y[318];
            y[324] = 0.05 * v[25] + -0.06 * v[33] + -0.02 * v[35] + y[316];
            y[325] = 0.05 * v[28] + -0.06 * v[36] + -0.02 * v[37] + y[317];
            y[326] = 0.05 * v[24] + -0.06 * v[32] + -0.02 * v[34] + y[318];
            y[328] = 0.1 * v[25] + -0.06 * v[33] + 0.02 * v[35] + y[316];
            y[329] = 0.1 * v[28] + -0.06 * v[36] + 0.02 * v[37] + y[317];
            y[330] = 0.1 * v[24] + -0.06 * v[32] + 0.02 * v[34] + y[318];
            y[332] = 0.1 * v[25] + -0.06 * v[33] + -0.02 * v[35] + y[316];
            y[333] = 0.1 * v[28] + -0.06 * v[36] + -0.02 * v[37] + y[317];
            y[334] = 0.1 * v[24] + -0.06 * v[32] + -0.02 * v[34] + y[318];
            y[336] = 0.15 * v[25] + -0.06 * v[33] + 0.02 * v[35] + y[316];
            y[337] = 0.15 * v[28] + -0.06 * v[36] + 0.02 * v[37] + y[317];
            y[338] = 0.15 * v[24] + -0.06 * v[32] + 0.02 * v[34] + y[318];
            y[340] = 0.15 * v[25] + -0.06 * v[33] + -0.02 * v[35] + y[316];
            y[341] = 0.15 * v[28] + -0.06 * v[36] + -0.02 * v[37] + y[317];
            y[342] = 0.15 * v[24] + -0.06 * v[32] + -0.02 * v[34] + y[318];
            y[344] = y[316] + 0.1245 * v[25];
            y[345] = y[317] + 0.1245 * v[28];
            y[346] = y[318] + 0.1245 * v[24];
            v[31] = cos(x[6]);
            v[38] = sin(x[6]);
            v[39] = -v[38];
            v[40] = v[25] * v[31] + v[35] * v[39];
            y[348] = 0.06 * v[40] + y[344];
            v[41] = v[28] * v[31] + v[37] * v[39];
            y[349] = 0.06 * v[41] + y[345];
            v[39] = v[24] * v[31] + v[34] * v[39];
            y[350] = 0.06 * v[39] + y[346];
            v[42] = v[25] * v[38] + v[35] * v[31];
            y[352] = 0.02 * v[40] + 0.045 * v[33] + 0.02 * v[42] + y[344];
            v[43] = v[28] * v[38] + v[37] * v[31];
            y[353] = 0.02 * v[41] + 0.045 * v[36] + 0.02 * v[43] + y[345];
            v[38] = v[24] * v[38] + v[34] * v[31];
            y[354] = 0.02 * v[39] + 0.045 * v[32] + 0.02 * v[38] + y[346];
            y[356] = 0.02 * v[40] + 0.045 * v[33] + -0.02 * v[42] + y[344];
            y[357] = 0.02 * v[41] + 0.045 * v[36] + -0.02 * v[43] + y[345];
            y[358] = 0.02 * v[39] + 0.045 * v[32] + -0.02 * v[38] + y[346];
            y[360] = -0.02 * v[40] + 0.045 * v[33] + 0.02 * v[42] + y[344];
            y[361] = -0.02 * v[41] + 0.045 * v[36] + 0.02 * v[43] + y[345];
            y[362] = -0.02 * v[39] + 0.045 * v[32] + 0.02 * v[38] + y[346];
            y[364] = -0.02 * v[40] + 0.045 * v[33] + -0.02 * v[42] + y[344];
            y[365] = -0.02 * v[41] + 0.045 * v[36] + -0.02 * v[43] + y[345];
            y[366] = -0.02 * v[39] + 0.045 * v[32] + -0.02 * v[38] + y[346];
            y[372] = y[344] + 0.1385 * v[40];
            y[368] = -0.03 * v[40] + y[372];
            y[373] = y[345] + 0.1385 * v[41];
            y[369] = -0.03 * v[41] + y[373];
            y[374] = y[346] + 0.1385 * v[39];
            y[370] = -0.03 * v[39] + y[374];
            v[31] = cos(x[7]);
            v[44] = sin(x[7]);
            v[45] = v[33] * v[31] + v[42] * v[44];
            y[376] = 0.09645 * v[40] + 0.02 * v[45] + y[372];
            v[46] = v[36] * v[31] + v[43] * v[44];
            y[377] = 0.09645 * v[41] + 0.02 * v[46] + y[373];
            v[47] = v[32] * v[31] + v[38] * v[44];
            y[378] = 0.09645 * v[39] + 0.02 * v[47] + y[374];
            y[380] = 0.09645 * v[40] + -0.02 * v[45] + y[372];
            y[381] = 0.09645 * v[41] + -0.02 * v[46] + y[373];
            y[382] = 0.09645 * v[39] + -0.02 * v[47] + y[374];
            y[384] = 0.06645 * v[40] + 0.02 * v[45] + y[372];
            y[385] = 0.06645 * v[41] + 0.02 * v[46] + y[373];
            y[386] = 0.06645 * v[39] + 0.02 * v[47] + y[374];
            y[388] = 0.06645 * v[40] + -0.02 * v[45] + y[372];
            y[389] = 0.06645 * v[41] + -0.02 * v[46] + y[373];
            y[390] = 0.06645 * v[39] + -0.02 * v[47] + y[374];
            v[44] = -v[44];
            v[48] = v[33] * v[44] + v[42] * v[31];
            y[392] = 0.18345 * v[40] + -0.056925 * v[45] + -0.005 * v[48] + y[372];
            v[49] = v[36] * v[44] + v[43] * v[31];
            y[393] = 0.18345 * v[41] + -0.056925 * v[46] + -0.005 * v[49] + y[373];
            v[44] = v[32] * v[44] + v[38] * v[31];
            y[394] = 0.18345 * v[39] + -0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[396] = 0.18345 * v[40] + -0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[397] = 0.18345 * v[41] + -0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[398] = 0.18345 * v[39] + -0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[400] = 0.16645 * v[40] + -0.056925 * v[45] + -0.005 * v[48] + y[372];
            y[401] = 0.16645 * v[41] + -0.056925 * v[46] + -0.005 * v[49] + y[373];
            y[402] = 0.16645 * v[39] + -0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[404] = 0.16645 * v[40] + -0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[405] = 0.16645 * v[41] + -0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[406] = 0.16645 * v[39] + -0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[408] = 0.14945 * v[40] + -0.056925 * v[45] + -0.005 * v[48] + y[372];
            y[409] = 0.14945 * v[41] + -0.056925 * v[46] + -0.005 * v[49] + y[373];
            y[410] = 0.14945 * v[39] + -0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[412] = 0.14945 * v[40] + -0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[413] = 0.14945 * v[41] + -0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[414] = 0.14945 * v[39] + -0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[416] = 0.18345 * v[40] + 0.056925 * v[45] + -0.005 * v[48] + y[372];
            y[417] = 0.18345 * v[41] + 0.056925 * v[46] + -0.005 * v[49] + y[373];
            y[418] = 0.18345 * v[39] + 0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[420] = 0.18345 * v[40] + 0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[421] = 0.18345 * v[41] + 0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[422] = 0.18345 * v[39] + 0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[424] = 0.16645 * v[40] + 0.056925 * v[45] + -0.005 * v[48] + y[372];
            y[425] = 0.16645 * v[41] + 0.056925 * v[46] + -0.005 * v[49] + y[373];
            y[426] = 0.16645 * v[39] + 0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[428] = 0.16645 * v[40] + 0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[429] = 0.16645 * v[41] + 0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[430] = 0.16645 * v[39] + 0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[432] = 0.14945 * v[40] + 0.056925 * v[45] + -0.005 * v[48] + y[372];
            y[433] = 0.14945 * v[41] + 0.056925 * v[46] + -0.005 * v[49] + y[373];
            y[434] = 0.14945 * v[39] + 0.056925 * v[47] + -0.005 * v[44] + y[374];
            y[436] = 0.14945 * v[40] + 0.056925 * v[45] + 0.005 * v[48] + y[372];
            y[437] = 0.14945 * v[41] + 0.056925 * v[46] + 0.005 * v[49] + y[373];
            y[438] = 0.14945 * v[39] + 0.056925 * v[47] + 0.005 * v[44] + y[374];
            y[442] = 0.24 + v[0];
            y[454] = 0.300000011920929 + v[0];
            y[458] = 0.662302553653717 + v[0];
            y[460] = 0.03265 + 0.0599999986588955 * v[1] + -0.0149999996647239 * v[3];
            y[461] = 0.0599999986588955 * v[2] + -0.0149999996647239 * v[1];
            y[462] = 0.0299999993294477 + y[218];
            y[464] =
                0.0631452798843384 * v[5] + 0.018631448969245 * v[3] + 1.73472347597681e-18 * v[7] + v[8];
            y[465] =
                0.0631452798843384 * v[9] + 0.018631448969245 * v[1] + 1.73472347597681e-18 * v[10] + v[11];
            y[466] = 0.0631452798843384 * v[6] + 1.73472347597681e-18 * v[4] + v[12];
            y[468] = 0.056335523724556 * v[5] + -0.0196291357278824 * v[18] + -1.73472347597681e-18 * v[20] +
                     v[13];
            y[469] = 0.056335523724556 * v[9] + -0.0196291357278824 * v[21] + -1.73472347597681e-18 * v[19] +
                     v[14];
            y[470] = 0.056335523724556 * v[6] + -0.0196291357278824 * v[17] + -1.73472347597681e-18 * v[16] +
                     v[15];
            y[472] = 0.0710262209177017 * v[25] + 0.0193988755345345 * v[18] + 1.73472347597681e-18 * v[26] +
                     v[27];
            y[473] = 0.0710262209177017 * v[28] + 0.0193988755345345 * v[21] + 1.73472347597681e-18 * v[29] +
                     v[30];
            y[474] = 0.0710262209177017 * v[24] + 0.0193988755345345 * v[17] + 1.73472347597681e-18 * v[23] +
                     v[22];
            y[476] = 0.0643894299864769 * v[25] + -0.0257557742297649 * v[33] +
                     -1.73472347597681e-18 * v[35] + y[316];
            y[477] = 0.0643894299864769 * v[28] + -0.0257557742297649 * v[36] +
                     -1.73472347597681e-18 * v[37] + y[317];
            y[478] = 0.0643894299864769 * v[24] + -0.0257557742297649 * v[32] +
                     -1.73472347597681e-18 * v[34] + y[318];
            y[480] = 0.0294021144509315 * v[40] + 0.0172113105654716 * v[33] + 4.33680868994202e-18 * v[42] +
                     y[344];
            y[481] = 0.0294021144509315 * v[41] + 0.0172113105654716 * v[36] + 4.33680868994202e-18 * v[43] +
                     y[345];
            y[482] = 0.0294021144509315 * v[39] + 0.0172113105654716 * v[32] + 4.33680868994202e-18 * v[38] +
                     y[346];
            y[484] = -0.0149999996647239 * v[40] + y[372];
            y[485] = -0.0149999996647239 * v[41] + y[373];
            y[486] = -0.0149999996647239 * v[39] + y[374];
            y[488] = 0.0814500004053116 * v[40] + y[372];
            y[489] = 0.0814500004053116 * v[41] + y[373];
            y[490] = 0.0814500004053116 * v[39] + y[374];
            y[492] = 0.166449993848801 * v[40] + -0.0569249987602234 * v[45] + y[372];
            y[493] = 0.166449993848801 * v[41] + -0.0569249987602234 * v[46] + y[373];
            y[494] = 0.166449993848801 * v[39] + -0.0569249987602234 * v[47] + y[374];
            y[496] = 0.166449993848801 * v[40] + 0.0569249987602234 * v[45] + y[372];
            y[497] = 0.166449993848801 * v[41] + 0.0569249987602234 * v[46] + y[373];
            y[498] = 0.166449993848801 * v[39] + 0.0569249987602234 * v[47] + y[374];
            y[502] = 0.239999994635582 + v[0];
            // dependent variables without operations
            y[0] = -0.12;
            y[1] = 0.;
            y[2] = 0.182;
            y[3] = 0.239999994635582;
            y[4] = 0.225;
            y[5] = 0.;
            y[6] = 0.31;
            y[7] = 0.0659999996423721;
            y[8] = 0.08;
            y[9] = -0.06;
            y[10] = 0.16;
            y[11] = 0.219999998807907;
            y[12] = 0.215;
            y[13] = -0.07;
            y[14] = 0.31;
            y[15] = 0.0659999996423721;
            y[16] = 0.185;
            y[17] = -0.135;
            y[18] = 0.31;
            y[19] = 0.0659999996423721;
            y[20] = 0.13;
            y[21] = -0.185;
            y[22] = 0.31;
            y[23] = 0.0659999996423721;
            y[24] = 0.065;
            y[25] = -0.2;
            y[26] = 0.31;
            y[27] = 0.0659999996423721;
            y[28] = 0.01;
            y[29] = -0.2;
            y[30] = 0.31;
            y[31] = 0.0659999996423721;
            y[32] = 0.08;
            y[33] = 0.06;
            y[34] = 0.16;
            y[35] = 0.219999998807907;
            y[36] = 0.215;
            y[37] = 0.07;
            y[38] = 0.31;
            y[39] = 0.0659999996423721;
            y[40] = 0.185;
            y[41] = 0.135;
            y[42] = 0.31;
            y[43] = 0.0659999996423721;
            y[44] = 0.13;
            y[45] = 0.185;
            y[46] = 0.31;
            y[47] = 0.0659999996423721;
            y[48] = 0.065;
            y[49] = 0.2;
            y[50] = 0.31;
            y[51] = 0.0659999996423721;
            y[52] = 0.01;
            y[53] = 0.2;
            y[54] = 0.31;
            y[55] = 0.0659999996423721;
            y[56] = -0.186875;
            y[57] = -0.07;
            y[58] = 0.727425;
            y[59] = 0.119999997317791;
            y[60] = -0.186875;
            y[61] = 0.07;
            y[62] = 0.727425;
            y[63] = 0.119999997317791;
            y[64] = -0.186875;
            y[65] = -0.07;
            y[66] = 0.577425;
            y[67] = 0.119999997317791;
            y[68] = -0.186875;
            y[69] = 0.07;
            y[70] = 0.577425;
            y[71] = 0.119999997317791;
            y[72] = -0.186875;
            y[73] = 0.07;
            y[74] = 0.447425;
            y[75] = 0.119999997317791;
            y[76] = -0.186875;
            y[77] = -0.07;
            y[78] = 0.447425;
            y[79] = 0.119999997317791;
            y[80] = -0.186875;
            y[81] = -0.05;
            y[83] = 0.150000005960464;
            y[84] = -0.186875;
            y[85] = 0.05;
            y[87] = 0.150000005960464;
            y[88] = -0.186875;
            y[89] = 0.05;
            y[91] = 0.150000005960464;
            y[92] = -0.186875;
            y[93] = 0.05;
            y[95] = 0.150000005960464;
            y[96] = -0.186875;
            y[97] = -0.05;
            y[99] = 0.150000005960464;
            y[100] = -0.186875;
            y[101] = -0.05;
            y[103] = 0.150000005960464;
            y[104] = -0.03375;
            y[105] = 0.;
            y[107] = 0.150000005960464;
            y[108] = 0.11125;
            y[109] = 0.;
            y[111] = 0.0500000007450581;
            y[112] = 0.11125;
            y[113] = -0.0425;
            y[115] = 0.0500000007450581;
            y[116] = 0.11125;
            y[117] = 0.0425;
            y[119] = 0.0500000007450581;
            y[120] = 0.11125;
            y[121] = 0.085;
            y[123] = 0.0500000007450581;
            y[124] = 0.11125;
            y[125] = -0.085;
            y[127] = 0.0500000007450581;
            y[128] = 0.02875;
            y[129] = -0.115;
            y[131] = 0.0299999993294477;
            y[132] = 0.05425;
            y[133] = -0.115;
            y[135] = 0.0299999993294477;
            y[136] = 0.07975;
            y[137] = -0.115;
            y[139] = 0.0299999993294477;
            y[140] = 0.10525;
            y[141] = -0.115;
            y[143] = 0.0299999993294477;
            y[144] = 0.02875;
            y[145] = -0.115;
            y[147] = 0.0299999993294477;
            y[148] = 0.05425;
            y[149] = -0.115;
            y[151] = 0.0299999993294477;
            y[152] = 0.07975;
            y[153] = -0.115;
            y[155] = 0.0299999993294477;
            y[156] = 0.10525;
            y[157] = -0.115;
            y[159] = 0.0299999993294477;
            y[160] = 0.12625;
            y[161] = -0.115;
            y[163] = 0.0299999993294477;
            y[164] = 0.13425;
            y[165] = -0.115;
            y[167] = 0.0299999993294477;
            y[168] = 0.12625;
            y[169] = -0.115;
            y[171] = 0.0299999993294477;
            y[172] = 0.02875;
            y[173] = 0.115;
            y[175] = 0.0299999993294477;
            y[176] = 0.05425;
            y[177] = 0.115;
            y[179] = 0.0299999993294477;
            y[180] = 0.07975;
            y[181] = 0.115;
            y[183] = 0.0299999993294477;
            y[184] = 0.10525;
            y[185] = 0.115;
            y[187] = 0.0299999993294477;
            y[188] = 0.02875;
            y[189] = 0.115;
            y[191] = 0.0299999993294477;
            y[192] = 0.05425;
            y[193] = 0.115;
            y[195] = 0.0299999993294477;
            y[196] = 0.07975;
            y[197] = 0.115;
            y[199] = 0.0299999993294477;
            y[200] = 0.10525;
            y[201] = 0.115;
            y[203] = 0.0299999993294477;
            y[204] = 0.12625;
            y[205] = 0.115;
            y[207] = 0.0299999993294477;
            y[208] = 0.13425;
            y[209] = 0.115;
            y[211] = 0.0299999993294477;
            y[212] = 0.12625;
            y[213] = 0.115;
            y[215] = 0.0299999993294477;
            y[216] = 0.03265;
            y[217] = 0.;
            y[219] = 0.0549999997019768;
            y[223] = 0.0549999997019768;
            y[227] = 0.0549999997019768;
            y[231] = 0.0549999997019768;
            y[235] = 0.0399999991059303;
            y[239] = 0.0399999991059303;
            y[243] = 0.0399999991059303;
            y[247] = 0.0399999991059303;
            y[251] = 0.0549999997019768;
            y[255] = 0.0549999997019768;
            y[259] = 0.0549999997019768;
            y[263] = 0.0549999997019768;
            y[267] = 0.0549999997019768;
            y[271] = 0.0549999997019768;
            y[275] = 0.0299999993294477;
            y[279] = 0.0299999993294477;
            y[283] = 0.0299999993294477;
            y[287] = 0.0299999993294477;
            y[291] = 0.0549999997019768;
            y[295] = 0.0299999993294477;
            y[299] = 0.0299999993294477;
            y[303] = 0.0299999993294477;
            y[307] = 0.0299999993294477;
            y[311] = 0.0549999997019768;
            y[315] = 0.0549999997019768;
            y[319] = 0.0549999997019768;
            y[323] = 0.0299999993294477;
            y[327] = 0.0299999993294477;
            y[331] = 0.0299999993294477;
            y[335] = 0.0299999993294477;
            y[339] = 0.0299999993294477;
            y[343] = 0.0299999993294477;
            y[347] = 0.0549999997019768;
            y[351] = 0.0549999997019768;
            y[355] = 0.0299999993294477;
            y[359] = 0.0299999993294477;
            y[363] = 0.0299999993294477;
            y[367] = 0.0299999993294477;
            y[371] = 0.0549999997019768;
            y[375] = 0.0549999997019768;
            y[379] = 0.0500000007450581;
            y[383] = 0.0500000007450581;
            y[387] = 0.0500000007450581;
            y[391] = 0.0500000007450581;
            y[395] = 0.0120000001043081;
            y[399] = 0.0120000001043081;
            y[403] = 0.0120000001043081;
            y[407] = 0.0120000001043081;
            y[411] = 0.0120000001043081;
            y[415] = 0.0120000001043081;
            y[419] = 0.0120000001043081;
            y[423] = 0.0120000001043081;
            y[427] = 0.0120000001043081;
            y[431] = 0.0120000001043081;
            y[435] = 0.0120000001043081;
            y[439] = 0.0120000001043081;
            y[440] = 0.013125;
            y[441] = 0.;
            y[443] = 0.0700000002980232;
            y[444] = -0.0201118849217892;
            y[445] = -5.20417042793042e-17;
            y[446] = 0.188239961862564;
            y[447] = 0.340082824230194;
            y[448] = -0.186875000596046;
            y[449] = 2.77555756156289e-17;
            y[450] = 0.587424993515015;
            y[451] = 0.276524752378464;
            y[452] = -0.186875001490116;
            y[453] = 0.;
            y[455] = 0.308113902807236;
            y[456] = 0.0132126799225807;
            y[457] = -2.42861286636753e-17;
            y[459] = 0.196967884898186;
            y[463] = 0.123738631606102;
            y[467] = 0.134080842137337;
            y[471] = 0.133818879723549;
            y[475] = 0.12664982676506;
            y[479] = 0.124349541962147;
            y[483] = 0.0901064053177834;
            y[487] = 0.0700000002980232;
            y[491] = 0.0750000029802322;
            y[495] = 0.0297200456261635;
            y[499] = 0.0297200456261635;
            y[500] = 0.0131250014901161;
            y[501] = 0.;
            y[503] = 0.0700000002980232;

            //
            // environment vs. robot collisions
            //

            // torso_lift_link_collision_2
            if (sphere_environment_in_collision(environment, y[500], y[501], y[502], y[503]))
            {
                if (sphere_environment_in_collision(environment, y[440], y[441], y[442], y[443]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // r_gripper_finger_link
            if (sphere_environment_in_collision(environment, y[496], y[497], y[498], y[499]))
            {
                if (sphere_environment_in_collision(environment, y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[436], y[437], y[438], y[439]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // l_gripper_finger_link
            if (sphere_environment_in_collision(environment, y[492], y[493], y[494], y[495]))
            {
                if (sphere_environment_in_collision(environment, y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[412], y[413], y[414], y[415]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // gripper_link
            if (sphere_environment_in_collision(environment, y[488], y[489], y[490], y[491]))
            {
                if (sphere_environment_in_collision(environment, y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[388], y[389], y[390], y[391]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // wrist_roll_link
            if (sphere_environment_in_collision(environment, y[484], y[485], y[486], y[487]))
            {
                if (sphere_environment_in_collision(environment, y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[372], y[373], y[374], y[375]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // wrist_flex_link
            if (sphere_environment_in_collision(environment, y[480], y[481], y[482], y[483]))
            {
                if (sphere_environment_in_collision(environment, y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[364], y[365], y[366], y[367]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // forearm_roll_link
            if (sphere_environment_in_collision(environment, y[476], y[477], y[478], y[479]))
            {
                if (sphere_environment_in_collision(environment, y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[340], y[341], y[342], y[343]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // elbow_flex_link
            if (sphere_environment_in_collision(environment, y[472], y[473], y[474], y[475]))
            {
                if (sphere_environment_in_collision(environment, y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[312], y[313], y[314], y[315]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // upperarm_roll_link
            if (sphere_environment_in_collision(environment, y[468], y[469], y[470], y[471]))
            {
                if (sphere_environment_in_collision(environment, y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[288], y[289], y[290], y[291]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // shoulder_lift_link
            if (sphere_environment_in_collision(environment, y[464], y[465], y[466], y[467]))
            {
                if (sphere_environment_in_collision(environment, y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[236], y[237], y[238], y[239]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[240], y[241], y[242], y[243]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[244], y[245], y[246], y[247]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[248], y[249], y[250], y[251]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[252], y[253], y[254], y[255]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[256], y[257], y[258], y[259]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // shoulder_pan_link
            if (sphere_environment_in_collision(environment, y[460], y[461], y[462], y[463]))
            {
                if (sphere_environment_in_collision(environment, y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[228], y[229], y[230], y[231]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // head_pan_link
            if (sphere_environment_in_collision(environment, y[456], y[457], y[458], y[459]))
            {
                if (sphere_environment_in_collision(environment, y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[212], y[213], y[214], y[215]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // torso_lift_link
            if (sphere_environment_in_collision(environment, y[452], y[453], y[454], y[455]))
            {
                if (sphere_environment_in_collision(environment, y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[100], y[101], y[102], y[103]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // torso_fixed_link
            if (sphere_environment_in_collision(environment, y[448], y[449], y[450], y[451]))
            {
                if (sphere_environment_in_collision(environment, y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[60], y[61], y[62], y[63]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[64], y[65], y[66], y[67]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[76], y[77], y[78], y[79]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // base_link
            if (sphere_environment_in_collision(environment, y[444], y[445], y[446], y[447]))
            {
                if (sphere_environment_in_collision(environment, y[0], y[1], y[2], y[3]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[4], y[5], y[6], y[7]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[8], y[9], y[10], y[11]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[12], y[13], y[14], y[15]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[16], y[17], y[18], y[19]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[20], y[21], y[22], y[23]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[24], y[25], y[26], y[27]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[32], y[33], y[34], y[35]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[36], y[37], y[38], y[39]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[48], y[49], y[50], y[51]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[52], y[53], y[54], y[55]))
                {
                    return false;
                }
            }

            //
            // robot self-collisions
            //

            // base_link vs. elbow_flex_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[444], y[445], y[446], y[447], y[472], y[473], y[474], y[475]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }
            }

            // base_link vs. forearm_roll_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[444], y[445], y[446], y[447], y[476], y[477], y[478], y[479]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }
            }

            // base_link vs. wrist_flex_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[444], y[445], y[446], y[447], y[480], y[481], y[482], y[483]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }
            }

            // base_link vs. wrist_roll_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[444], y[445], y[446], y[447], y[484], y[485], y[486], y[487]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }
            }

            // base_link vs. gripper_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[444], y[445], y[446], y[447], y[488], y[489], y[490], y[491]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }
            }

            // base_link vs. l_gripper_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[444], y[445], y[446], y[447], y[492], y[493], y[494], y[495]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }
            }

            // base_link vs. r_gripper_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[444], y[445], y[446], y[447], y[496], y[497], y[498], y[499]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[36], y[37], y[38], y[39], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[40], y[41], y[42], y[43], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[44], y[45], y[46], y[47], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[48], y[49], y[50], y[51], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[52], y[53], y[54], y[55], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }
            }

            // torso_fixed_link vs. upperarm_roll_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[448], y[449], y[450], y[451], y[468], y[469], y[470], y[471]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }
            }

            // torso_fixed_link vs. elbow_flex_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[448], y[449], y[450], y[451], y[472], y[473], y[474], y[475]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }
            }

            // torso_fixed_link vs. forearm_roll_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[448], y[449], y[450], y[451], y[476], y[477], y[478], y[479]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }
            }

            // torso_fixed_link vs. wrist_flex_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[448], y[449], y[450], y[451], y[480], y[481], y[482], y[483]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }
            }

            // torso_fixed_link vs. wrist_roll_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[448], y[449], y[450], y[451], y[484], y[485], y[486], y[487]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }
            }

            // torso_fixed_link vs. gripper_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[448], y[449], y[450], y[451], y[488], y[489], y[490], y[491]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }
            }

            // torso_fixed_link vs. l_gripper_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[448], y[449], y[450], y[451], y[492], y[493], y[494], y[495]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }
            }

            // torso_fixed_link vs. r_gripper_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[448], y[449], y[450], y[451], y[496], y[497], y[498], y[499]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[56], y[57], y[58], y[59], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[60], y[61], y[62], y[63], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[64], y[65], y[66], y[67], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }
            }

            // torso_lift_link vs. upperarm_roll_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[452], y[453], y[454], y[455], y[468], y[469], y[470], y[471]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }
            }

            // torso_lift_link vs. elbow_flex_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[452], y[453], y[454], y[455], y[472], y[473], y[474], y[475]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[292], y[293], y[294], y[295]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[296], y[297], y[298], y[299]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[300], y[301], y[302], y[303]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[304], y[305], y[306], y[307]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[308], y[309], y[310], y[311]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[312], y[313], y[314], y[315]))
                {
                    return false;
                }
            }

            // torso_lift_link vs. forearm_roll_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[452], y[453], y[454], y[455], y[476], y[477], y[478], y[479]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[316], y[317], y[318], y[319]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[320], y[321], y[322], y[323]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[324], y[325], y[326], y[327]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[328], y[329], y[330], y[331]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[332], y[333], y[334], y[335]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[336], y[337], y[338], y[339]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[340], y[341], y[342], y[343]))
                {
                    return false;
                }
            }

            // torso_lift_link vs. wrist_flex_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[452], y[453], y[454], y[455], y[480], y[481], y[482], y[483]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[344], y[345], y[346], y[347]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[348], y[349], y[350], y[351]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[352], y[353], y[354], y[355]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[356], y[357], y[358], y[359]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[360], y[361], y[362], y[363]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[364], y[365], y[366], y[367]))
                {
                    return false;
                }
            }

            // torso_lift_link vs. wrist_roll_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[452], y[453], y[454], y[455], y[484], y[485], y[486], y[487]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[368], y[369], y[370], y[371]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[372], y[373], y[374], y[375]))
                {
                    return false;
                }
            }

            // torso_lift_link vs. gripper_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[452], y[453], y[454], y[455], y[488], y[489], y[490], y[491]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[376], y[377], y[378], y[379]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[380], y[381], y[382], y[383]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[384], y[385], y[386], y[387]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[388], y[389], y[390], y[391]))
                {
                    return false;
                }
            }

            // torso_lift_link vs. l_gripper_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[452], y[453], y[454], y[455], y[492], y[493], y[494], y[495]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[392], y[393], y[394], y[395]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[396], y[397], y[398], y[399]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[400], y[401], y[402], y[403]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[404], y[405], y[406], y[407]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[408], y[409], y[410], y[411]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[412], y[413], y[414], y[415]))
                {
                    return false;
                }
            }

            // torso_lift_link vs. r_gripper_finger_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[452], y[453], y[454], y[455], y[496], y[497], y[498], y[499]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[416], y[417], y[418], y[419]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[420], y[421], y[422], y[423]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[424], y[425], y[426], y[427]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[428], y[429], y[430], y[431]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[432], y[433], y[434], y[435]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[436], y[437], y[438], y[439]))
                {
                    return false;
                }
            }

            // head_pan_link vs. upperarm_roll_link
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[456], y[457], y[458], y[459], y[468], y[469], y[470], y[471]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[116], y[117], y[118], y[119], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[120], y[121], y[122], y[123], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[124], y[125], y[126], y[127], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[128], y[129], y[130], y[131], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[132], y[133], y[134], y[135], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[136], y[137], y[138], y[139], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[140], y[141], y[142], y[143], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[144], y[145], y[146], y[147], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[148], y[149], y[150], y[151], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[152], y[153], y[154], y[155], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[156], y[157], y[158], y[159], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[160], y[161], y[162], y[163], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[164], y[165], y[166], y[167], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[168], y[169], y[170], y[171], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[168], y[169], y[170], y[171], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[168], y[169], y[170], y[171], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[168], y[169], y[170], y[171], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[168], y[169], y[170], y[171], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[168], y[169], y[170], y[171], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[168], y[169], y[170], y[171], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[168], y[169], y[170], y[171], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[172], y[173], y[174], y[175], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[172], y[173], y[174], y[175], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[172], y[173], y[174], y[175], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[172], y[173], y[174], y[175], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[172], y[173], y[174], y[175], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[172], y[173], y[174], y[175], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[172], y[173], y[174], y[175], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[172], y[173], y[174], y[175], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[176], y[177], y[178], y[179], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[176], y[177], y[178], y[179], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[176], y[177], y[178], y[179], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[176], y[177], y[178], y[179], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[176], y[177], y[178], y[179], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[176], y[177], y[178], y[179], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[176], y[177], y[178], y[179], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[176], y[177], y[178], y[179], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[180], y[181], y[182], y[183], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[180], y[181], y[182], y[183], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[180], y[181], y[182], y[183], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[180], y[181], y[182], y[183], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[180], y[181], y[182], y[183], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[180], y[181], y[182], y[183], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[180], y[181], y[182], y[183], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[180], y[181], y[182], y[183], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[184], y[185], y[186], y[187], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[184], y[185], y[186], y[187], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[184], y[185], y[186], y[187], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[184], y[185], y[186], y[187], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[184], y[185], y[186], y[187], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[184], y[185], y[186], y[187], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[184], y[185], y[186], y[187], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[184], y[185], y[186], y[187], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[188], y[189], y[190], y[191], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[188], y[189], y[190], y[191], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[188], y[189], y[190], y[191], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[188], y[189], y[190], y[191], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[188], y[189], y[190], y[191], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[188], y[189], y[190], y[191], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[188], y[189], y[190], y[191], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[188], y[189], y[190], y[191], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[192], y[193], y[194], y[195], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[192], y[193], y[194], y[195], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[192], y[193], y[194], y[195], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[192], y[193], y[194], y[195], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[192], y[193], y[194], y[195], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[192], y[193], y[194], y[195], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[192], y[193], y[194], y[195], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[192], y[193], y[194], y[195], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[196], y[197], y[198], y[199], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[196], y[197], y[198], y[199], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[196], y[197], y[198], y[199], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[196], y[197], y[198], y[199], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[196], y[197], y[198], y[199], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[196], y[197], y[198], y[199], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[196], y[197], y[198], y[199], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[196], y[197], y[198], y[199], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[200], y[201], y[202], y[203], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[200], y[201], y[202], y[203], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[200], y[201], y[202], y[203], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[200], y[201], y[202], y[203], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[200], y[201], y[202], y[203], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[200], y[201], y[202], y[203], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[200], y[201], y[202], y[203], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[200], y[201], y[202], y[203], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[204], y[205], y[206], y[207], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[204], y[205], y[206], y[207], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[204], y[205], y[206], y[207], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[204], y[205], y[206], y[207], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[204], y[205], y[206], y[207], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[204], y[205], y[206], y[207], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[204], y[205], y[206], y[207], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[204], y[205], y[206], y[207], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[208], y[209], y[210], y[211], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[208], y[209], y[210], y[211], y[264], y[265], y[266], y[267]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[208], y[209], y[210], y[211], y[268], y[269], y[270], y[271]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[208], y[209], y[210], y[211], y[272], y[273], y[274], y[275]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[208], y[209], y[210], y[211], y[276], y[277], y[278], y[279]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[208], y[209], y[210], y[211], y[280], y[281], y[282], y[283]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[208], y[209], y[210], y[211], y[284], y[285], y[286], y[287]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[208], y[209], y[210], y[211], y[288], y[289], y[290], y[291]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[212], y[213], y[214], y[215], y[260], y[261], y[262], y[263]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[212], y[213], y[214], y[215], y[264],