#pragma once

#include <vamp/vector.hh>
#include <vamp/vector/math.hh>
#include <vamp/collision/environment.hh>
#include <vamp/collision/validity.hh>

// NOLINTBEGIN(*-magic-numbers)
namespace vamp::robots
{
    struct Panda
    {
        static constexpr char *name = "panda";
        static constexpr std::size_t dimension = 7;
        static constexpr std::size_t n_spheres = 59;
        static constexpr float min_radius = 0.012000000104308128;
        static constexpr float max_radius = 0.07999999821186066;
        static constexpr std::size_t resolution = 32;

        static constexpr std::array<std::string_view, dimension> joint_names = {
            "panda_joint1",
            "panda_joint2",
            "panda_joint3",
            "panda_joint4",
            "panda_joint5",
            "panda_joint6",
            "panda_joint7"};
        static constexpr char *end_effector = "panda_grasptarget";

        using Configuration = FloatVector<dimension>;
        using ConfigurationArray = std::array<FloatT, dimension>;

        struct alignas(FloatVectorAlignment) ConfigurationBuffer
          : std::array<float, Configuration::num_scalars_rounded>
        {
        };

        template <std::size_t rake>
        using ConfigurationBlock = FloatVector<rake, dimension>;

        template <std::size_t rake>
        struct Spheres
        {
            FloatVector<rake, n_spheres> x;
            FloatVector<rake, n_spheres> y;
            FloatVector<rake, n_spheres> z;
            FloatVector<rake, n_spheres> r;
        };

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> s_m{
            5.934199810028076,
            3.6651999950408936,
            5.934199810028076,
            3.2288999557495117,
            5.934199810028076,
            3.909600019454956,
            5.934199810028076};

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> s_a{
            -2.967099905014038,
            -1.8325999975204468,
            -2.967099905014038,
            -3.1415998935699463,
            -2.967099905014038,
            -0.08730000257492065,
            -2.967099905014038};

        alignas(Configuration::S::Alignment) static constexpr std::array<float, dimension> d_m{
            0.16851471364498138,
            0.2728364169597626,
            0.16851471364498138,
            0.30970299243927,
            0.16851471364498138,
            0.2557806372642517,
            0.16851471364498138};

        inline static void scale_configuration(Configuration &q) noexcept
        {
            q = q * Configuration(s_m) + Configuration(s_a);
        }

        inline static void descale_configuration(Configuration &q) noexcept
        {
            q = (q - Configuration(s_a)) * Configuration(d_m);
        }

        template <std::size_t rake>
        inline static void scale_configuration_block(ConfigurationBlock<rake> &q) noexcept
        {
            q[0] = -2.967099905014038 + (q[0] * 5.934199810028076);
            q[1] = -1.8325999975204468 + (q[1] * 3.6651999950408936);
            q[2] = -2.967099905014038 + (q[2] * 5.934199810028076);
            q[3] = -3.1415998935699463 + (q[3] * 3.2288999557495117);
            q[4] = -2.967099905014038 + (q[4] * 5.934199810028076);
            q[5] = -0.08730000257492065 + (q[5] * 3.909600019454956);
            q[6] = -2.967099905014038 + (q[6] * 5.934199810028076);
        }

        template <std::size_t rake>
        inline static void descale_configuration_block(ConfigurationBlock<rake> &q) noexcept
        {
            q[0] = 0.16851471364498138 * (q[0] - -2.967099905014038);
            q[1] = 0.2728364169597626 * (q[1] - -1.8325999975204468);
            q[2] = 0.16851471364498138 * (q[2] - -2.967099905014038);
            q[3] = 0.30970299243927 * (q[3] - -3.1415998935699463);
            q[4] = 0.16851471364498138 * (q[4] - -2.967099905014038);
            q[5] = 0.2557806372642517 * (q[5] - -0.08730000257492065);
            q[6] = 0.16851471364498138 * (q[6] - -2.967099905014038);
        }

        inline static auto space_measure() noexcept -> float
        {
            return 57376.4026747593;
        }

        template <std::size_t rake>
        inline static void sphere_fk(const ConfigurationBlock<rake> &x, Spheres<rake> &out) noexcept
        {
            std::array<FloatVector<rake, 1>, 22> v;
            std::array<FloatVector<rake, 1>, 236> y;

            v[0] = sin(x[0]);
            v[1] = -v[0];
            y[4] = -0.08 * v[1];
            v[2] = cos(x[0]);
            y[5] = -0.08 * v[2];
            y[8] = -0.03 * v[1];
            y[9] = -0.03 * v[2];
            y[20] = 0.03 * v[1];
            y[21] = 0.03 * v[2];
            y[24] = 0.08 * v[1];
            y[25] = 0.08 * v[2];
            v[3] = sin(x[1]);
            v[4] = -v[3];
            v[5] = cos(x[1]);
            v[6] = 4.89663865010925e-12 * v[5];
            v[7] = v[2] * v[4] + v[1] * v[6];
            y[28] = -0.12 * v[7];
            v[6] = v[0] * v[4] + v[2] * v[6];
            y[29] = -0.12 * v[6];
            v[4] = -1. * v[5];
            y[30] = 0.333 + -0.12 * v[4];
            y[32] = -0.17 * v[7];
            y[33] = -0.17 * v[6];
            y[34] = 0.333 + -0.17 * v[4];
            v[8] = -1. * v[7] + 4.89663865010925e-12 * v[1];
            v[9] = -0.316 * v[7];
            y[36] = -0.1 * v[8] + v[9];
            v[10] = -1. * v[6] + 4.89663865010925e-12 * v[2];
            v[11] = -0.316 * v[6];
            y[37] = -0.1 * v[10] + v[11];
            v[12] = 2.39770700697438e-23 + -1. * v[4];
            v[13] = 0.333 + -0.316 * v[4];
            y[38] = -0.1 * v[12] + v[13];
            y[40] = -0.06 * v[8] + v[9];
            y[41] = -0.06 * v[10] + v[11];
            y[42] = -0.06 * v[12] + v[13];
            v[14] = 4.89663865010925e-12 * v[3];
            v[15] = v[2] * v[5] + v[1] * v[14];
            v[16] = cos(x[2]);
            v[17] = sin(x[2]);
            v[18] = 4.89663865010925e-12 * v[17];
            v[19] = v[15] * v[16] + v[7] * v[18] + v[1] * v[17];
            v[20] = -v[17];
            v[21] = 4.89663865010925e-12 * v[16];
            v[15] = v[15] * v[20] + v[7] * v[21] + v[1] * v[16];
            y[44] = 0.08 * v[19] + 0.06 * v[15] + v[9];
            v[14] = v[0] * v[5] + v[2] * v[14];
            v[5] = v[14] * v[16] + v[6] * v[18] + v[2] * v[17];
            v[14] = v[14] * v[20] + v[6] * v[21] + v[2] * v[16];
            y[45] = 0.08 * v[5] + 0.06 * v[14] + v[11];
            v[3] = -1. * v[3];
            v[18] = v[3] * v[16] + v[4] * v[18] + 4.89663865010925e-12 * v[17];
            v[3] = v[3] * v[20] + v[4] * v[21] + 4.89663865010925e-12 * v[16];
            y[46] = 0.08 * v[18] + 0.06 * v[3] + v[13];
            y[48] = 0.08 * v[19] + 0.02 * v[15] + v[9];
            y[49] = 0.08 * v[5] + 0.02 * v[14] + v[11];
            y[50] = 0.08 * v[18] + 0.02 * v[3] + v[13];
            v[21] = cos(x[3]);
            v[20] = sin(x[3]);
            v[16] = 4.89663865010925e-12 * v[20];
            v[4] = v[19] * v[21] + v[15] * v[16] + v[8] * v[20];
            v[17] = -v[20];
            v[6] = 4.89663865010925e-12 * v[21];
            v[2] = v[19] * v[17] + v[15] * v[6] + v[8] * v[21];
            v[19] = v[9] + 0.0825 * v[19];
            y[52] = -0.08 * v[4] + 0.095 * v[2] + v[19];
            v[9] = v[5] * v[21] + v[14] * v[16] + v[10] * v[20];
            v[0] = v[5] * v[17] + v[14] * v[6] + v[10] * v[21];
            v[5] = v[11] + 0.0825 * v[5];
            y[53] = -0.08 * v[9] + 0.095 * v[0] + v[5];
            v[16] = v[18] * v[21] + v[3] * v[16] + v[12] * v[20];
            v[6] = v[18] * v[17] + v[3] * v[6] + v[12] * v[21];
            v[18] = v[13] + 0.0825 * v[18];
            y[54] = -0.08 * v[16] + 0.095 * v[6] + v[18];
            v[15] = -1. * v[15] + 4.89663865010925e-12 * v[8];
            y[56] = 0.02 * v[15] + v[19];
            v[14] = -1. * v[14] + 4.89663865010925e-12 * v[10];
            y[57] = 0.02 * v[14] + v[5];
            v[3] = -1. * v[3] + 4.89663865010925e-12 * v[12];
            y[58] = 0.02 * v[3] + v[18];
            y[60] = 0.06 * v[15] + v[19];
            y[61] = 0.06 * v[14] + v[5];
            y[62] = 0.06 * v[3] + v[18];
            y[64] = -0.08 * v[4] + 0.06 * v[2] + v[19];
            y[65] = -0.08 * v[9] + 0.06 * v[0] + v[5];
            y[66] = -0.08 * v[16] + 0.06 * v[6] + v[18];
            v[12] = sin(x[4]);
            v[10] = -v[12];
            v[8] = cos(x[4]);
            v[13] = 4.89663865010925e-12 * v[8];
            v[17] = -1. * v[8];
            v[21] = v[4] * v[10] + v[2] * v[13] + v[15] * v[17];
            y[116] = v[19] + -0.0825 * v[4] + 0.384 * v[2];
            y[68] = 0.055 * v[21] + y[116];
            v[19] = v[9] * v[10] + v[0] * v[13] + v[14] * v[17];
            y[117] = v[5] + -0.0825 * v[9] + 0.384 * v[0];
            y[69] = 0.055 * v[19] + y[117];
            v[17] = v[16] * v[10] + v[6] * v[13] + v[3] * v[17];
            y[118] = v[18] + -0.0825 * v[16] + 0.384 * v[6];
            y[70] = 0.055 * v[17] + y[118];
            y[72] = 0.075 * v[21] + y[116];
            y[73] = 0.075 * v[19] + y[117];
            y[74] = 0.075 * v[17] + y[118];
            v[18] = v[2] + 4.89663865010925e-12 * v[15];
            y[76] = -0.22 * v[18] + y[116];
            v[13] = v[0] + 4.89663865010925e-12 * v[14];
            y[77] = -0.22 * v[13] + y[117];
            v[10] = v[6] + 4.89663865010925e-12 * v[3];
            y[78] = -0.22 * v[10] + y[118];
            y[80] = 0.05 * v[21] + -0.18 * v[18] + y[116];
            y[81] = 0.05 * v[19] + -0.18 * v[13] + y[117];
            y[82] = 0.05 * v[17] + -0.18 * v[10] + y[118];
            v[5] = 4.89663865010925e-12 * v[12];
            v[12] = -1. * v[12];
            v[15] = v[4] * v[8] + v[2] * v[5] + v[15] * v[12];
            y[84] = 0.01 * v[15] + 0.08 * v[21] + -0.14 * v[18] + y[116];
            v[14] = v[9] * v[8] + v[0] * v[5] + v[14] * v[12];
            y[85] = 0.01 * v[14] + 0.08 * v[19] + -0.14 * v[13] + y[117];
            v[12] = v[16] * v[8] + v[6] * v[5] + v[3] * v[12];
            y[86] = 0.01 * v[12] + 0.08 * v[17] + -0.14 * v[10] + y[118];
            y[88] = 0.01 * v[15] + 0.085 * v[21] + -0.11 * v[18] + y[116];
            y[89] = 0.01 * v[14] + 0.085 * v[19] + -0.11 * v[13] + y[117];
            y[90] = 0.01 * v[12] + 0.085 * v[17] + -0.11 * v[10] + y[118];
            y[92] = 0.01 * v[15] + 0.09 * v[21] + -0.08 * v[18] + y[116];
            y[93] = 0.01 * v[14] + 0.09 * v[19] + -0.08 * v[13] + y[117];
            y[94] = 0.01 * v[12] + 0.09 * v[17] + -0.08 * v[10] + y[118];
            y[96] = 0.01 * v[15] + 0.095 * v[21] + -0.05 * v[18] + y[116];
            y[97] = 0.01 * v[14] + 0.095 * v[19] + -0.05 * v[13] + y[117];
            y[98] = 0.01 * v[12] + 0.095 * v[17] + -0.05 * v[10] + y[118];
            y[100] = -0.01 * v[15] + 0.08 * v[21] + -0.14 * v[18] + y[116];
            y[101] = -0.01 * v[14] + 0.08 * v[19] + -0.14 * v[13] + y[117];
            y[102] = -0.01 * v[12] + 0.08 * v[17] + -0.14 * v[10] + y[118];
            y[104] = -0.01 * v[15] + 0.085 * v[21] + -0.11 * v[18] + y[116];
            y[105] = -0.01 * v[14] + 0.085 * v[19] + -0.11 * v[13] + y[117];
            y[106] = -0.01 * v[12] + 0.085 * v[17] + -0.11 * v[10] + y[118];
            y[108] = -0.01 * v[15] + 0.09 * v[21] + -0.08 * v[18] + y[116];
            y[109] = -0.01 * v[14] + 0.09 * v[19] + -0.08 * v[13] + y[117];
            y[110] = -0.01 * v[12] + 0.09 * v[17] + -0.08 * v[10] + y[118];
            y[112] = -0.01 * v[15] + 0.095 * v[21] + -0.05 * v[18] + y[116];
            y[113] = -0.01 * v[14] + 0.095 * v[19] + -0.05 * v[13] + y[117];
            y[114] = -0.01 * v[12] + 0.095 * v[17] + -0.05 * v[10] + y[118];
            v[5] = cos(x[5]);
            v[8] = sin(x[5]);
            v[3] = 4.89663865010925e-12 * v[8];
            v[6] = v[15] * v[5] + v[21] * v[3] + v[18] * v[8];
            v[16] = -v[8];
            v[0] = 4.89663865010925e-12 * v[5];
            v[15] = v[15] * v[16] + v[21] * v[0] + v[18] * v[5];
            y[120] = 0.08 * v[6] + -0.01 * v[15] + y[116];
            v[9] = v[14] * v[5] + v[19] * v[3] + v[13] * v[8];
            v[14] = v[14] * v[16] + v[19] * v[0] + v[13] * v[5];
            y[121] = 0.08 * v[9] + -0.01 * v[14] + y[117];
            v[3] = v[12] * v[5] + v[17] * v[3] + v[10] * v[8];
            v[0] = v[12] * v[16] + v[17] * v[0] + v[10] * v[5];
            y[122] = 0.08 * v[3] + -0.01 * v[0] + y[118];
            y[124] = 0.08 * v[6] + 0.035 * v[15] + y[116];
            y[125] = 0.08 * v[9] + 0.035 * v[14] + y[117];
            y[126] = 0.08 * v[3] + 0.035 * v[0] + y[118];
            v[18] = -1. * v[21] + 4.89663865010925e-12 * v[18];
            v[21] = -1. * v[15] + 4.89663865010925e-12 * v[18];
            v[16] = y[116] + 0.088 * v[6];
            y[128] = 0.07 * v[21] + v[16];
            v[13] = -1. * v[19] + 4.89663865010925e-12 * v[13];
            v[19] = -1. * v[14] + 4.89663865010925e-12 * v[13];
            v[5] = y[117] + 0.088 * v[9];
            y[129] = 0.07 * v[19] + v[5];
            v[10] = -1. * v[17] + 4.89663865010925e-12 * v[10];
            v[17] = -1. * v[0] + 4.89663865010925e-12 * v[10];
            v[12] = y[118] + 0.088 * v[3];
            y[130] = 0.07 * v[17] + v[12];
            v[8] = cos(x[6]);
            v[2] = sin(x[6]);
            v[4] = 4.89663865010925e-12 * v[2];
            v[20] = v[6] * v[8] + v[15] * v[4] + v[18] * v[2];
            v[11] = -v[2];
            v[7] = 4.89663865010925e-12 * v[8];
            v[18] = v[6] * v[11] + v[15] * v[7] + v[18] * v[8];
            y[132] = 0.02 * v[20] + 0.04 * v[18] + 0.08 * v[21] + v[16];
            v[15] = v[9] * v[8] + v[14] * v[4] + v[13] * v[2];
            v[13] = v[9] * v[11] + v[14] * v[7] + v[13] * v[8];
            y[133] = 0.02 * v[15] + 0.04 * v[13] + 0.08 * v[19] + v[5];
            v[4] = v[3] * v[8] + v[0] * v[4] + v[10] * v[2];
            v[7] = v[3] * v[11] + v[0] * v[7] + v[10] * v[8];
            y[134] = 0.02 * v[4] + 0.04 * v[7] + 0.08 * v[17] + v[12];
            y[136] = 0.04 * v[20] + 0.02 * v[18] + 0.08 * v[21] + v[16];
            y[137] = 0.04 * v[15] + 0.02 * v[13] + 0.08 * v[19] + v[5];
            y[138] = 0.04 * v[4] + 0.02 * v[7] + 0.08 * v[17] + v[12];
            y[140] = 0.04 * v[20] + 0.06 * v[18] + 0.085 * v[21] + v[16];
            y[141] = 0.04 * v[15] + 0.06 * v[13] + 0.085 * v[19] + v[5];
            y[142] = 0.04 * v[4] + 0.06 * v[7] + 0.085 * v[17] + v[12];
            y[144] = 0.06 * v[20] + 0.04 * v[18] + 0.085 * v[21] + v[16];
            y[145] = 0.06 * v[15] + 0.04 * v[13] + 0.085 * v[19] + v[5];
            y[146] = 0.06 * v[4] + 0.04 * v[7] + 0.085 * v[17] + v[12];
            y[148] = -0.0530330085889673 * v[20] + -0.0530330085890148 * v[18] + 0.117 * v[21] + v[16];
            y[149] = -0.0530330085889673 * v[15] + -0.0530330085890148 * v[13] + 0.117 * v[19] + v[5];
            y[150] = -0.0530330085889673 * v[4] + -0.0530330085890148 * v[7] + 0.117 * v[17] + v[12];
            y[152] = -0.0318198051533804 * v[20] + -0.0318198051534089 * v[18] + 0.117 * v[21] + v[16];
            y[153] = -0.0318198051533804 * v[15] + -0.0318198051534089 * v[13] + 0.117 * v[19] + v[5];
            y[154] = -0.0318198051533804 * v[4] + -0.0318198051534089 * v[7] + 0.117 * v[17] + v[12];
            y[156] = -0.0106066017177935 * v[20] + -0.010606601717803 * v[18] + 0.117 * v[21] + v[16];
            y[157] = -0.0106066017177935 * v[15] + -0.010606601717803 * v[13] + 0.117 * v[19] + v[5];
            y[158] = -0.0106066017177935 * v[4] + -0.010606601717803 * v[7] + 0.117 * v[17] + v[12];
            y[160] = 0.0106066017177935 * v[20] + 0.010606601717803 * v[18] + 0.117 * v[21] + v[16];
            y[161] = 0.0106066017177935 * v[15] + 0.010606601717803 * v[13] + 0.117 * v[19] + v[5];
            y[162] = 0.0106066017177935 * v[4] + 0.010606601717803 * v[7] + 0.117 * v[17] + v[12];
            y[164] = 0.0318198051533804 * v[20] + 0.0318198051534089 * v[18] + 0.117 * v[21] + v[16];
            y[165] = 0.0318198051533804 * v[15] + 0.0318198051534089 * v[13] + 0.117 * v[19] + v[5];
            y[166] = 0.0318198051533804 * v[4] + 0.0318198051534089 * v[7] + 0.117 * v[17] + v[12];
            y[168] = 0.0530330085889673 * v[20] + 0.0530330085890148 * v[18] + 0.117 * v[21] + v[16];
            y[169] = 0.0530330085889673 * v[15] + 0.0530330085890148 * v[13] + 0.117 * v[19] + v[5];
            y[170] = 0.0530330085889673 * v[4] + 0.0530330085890148 * v[7] + 0.117 * v[17] + v[12];
            y[172] = -0.0530330085889673 * v[20] + -0.0530330085890148 * v[18] + 0.137 * v[21] + v[16];
            y[173] = -0.0530330085889673 * v[15] + -0.0530330085890148 * v[13] + 0.137 * v[19] + v[5];
            y[174] = -0.0530330085889673 * v[4] + -0.0530330085890148 * v[7] + 0.137 * v[17] + v[12];
            y[176] = -0.0318198051533804 * v[20] + -0.0318198051534089 * v[18] + 0.137 * v[21] + v[16];
            y[177] = -0.0318198051533804 * v[15] + -0.0318198051534089 * v[13] + 0.137 * v[19] + v[5];
            y[178] = -0.0318198051533804 * v[4] + -0.0318198051534089 * v[7] + 0.137 * v[17] + v[12];
            y[180] = -0.0106066017177935 * v[20] + -0.010606601717803 * v[18] + 0.137 * v[21] + v[16];
            y[181] = -0.0106066017177935 * v[15] + -0.010606601717803 * v[13] + 0.137 * v[19] + v[5];
            y[182] = -0.0106066017177935 * v[4] + -0.010606601717803 * v[7] + 0.137 * v[17] + v[12];
            y[184] = 0.0106066017177935 * v[20] + 0.010606601717803 * v[18] + 0.137 * v[21] + v[16];
            y[185] = 0.0106066017177935 * v[15] + 0.010606601717803 * v[13] + 0.137 * v[19] + v[5];
            y[186] = 0.0106066017177935 * v[4] + 0.010606601717803 * v[7] + 0.137 * v[17] + v[12];
            y[188] = 0.0318198051533804 * v[20] + 0.0318198051534089 * v[18] + 0.137 * v[21] + v[16];
            y[189] = 0.0318198051533804 * v[15] + 0.0318198051534089 * v[13] + 0.137 * v[19] + v[5];
            y[190] = 0.0318198051533804 * v[4] + 0.0318198051534089 * v[7] + 0.137 * v[17] + v[12];
            y[192] = 0.0530330085889673 * v[20] + 0.0530330085890148 * v[18] + 0.137 * v[21] + v[16];
            y[193] = 0.0530330085889673 * v[15] + 0.0530330085890148 * v[13] + 0.137 * v[19] + v[5];
            y[194] = 0.0530330085889673 * v[4] + 0.0530330085890148 * v[7] + 0.137 * v[17] + v[12];
            y[196] = -0.0530330085889673 * v[20] + -0.0530330085890148 * v[18] + 0.157 * v[21] + v[16];
            y[197] = -0.0530330085889673 * v[15] + -0.0530330085890148 * v[13] + 0.157 * v[19] + v[5];
            y[198] = -0.0530330085889673 * v[4] + -0.0530330085890148 * v[7] + 0.157 * v[17] + v[12];
            y[200] = -0.0318198051533804 * v[20] + -0.0318198051534089 * v[18] + 0.157 * v[21] + v[16];
            y[201] = -0.0318198051533804 * v[15] + -0.0318198051534089 * v[13] + 0.157 * v[19] + v[5];
            y[202] = -0.0318198051533804 * v[4] + -0.0318198051534089 * v[7] + 0.157 * v[17] + v[12];
            y[204] = -0.0106066017177935 * v[20] + -0.010606601717803 * v[18] + 0.157 * v[21] + v[16];
            y[205] = -0.0106066017177935 * v[15] + -0.010606601717803 * v[13] + 0.157 * v[19] + v[5];
            y[206] = -0.0106066017177935 * v[4] + -0.010606601717803 * v[7] + 0.157 * v[17] + v[12];
            y[208] = 0.0106066017177935 * v[20] + 0.010606601717803 * v[18] + 0.157 * v[21] + v[16];
            y[209] = 0.0106066017177935 * v[15] + 0.010606601717803 * v[13] + 0.157 * v[19] + v[5];
            y[210] = 0.0106066017177935 * v[4] + 0.010606601717803 * v[7] + 0.157 * v[17] + v[12];
            y[212] = 0.0318198051533804 * v[20] + 0.0318198051534089 * v[18] + 0.157 * v[21] + v[16];
            y[213] = 0.0318198051533804 * v[15] + 0.0318198051534089 * v[13] + 0.157 * v[19] + v[5];
            y[214] = 0.0318198051533804 * v[4] + 0.0318198051534089 * v[7] + 0.157 * v[17] + v[12];
            y[216] = 0.0530330085889673 * v[20] + 0.0530330085890148 * v[18] + 0.157 * v[21] + v[16];
            y[217] = 0.0530330085889673 * v[15] + 0.0530330085890148 * v[13] + 0.157 * v[19] + v[5];
            y[218] = 0.0530330085889673 * v[4] + 0.0530330085890148 * v[7] + 0.157 * v[17] + v[12];
            y[220] = 0.0565685424948984 * v[20] + 0.0565685424949492 * v[18] + 0.1874 * v[21] + v[16];
            y[221] = 0.0565685424948984 * v[15] + 0.0565685424949492 * v[13] + 0.1874 * v[19] + v[5];
            y[222] = 0.0565685424948984 * v[4] + 0.0565685424949492 * v[7] + 0.1874 * v[17] + v[12];
            y[224] = 0.0516187950265948 * v[20] + 0.0516187950266411 * v[18] + 0.2094 * v[21] + v[16];
            y[225] = 0.0516187950265948 * v[15] + 0.0516187950266411 * v[13] + 0.2094 * v[19] + v[5];
            y[226] = 0.0516187950265948 * v[4] + 0.0516187950266411 * v[7] + 0.2094 * v[17] + v[12];
            y[228] = -0.0565685424948984 * v[20] + -0.0565685424949492 * v[18] + 0.1874 * v[21] + v[16];
            y[229] = -0.0565685424948984 * v[15] + -0.0565685424949492 * v[13] + 0.1874 * v[19] + v[5];
            y[230] = -0.0565685424948984 * v[4] + -0.0565685424949492 * v[7] + 0.1874 * v[17] + v[12];
            y[232] = -0.0516187950265948 * v[20] + -0.0516187950266411 * v[18] + 0.2094 * v[21] + v[16];
            y[233] = -0.0516187950265948 * v[15] + -0.0516187950266411 * v[13] + 0.2094 * v[19] + v[5];
            y[234] = -0.0516187950265948 * v[4] + -0.0516187950266411 * v[7] + 0.2094 * v[17] + v[12];
            // dependent variables without operations
            y[0] = 0.;
            y[1] = 0.;
            y[2] = 0.05;
            y[3] = 0.0799999982118607;
            y[6] = 0.333;
            y[7] = 0.0599999986588955;
            y[10] = 0.333;
            y[11] = 0.0599999986588955;
            y[12] = 0.;
            y[13] = 0.;
            y[14] = 0.213;
            y[15] = 0.0599999986588955;
            y[16] = 0.;
            y[17] = 0.;
            y[18] = 0.163;
            y[19] = 0.0599999986588955;
            y[22] = 0.333000000000147;
            y[23] = 0.0599999986588955;
            y[26] = 0.333000000000392;
            y[27] = 0.0599999986588955;
            y[31] = 0.0599999986588955;
            y[35] = 0.0599999986588955;
            y[39] = 0.0599999986588955;
            y[43] = 0.0500000007450581;
            y[47] = 0.0549999997019768;
            y[51] = 0.0549999997019768;
            y[55] = 0.0599999986588955;
            y[59] = 0.0549999997019768;
            y[63] = 0.0549999997019768;
            y[67] = 0.0549999997019768;
            y[71] = 0.0599999986588955;
            y[75] = 0.0599999986588955;
            y[79] = 0.0599999986588955;
            y[83] = 0.0500000007450581;
            y[87] = 0.025000000372529;
            y[91] = 0.025000000372529;
            y[95] = 0.025000000372529;
            y[99] = 0.025000000372529;
            y[103] = 0.025000000372529;
            y[107] = 0.025000000372529;
            y[111] = 0.025000000372529;
            y[115] = 0.025000000372529;
            y[119] = 0.0500000007450581;
            y[123] = 0.0500000007450581;
            y[127] = 0.0520000010728836;
            y[131] = 0.0500000007450581;
            y[135] = 0.025000000372529;
            y[139] = 0.025000000372529;
            y[143] = 0.0199999995529652;
            y[147] = 0.0199999995529652;
            y[151] = 0.0280000008642673;
            y[155] = 0.0280000008642673;
            y[159] = 0.0280000008642673;
            y[163] = 0.0280000008642673;
            y[167] = 0.0280000008642673;
            y[171] = 0.0280000008642673;
            y[175] = 0.0260000005364418;
            y[179] = 0.0260000005364418;
            y[183] = 0.0260000005364418;
            y[187] = 0.0260000005364418;
            y[191] = 0.0260000005364418;
            y[195] = 0.0260000005364418;
            y[199] = 0.0240000002086163;
            y[203] = 0.0240000002086163;
            y[207] = 0.0240000002086163;
            y[211] = 0.0240000002086163;
            y[215] = 0.0240000002086163;
            y[219] = 0.0240000002086163;
            y[223] = 0.0120000001043081;
            y[227] = 0.0120000001043081;
            y[231] = 0.0120000001043081;
            y[235] = 0.0120000001043081;

            for (auto i = 0U; i < 59; ++i)
            {
                out.x[i] = y[i * 4 + 0];
                out.y[i] = y[i * 4 + 1];
                out.z[i] = y[i * 4 + 2];
                out.r[i] = y[i * 4 + 3];
            }
        }

        using Debug = std::
            pair<std::vector<std::vector<std::string>>, std::vector<std::pair<std::size_t, std::size_t>>>;

        template <std::size_t rake>
        inline static auto fkcc_debug(
            const vamp::collision::Environment<FloatVector<rake>> &environment,
            const ConfigurationBlock<rake> &x) noexcept -> Debug
        {
            std::array<FloatVector<rake, 1>, 57> v;
            std::array<FloatVector<rake, 1>, 280> y;

            v[0] = sin(x[0]);
            v[1] = -v[0];
            y[4] = -0.08 * v[1];
            v[2] = cos(x[0]);
            y[5] = -0.08 * v[2];
            y[8] = -0.03 * v[1];
            y[9] = -0.03 * v[2];
            y[20] = 0.03 * v[1];
            y[21] = 0.03 * v[2];
            y[24] = 0.08 * v[1];
            y[25] = 0.08 * v[2];
            v[3] = sin(x[1]);
            v[4] = -v[3];
            v[5] = cos(x[1]);
            v[6] = 4.89663865010925e-12 * v[5];
            v[7] = v[2] * v[4] + v[1] * v[6];
            y[28] = -0.12 * v[7];
            v[6] = v[0] * v[4] + v[2] * v[6];
            y[29] = -0.12 * v[6];
            v[4] = -1. * v[5];
            y[30] = 0.333 + -0.12 * v[4];
            y[32] = -0.17 * v[7];
            y[33] = -0.17 * v[6];
            y[34] = 0.333 + -0.17 * v[4];
            v[8] = -1. * v[7] + 4.89663865010925e-12 * v[1];
            v[9] = -0.316 * v[7];
            y[36] = -0.1 * v[8] + v[9];
            v[10] = -1. * v[6] + 4.89663865010925e-12 * v[2];
            v[11] = -0.316 * v[6];
            y[37] = -0.1 * v[10] + v[11];
            v[12] = 2.39770700697438e-23 + -1. * v[4];
            v[13] = 0.333 + -0.316 * v[4];
            y[38] = -0.1 * v[12] + v[13];
            y[40] = -0.06 * v[8] + v[9];
            y[41] = -0.06 * v[10] + v[11];
            y[42] = -0.06 * v[12] + v[13];
            v[14] = 4.89663865010925e-12 * v[3];
            v[15] = v[2] * v[5] + v[1] * v[14];
            v[16] = cos(x[2]);
            v[17] = sin(x[2]);
            v[18] = 4.89663865010925e-12 * v[17];
            v[19] = v[15] * v[16] + v[7] * v[18] + v[1] * v[17];
            v[20] = -v[17];
            v[21] = 4.89663865010925e-12 * v[16];
            v[15] = v[15] * v[20] + v[7] * v[21] + v[1] * v[16];
            y[44] = 0.08 * v[19] + 0.06 * v[15] + v[9];
            v[14] = v[0] * v[5] + v[2] * v[14];
            v[5] = v[14] * v[16] + v[6] * v[18] + v[2] * v[17];
            v[14] = v[14] * v[20] + v[6] * v[21] + v[2] * v[16];
            y[45] = 0.08 * v[5] + 0.06 * v[14] + v[11];
            v[3] = -1. * v[3];
            v[18] = v[3] * v[16] + v[4] * v[18] + 4.89663865010925e-12 * v[17];
            v[3] = v[3] * v[20] + v[4] * v[21] + 4.89663865010925e-12 * v[16];
            y[46] = 0.08 * v[18] + 0.06 * v[3] + v[13];
            y[48] = 0.08 * v[19] + 0.02 * v[15] + v[9];
            y[49] = 0.08 * v[5] + 0.02 * v[14] + v[11];
            y[50] = 0.08 * v[18] + 0.02 * v[3] + v[13];
            v[21] = cos(x[3]);
            v[20] = sin(x[3]);
            v[16] = 4.89663865010925e-12 * v[20];
            v[17] = v[19] * v[21] + v[15] * v[16] + v[8] * v[20];
            v[0] = -v[20];
            v[22] = 4.89663865010925e-12 * v[21];
            v[23] = v[19] * v[0] + v[15] * v[22] + v[8] * v[21];
            v[24] = v[9] + 0.0825 * v[19];
            y[52] = -0.08 * v[17] + 0.095 * v[23] + v[24];
            v[25] = v[5] * v[21] + v[14] * v[16] + v[10] * v[20];
            v[26] = v[5] * v[0] + v[14] * v[22] + v[10] * v[21];
            v[27] = v[11] + 0.0825 * v[5];
            y[53] = -0.08 * v[25] + 0.095 * v[26] + v[27];
            v[16] = v[18] * v[21] + v[3] * v[16] + v[12] * v[20];
            v[22] = v[18] * v[0] + v[3] * v[22] + v[12] * v[21];
            v[0] = v[13] + 0.0825 * v[18];
            y[54] = -0.08 * v[16] + 0.095 * v[22] + v[0];
            v[21] = -1. * v[15] + 4.89663865010925e-12 * v[8];
            y[56] = 0.02 * v[21] + v[24];
            v[20] = -1. * v[14] + 4.89663865010925e-12 * v[10];
            y[57] = 0.02 * v[20] + v[27];
            v[28] = -1. * v[3] + 4.89663865010925e-12 * v[12];
            y[58] = 0.02 * v[28] + v[0];
            y[60] = 0.06 * v[21] + v[24];
            y[61] = 0.06 * v[20] + v[27];
            y[62] = 0.06 * v[28] + v[0];
            y[64] = -0.08 * v[17] + 0.06 * v[23] + v[24];
            y[65] = -0.08 * v[25] + 0.06 * v[26] + v[27];
            y[66] = -0.08 * v[16] + 0.06 * v[22] + v[0];
            v[29] = sin(x[4]);
            v[30] = -v[29];
            v[31] = cos(x[4]);
            v[32] = 4.89663865010925e-12 * v[31];
            v[33] = -1. * v[31];
            v[34] = v[17] * v[30] + v[23] * v[32] + v[21] * v[33];
            y[116] = v[24] + -0.0825 * v[17] + 0.384 * v[23];
            y[68] = 0.055 * v[34] + y[116];
            v[35] = v[25] * v[30] + v[26] * v[32] + v[20] * v[33];
            y[117] = v[27] + -0.0825 * v[25] + 0.384 * v[26];
            y[69] = 0.055 * v[35] + y[117];
            v[33] = v[16] * v[30] + v[22] * v[32] + v[28] * v[33];
            y[118] = v[0] + -0.0825 * v[16] + 0.384 * v[22];
            y[70] = 0.055 * v[33] + y[118];
            y[72] = 0.075 * v[34] + y[116];
            y[73] = 0.075 * v[35] + y[117];
            y[74] = 0.075 * v[33] + y[118];
            v[32] = v[23] + 4.89663865010925e-12 * v[21];
            y[76] = -0.22 * v[32] + y[116];
            v[30] = v[26] + 4.89663865010925e-12 * v[20];
            y[77] = -0.22 * v[30] + y[117];
            v[36] = v[22] + 4.89663865010925e-12 * v[28];
            y[78] = -0.22 * v[36] + y[118];
            y[80] = 0.05 * v[34] + -0.18 * v[32] + y[116];
            y[81] = 0.05 * v[35] + -0.18 * v[30] + y[117];
            y[82] = 0.05 * v[33] + -0.18 * v[36] + y[118];
            v[37] = 4.89663865010925e-12 * v[29];
            v[29] = -1. * v[29];
            v[38] = v[17] * v[31] + v[23] * v[37] + v[21] * v[29];
            y[84] = 0.01 * v[38] + 0.08 * v[34] + -0.14 * v[32] + y[116];
            v[39] = v[25] * v[31] + v[26] * v[37] + v[20] * v[29];
            y[85] = 0.01 * v[39] + 0.08 * v[35] + -0.14 * v[30] + y[117];
            v[29] = v[16] * v[31] + v[22] * v[37] + v[28] * v[29];
            y[86] = 0.01 * v[29] + 0.08 * v[33] + -0.14 * v[36] + y[118];
            y[88] = 0.01 * v[38] + 0.085 * v[34] + -0.11 * v[32] + y[116];
            y[89] = 0.01 * v[39] + 0.085 * v[35] + -0.11 * v[30] + y[117];
            y[90] = 0.01 * v[29] + 0.085 * v[33] + -0.11 * v[36] + y[118];
            y[92] = 0.01 * v[38] + 0.09 * v[34] + -0.08 * v[32] + y[116];
            y[93] = 0.01 * v[39] + 0.09 * v[35] + -0.08 * v[30] + y[117];
            y[94] = 0.01 * v[29] + 0.09 * v[33] + -0.08 * v[36] + y[118];
            y[96] = 0.01 * v[38] + 0.095 * v[34] + -0.05 * v[32] + y[116];
            y[97] = 0.01 * v[39] + 0.095 * v[35] + -0.05 * v[30] + y[117];
            y[98] = 0.01 * v[29] + 0.095 * v[33] + -0.05 * v[36] + y[118];
            y[100] = -0.01 * v[38] + 0.08 * v[34] + -0.14 * v[32] + y[116];
            y[101] = -0.01 * v[39] + 0.08 * v[35] + -0.14 * v[30] + y[117];
            y[102] = -0.01 * v[29] + 0.08 * v[33] + -0.14 * v[36] + y[118];
            y[104] = -0.01 * v[38] + 0.085 * v[34] + -0.11 * v[32] + y[116];
            y[105] = -0.01 * v[39] + 0.085 * v[35] + -0.11 * v[30] + y[117];
            y[106] = -0.01 * v[29] + 0.085 * v[33] + -0.11 * v[36] + y[118];
            y[108] = -0.01 * v[38] + 0.09 * v[34] + -0.08 * v[32] + y[116];
            y[109] = -0.01 * v[39] + 0.09 * v[35] + -0.08 * v[30] + y[117];
            y[110] = -0.01 * v[29] + 0.09 * v[33] + -0.08 * v[36] + y[118];
            y[112] = -0.01 * v[38] + 0.095 * v[34] + -0.05 * v[32] + y[116];
            y[113] = -0.01 * v[39] + 0.095 * v[35] + -0.05 * v[30] + y[117];
            y[114] = -0.01 * v[29] + 0.095 * v[33] + -0.05 * v[36] + y[118];
            v[37] = cos(x[5]);
            v[31] = sin(x[5]);
            v[40] = 4.89663865010925e-12 * v[31];
            v[41] = v[38] * v[37] + v[34] * v[40] + v[32] * v[31];
            v[42] = -v[31];
            v[43] = 4.89663865010925e-12 * v[37];
            v[38] = v[38] * v[42] + v[34] * v[43] + v[32] * v[37];
            y[120] = 0.08 * v[41] + -0.01 * v[38] + y[116];
            v[44] = v[39] * v[37] + v[35] * v[40] + v[30] * v[31];
            v[39] = v[39] * v[42] + v[35] * v[43] + v[30] * v[37];
            y[121] = 0.08 * v[44] + -0.01 * v[39] + y[117];
            v[40] = v[29] * v[37] + v[33] * v[40] + v[36] * v[31];
            v[43] = v[29] * v[42] + v[33] * v[43] + v[36] * v[37];
            y[122] = 0.08 * v[40] + -0.01 * v[43] + y[118];
            y[124] = 0.08 * v[41] + 0.035 * v[38] + y[116];
            y[125] = 0.08 * v[44] + 0.035 * v[39] + y[117];
            y[126] = 0.08 * v[40] + 0.035 * v[43] + y[118];
            v[42] = -1. * v[34] + 4.89663865010925e-12 * v[32];
            v[37] = -1. * v[38] + 4.89663865010925e-12 * v[42];
            v[29] = y[116] + 0.088 * v[41];
            y[128] = 0.07 * v[37] + v[29];
            v[31] = -1. * v[35] + 4.89663865010925e-12 * v[30];
            v[45] = -1. * v[39] + 4.89663865010925e-12 * v[31];
            v[46] = y[117] + 0.088 * v[44];
            y[129] = 0.07 * v[45] + v[46];
            v[47] = -1. * v[33] + 4.89663865010925e-12 * v[36];
            v[48] = -1. * v[43] + 4.89663865010925e-12 * v[47];
            v[49] = y[118] + 0.088 * v[40];
            y[130] = 0.07 * v[48] + v[49];
            v[50] = cos(x[6]);
            v[51] = sin(x[6]);
            v[52] = 4.89663865010925e-12 * v[51];
            v[53] = v[41] * v[50] + v[38] * v[52] + v[42] * v[51];
            v[54] = -v[51];
            v[55] = 4.89663865010925e-12 * v[50];
            v[42] = v[41] * v[54] + v[38] * v[55] + v[42] * v[50];
            y[132] = 0.02 * v[53] + 0.04 * v[42] + 0.08 * v[37] + v[29];
            v[56] = v[44] * v[50] + v[39] * v[52] + v[31] * v[51];
            v[31] = v[44] * v[54] + v[39] * v[55] + v[31] * v[50];
            y[133] = 0.02 * v[56] + 0.04 * v[31] + 0.08 * v[45] + v[46];
            v[52] = v[40] * v[50] + v[43] * v[52] + v[47] * v[51];
            v[55] = v[40] * v[54] + v[43] * v[55] + v[47] * v[50];
            y[134] = 0.02 * v[52] + 0.04 * v[55] + 0.08 * v[48] + v[49];
            y[136] = 0.04 * v[53] + 0.02 * v[42] + 0.08 * v[37] + v[29];
            y[137] = 0.04 * v[56] + 0.02 * v[31] + 0.08 * v[45] + v[46];
            y[138] = 0.04 * v[52] + 0.02 * v[55] + 0.08 * v[48] + v[49];
            y[140] = 0.04 * v[53] + 0.06 * v[42] + 0.085 * v[37] + v[29];
            y[141] = 0.04 * v[56] + 0.06 * v[31] + 0.085 * v[45] + v[46];
            y[142] = 0.04 * v[52] + 0.06 * v[55] + 0.085 * v[48] + v[49];
            y[144] = 0.06 * v[53] + 0.04 * v[42] + 0.085 * v[37] + v[29];
            y[145] = 0.06 * v[56] + 0.04 * v[31] + 0.085 * v[45] + v[46];
            y[146] = 0.06 * v[52] + 0.04 * v[55] + 0.085 * v[48] + v[49];
            y[148] = -0.0530330085889673 * v[53] + -0.0530330085890148 * v[42] + 0.117 * v[37] + v[29];
            y[149] = -0.0530330085889673 * v[56] + -0.0530330085890148 * v[31] + 0.117 * v[45] + v[46];
            y[150] = -0.0530330085889673 * v[52] + -0.0530330085890148 * v[55] + 0.117 * v[48] + v[49];
            y[152] = -0.0318198051533804 * v[53] + -0.0318198051534089 * v[42] + 0.117 * v[37] + v[29];
            y[153] = -0.0318198051533804 * v[56] + -0.0318198051534089 * v[31] + 0.117 * v[45] + v[46];
            y[154] = -0.0318198051533804 * v[52] + -0.0318198051534089 * v[55] + 0.117 * v[48] + v[49];
            y[156] = -0.0106066017177935 * v[53] + -0.010606601717803 * v[42] + 0.117 * v[37] + v[29];
            y[157] = -0.0106066017177935 * v[56] + -0.010606601717803 * v[31] + 0.117 * v[45] + v[46];
            y[158] = -0.0106066017177935 * v[52] + -0.010606601717803 * v[55] + 0.117 * v[48] + v[49];
            y[160] = 0.0106066017177935 * v[53] + 0.010606601717803 * v[42] + 0.117 * v[37] + v[29];
            y[161] = 0.0106066017177935 * v[56] + 0.010606601717803 * v[31] + 0.117 * v[45] + v[46];
            y[162] = 0.0106066017177935 * v[52] + 0.010606601717803 * v[55] + 0.117 * v[48] + v[49];
            y[164] = 0.0318198051533804 * v[53] + 0.0318198051534089 * v[42] + 0.117 * v[37] + v[29];
            y[165] = 0.0318198051533804 * v[56] + 0.0318198051534089 * v[31] + 0.117 * v[45] + v[46];
            y[166] = 0.0318198051533804 * v[52] + 0.0318198051534089 * v[55] + 0.117 * v[48] + v[49];
            y[168] = 0.0530330085889673 * v[53] + 0.0530330085890148 * v[42] + 0.117 * v[37] + v[29];
            y[169] = 0.0530330085889673 * v[56] + 0.0530330085890148 * v[31] + 0.117 * v[45] + v[46];
            y[170] = 0.0530330085889673 * v[52] + 0.0530330085890148 * v[55] + 0.117 * v[48] + v[49];
            y[172] = -0.0530330085889673 * v[53] + -0.0530330085890148 * v[42] + 0.137 * v[37] + v[29];
            y[173] = -0.0530330085889673 * v[56] + -0.0530330085890148 * v[31] + 0.137 * v[45] + v[46];
            y[174] = -0.0530330085889673 * v[52] + -0.0530330085890148 * v[55] + 0.137 * v[48] + v[49];
            y[176] = -0.0318198051533804 * v[53] + -0.0318198051534089 * v[42] + 0.137 * v[37] + v[29];
            y[177] = -0.0318198051533804 * v[56] + -0.0318198051534089 * v[31] + 0.137 * v[45] + v[46];
            y[178] = -0.0318198051533804 * v[52] + -0.0318198051534089 * v[55] + 0.137 * v[48] + v[49];
            y[180] = -0.0106066017177935 * v[53] + -0.010606601717803 * v[42] + 0.137 * v[37] + v[29];
            y[181] = -0.0106066017177935 * v[56] + -0.010606601717803 * v[31] + 0.137 * v[45] + v[46];
            y[182] = -0.0106066017177935 * v[52] + -0.010606601717803 * v[55] + 0.137 * v[48] + v[49];
            y[184] = 0.0106066017177935 * v[53] + 0.010606601717803 * v[42] + 0.137 * v[37] + v[29];
            y[185] = 0.0106066017177935 * v[56] + 0.010606601717803 * v[31] + 0.137 * v[45] + v[46];
            y[186] = 0.0106066017177935 * v[52] + 0.010606601717803 * v[55] + 0.137 * v[48] + v[49];
            y[188] = 0.0318198051533804 * v[53] + 0.0318198051534089 * v[42] + 0.137 * v[37] + v[29];
            y[189] = 0.0318198051533804 * v[56] + 0.0318198051534089 * v[31] + 0.137 * v[45] + v[46];
            y[190] = 0.0318198051533804 * v[52] + 0.0318198051534089 * v[55] + 0.137 * v[48] + v[49];
            y[192] = 0.0530330085889673 * v[53] + 0.0530330085890148 * v[42] + 0.137 * v[37] + v[29];
            y[193] = 0.0530330085889673 * v[56] + 0.0530330085890148 * v[31] + 0.137 * v[45] + v[46];
            y[194] = 0.0530330085889673 * v[52] + 0.0530330085890148 * v[55] + 0.137 * v[48] + v[49];
            y[196] = -0.0530330085889673 * v[53] + -0.0530330085890148 * v[42] + 0.157 * v[37] + v[29];
            y[197] = -0.0530330085889673 * v[56] + -0.0530330085890148 * v[31] + 0.157 * v[45] + v[46];
            y[198] = -0.0530330085889673 * v[52] + -0.0530330085890148 * v[55] + 0.157 * v[48] + v[49];
            y[200] = -0.0318198051533804 * v[53] + -0.0318198051534089 * v[42] + 0.157 * v[37] + v[29];
            y[201] = -0.0318198051533804 * v[56] + -0.0318198051534089 * v[31] + 0.157 * v[45] + v[46];
            y[202] = -0.0318198051533804 * v[52] + -0.0318198051534089 * v[55] + 0.157 * v[48] + v[49];
            y[204] = -0.0106066017177935 * v[53] + -0.010606601717803 * v[42] + 0.157 * v[37] + v[29];
            y[205] = -0.0106066017177935 * v[56] + -0.010606601717803 * v[31] + 0.157 * v[45] + v[46];
            y[206] = -0.0106066017177935 * v[52] + -0.010606601717803 * v[55] + 0.157 * v[48] + v[49];
            y[208] = 0.0106066017177935 * v[53] + 0.010606601717803 * v[42] + 0.157 * v[37] + v[29];
            y[209] = 0.0106066017177935 * v[56] + 0.010606601717803 * v[31] + 0.157 * v[45] + v[46];
            y[210] = 0.0106066017177935 * v[52] + 0.010606601717803 * v[55] + 0.157 * v[48] + v[49];
            y[212] = 0.0318198051533804 * v[53] + 0.0318198051534089 * v[42] + 0.157 * v[37] + v[29];
            y[213] = 0.0318198051533804 * v[56] + 0.0318198051534089 * v[31] + 0.157 * v[45] + v[46];
            y[214] = 0.0318198051533804 * v[52] + 0.0318198051534089 * v[55] + 0.157 * v[48] + v[49];
            y[216] = 0.0530330085889673 * v[53] + 0.0530330085890148 * v[42] + 0.157 * v[37] + v[29];
            y[217] = 0.0530330085889673 * v[56] + 0.0530330085890148 * v[31] + 0.157 * v[45] + v[46];
            y[218] = 0.0530330085889673 * v[52] + 0.0530330085890148 * v[55] + 0.157 * v[48] + v[49];
            y[220] = 0.0565685424948984 * v[53] + 0.0565685424949492 * v[42] + 0.1874 * v[37] + v[29];
            y[221] = 0.0565685424948984 * v[56] + 0.0565685424949492 * v[31] + 0.1874 * v[45] + v[46];
            y[222] = 0.0565685424948984 * v[52] + 0.0565685424949492 * v[55] + 0.1874 * v[48] + v[49];
            y[224] = 0.0516187950265948 * v[53] + 0.0516187950266411 * v[42] + 0.2094 * v[37] + v[29];
            y[225] = 0.0516187950265948 * v[56] + 0.0516187950266411 * v[31] + 0.2094 * v[45] + v[46];
            y[226] = 0.0516187950265948 * v[52] + 0.0516187950266411 * v[55] + 0.2094 * v[48] + v[49];
            y[228] = -0.0565685424948984 * v[53] + -0.0565685424949492 * v[42] + 0.1874 * v[37] + v[29];
            y[229] = -0.0565685424948984 * v[56] + -0.0565685424949492 * v[31] + 0.1874 * v[45] + v[46];
            y[230] = -0.0565685424948984 * v[52] + -0.0565685424949492 * v[55] + 0.1874 * v[48] + v[49];
            y[232] = -0.0516187950265948 * v[53] + -0.0516187950266411 * v[42] + 0.2094 * v[37] + v[29];
            y[233] = -0.0516187950265948 * v[56] + -0.0516187950266411 * v[31] + 0.2094 * v[45] + v[46];
            y[234] = -0.0516187950265948 * v[52] + -0.0516187950266411 * v[55] + 0.2094 * v[48] + v[49];
            y[240] = -0.0399999991059303 * v[1];
            y[241] = -0.0399999991059303 * v[2];
            y[244] = -0.0850000008940697 * v[7] + 0.0399999991059303 * v[1];
            y[245] = -0.0850000008940697 * v[6] + 0.0399999991059303 * v[2];
            y[246] = 0.333 + 1.95865541626434e-13 + -0.0850000008940697 * v[4];
            y[248] =
                0.0385857857763767 * v[19] + 0.0289393402636051 * v[15] + -0.0517677664756775 * v[8] + v[9];
            y[249] =
                0.0385857857763767 * v[5] + 0.0289393402636051 * v[14] + -0.0517677664756775 * v[10] + v[11];
            y[250] =
                0.0385857857763767 * v[18] + 0.0289393402636051 * v[3] + -0.0517677664756775 * v[12] + v[13];
            y[252] =
                -0.041449997574091 * v[17] + 0.0492218732833862 * v[23] + 0.0289125014096498 * v[21] + v[24];
            y[253] =
                -0.041449997574091 * v[25] + 0.0492218732833862 * v[26] + 0.0289125014096498 * v[20] + v[27];
            y[254] =
                -0.041449997574091 * v[16] + 0.0492218732833862 * v[22] + 0.0289125014096498 * v[28] + v[0];
            y[256] = 0.0375000014901161 * v[34] + -0.109999999403954 * v[32] + y[116];
            y[257] = 0.0375000014901161 * v[35] + -0.109999999403954 * v[30] + y[117];
            y[258] = 0.0375000014901161 * v[33] + -0.109999999403954 * v[36] + y[118];
            y[260] = 0.0424309633672237 * v[41] + 0.0144476993009448 * v[38] + y[116];
            y[261] = 0.0424309633672237 * v[44] + 0.0144476993009448 * v[39] + y[117];
            y[262] = 0.0424309633672237 * v[40] + 0.0144476993009448 * v[43] + y[118];
            y[264] =
                0.0153005504980683 * v[53] + 0.0153005504980683 * v[42] + 0.0745901614427567 * v[37] + v[29];
            y[265] =
                0.0153005504980683 * v[56] + 0.0153005504980683 * v[31] + 0.0745901614427567 * v[45] + v[46];
            y[266] =
                0.0153005504980683 * v[52] + 0.0153005504980683 * v[55] + 0.0745901614427567 * v[48] + v[49];
            y[268] = -5.20417042793042e-18 * v[53] + -2.16840434497101e-18 * v[42] +
                     0.129201397299767 * v[37] + v[29];
            y[269] = -5.20417042793042e-18 * v[56] + -2.16840434497101e-18 * v[31] +
                     0.129201397299767 * v[45] + v[46];
            y[270] = -5.20417042793042e-18 * v[52] + -2.16840434497101e-18 * v[55] +
                     0.129201397299767 * v[48] + v[49];
            y[272] =
                0.0540936700999737 * v[53] + 0.0540936700999737 * v[42] + 0.198400005698204 * v[37] + v[29];
            y[273] =
                0.0540936700999737 * v[56] + 0.0540936700999737 * v[31] + 0.198400005698204 * v[45] + v[46];
            y[274] =
                0.0540936700999737 * v[52] + 0.0540936700999737 * v[55] + 0.198400005698204 * v[48] + v[49];
            y[276] =
                -0.0540936700999737 * v[53] + -0.0540936700999737 * v[42] + 0.198400005698204 * v[37] + v[29];
            y[277] =
                -0.0540936700999737 * v[56] + -0.0540936700999737 * v[31] + 0.198400005698204 * v[45] + v[46];
            y[278] =
                -0.0540936700999737 * v[52] + -0.0540936700999737 * v[55] + 0.198400005698204 * v[48] + v[49];
            // dependent variables without operations
            y[0] = 0.;
            y[1] = 0.;
            y[2] = 0.05;
            y[3] = 0.0799999982118607;
            y[6] = 0.333;
            y[7] = 0.0599999986588955;
            y[10] = 0.333;
            y[11] = 0.0599999986588955;
            y[12] = 0.;
            y[13] = 0.;
            y[14] = 0.213;
            y[15] = 0.0599999986588955;
            y[16] = 0.;
            y[17] = 0.;
            y[18] = 0.163;
            y[19] = 0.0599999986588955;
            y[22] = 0.333000000000147;
            y[23] = 0.0599999986588955;
            y[26] = 0.333000000000392;
            y[27] = 0.0599999986588955;
            y[31] = 0.0599999986588955;
            y[35] = 0.0599999986588955;
            y[39] = 0.0599999986588955;
            y[43] = 0.0500000007450581;
            y[47] = 0.0549999997019768;
            y[51] = 0.0549999997019768;
            y[55] = 0.0599999986588955;
            y[59] = 0.0549999997019768;
            y[63] = 0.0549999997019768;
            y[67] = 0.0549999997019768;
            y[71] = 0.0599999986588955;
            y[75] = 0.0599999986588955;
            y[79] = 0.0599999986588955;
            y[83] = 0.0500000007450581;
            y[87] = 0.025000000372529;
            y[91] = 0.025000000372529;
            y[95] = 0.025000000372529;
            y[99] = 0.025000000372529;
            y[103] = 0.025000000372529;
            y[107] = 0.025000000372529;
            y[111] = 0.025000000372529;
            y[115] = 0.025000000372529;
            y[119] = 0.0500000007450581;
            y[123] = 0.0500000007450581;
            y[127] = 0.0520000010728836;
            y[131] = 0.0500000007450581;
            y[135] = 0.025000000372529;
            y[139] = 0.025000000372529;
            y[143] = 0.0199999995529652;
            y[147] = 0.0199999995529652;
            y[151] = 0.0280000008642673;
            y[155] = 0.0280000008642673;
            y[159] = 0.0280000008642673;
            y[163] = 0.0280000008642673;
            y[167] = 0.0280000008642673;
            y[171] = 0.0280000008642673;
            y[175] = 0.0260000005364418;
            y[179] = 0.0260000005364418;
            y[183] = 0.0260000005364418;
            y[187] = 0.0260000005364418;
            y[191] = 0.0260000005364418;
            y[195] = 0.0260000005364418;
            y[199] = 0.0240000002086163;
            y[203] = 0.0240000002086163;
            y[207] = 0.0240000002086163;
            y[211] = 0.0240000002086163;
            y[215] = 0.0240000002086163;
            y[219] = 0.0240000002086163;
            y[223] = 0.0120000001043081;
            y[227] = 0.0120000001043081;
            y[231] = 0.0120000001043081;
            y[235] = 0.0120000001043081;
            y[236] = 0.;
            y[237] = 0.;
            y[238] = 0.0500000007450581;
            y[239] = 0.0799999982118607;
            y[242] = 0.24799999910593;
            y[243] = 0.153941467404366;
            y[247] = 0.153941467404366;
            y[251] = 0.128210678696632;
            y[255] = 0.126465573906898;
            y[259] = 0.176216393709183;
            y[263] = 0.0948232412338257;
            y[267] = 0.0721197500824928;
            y[271] = 0.103986009955406;
            y[275] = 0.0235433969646692;
            y[279] = 0.0235433969646692;

            Debug output;

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[0], y[1], y[2], y[3]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[4], y[5], y[6], y[7]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[8], y[9], y[10], y[11]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[12], y[13], y[14], y[15]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[16], y[17], y[18], y[19]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[20], y[21], y[22], y[23]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[24], y[25], y[26], y[27]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[28], y[29], y[30], y[31]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[32], y[33], y[34], y[35]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[36], y[37], y[38], y[39]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[40], y[41], y[42], y[43]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[44], y[45], y[46], y[47]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[48], y[49], y[50], y[51]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[52], y[53], y[54], y[55]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[56], y[57], y[58], y[59]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[60], y[61], y[62], y[63]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[64], y[65], y[66], y[67]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[68], y[69], y[70], y[71]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[72], y[73], y[74], y[75]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[76], y[77], y[78], y[79]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[80], y[81], y[82], y[83]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[84], y[85], y[86], y[87]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[88], y[89], y[90], y[91]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[92], y[93], y[94], y[95]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(environment, y[96], y[97], y[98], y[99]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[100], y[101], y[102], y[103]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[104], y[105], y[106], y[107]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[108], y[109], y[110], y[111]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[112], y[113], y[114], y[115]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[116], y[117], y[118], y[119]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[120], y[121], y[122], y[123]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[124], y[125], y[126], y[127]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[128], y[129], y[130], y[131]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[132], y[133], y[134], y[135]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[136], y[137], y[138], y[139]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[140], y[141], y[142], y[143]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[144], y[145], y[146], y[147]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[148], y[149], y[150], y[151]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[152], y[153], y[154], y[155]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[156], y[157], y[158], y[159]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[160], y[161], y[162], y[163]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[164], y[165], y[166], y[167]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[168], y[169], y[170], y[171]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[172], y[173], y[174], y[175]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[176], y[177], y[178], y[179]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[180], y[181], y[182], y[183]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[184], y[185], y[186], y[187]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[188], y[189], y[190], y[191]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[192], y[193], y[194], y[195]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[196], y[197], y[198], y[199]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[200], y[201], y[202], y[203]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[204], y[205], y[206], y[207]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[208], y[209], y[210], y[211]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[212], y[213], y[214], y[215]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[216], y[217], y[218], y[219]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[220], y[221], y[222], y[223]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[224], y[225], y[226], y[227]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[228], y[229], y[230], y[231]));

            output.first.emplace_back(
                sphere_environment_get_collisions<decltype(x[0])>(
                    environment, y[232], y[233], y[234], y[235]));

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(0, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(0, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(0, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(0, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(0, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(0, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(0, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(0, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(0, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(0, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(0, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(0, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(0, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(0, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(0, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(0, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(0, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(0, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(0, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(0, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(0, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(0, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(0, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(0, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(0, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(0, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(0, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(0, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(0, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(0, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(0, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(0, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(0, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(0, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(0, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(0, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(0, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(0, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(0, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(0, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(0, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[0], y[1], y[2], y[3], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(0, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(1, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(1, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(1, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(1, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(1, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(1, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(1, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(1, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(1, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(1, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(1, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(1, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(2, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(2, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(2, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(2, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(2, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(2, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(2, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(2, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(2, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(2, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(2, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(2, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(3, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(3, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(3, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(3, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(3, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(3, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(3, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(3, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(3, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(3, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(3, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(3, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(4, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(4, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(4, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(4, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(4, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(4, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(4, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(4, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(4, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(4, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(4, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(4, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(1, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(1, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(1, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(2, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(2, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(2, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(3, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(3, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(3, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[116], y[117], y[118], y[119]))
            {
                output.second.emplace_back(4, 29);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[120], y[121], y[122], y[123]))
            {
                output.second.emplace_back(4, 30);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[124], y[125], y[126], y[127]))
            {
                output.second.emplace_back(4, 31);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(1, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(1, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(1, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(1, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(1, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(2, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(2, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(2, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(2, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(2, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(3, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(3, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(3, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(3, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(3, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(4, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(4, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(4, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(4, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(4, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(1, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(1, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(1, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(1, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(1, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(1, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(1, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(1, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(1, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(1, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(1, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(1, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(1, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(1, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(1, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(1, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(1, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(1, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(2, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(2, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(2, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(2, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(2, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(2, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(2, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(2, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(2, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(2, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(2, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(2, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(2, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(2, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(2, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(2, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(2, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(2, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(3, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(3, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(3, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(3, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(3, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(3, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(3, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(3, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(3, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(3, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(3, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(3, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(3, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(3, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(3, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(3, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(3, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(3, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(4, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(4, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(4, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(4, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(4, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(4, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(4, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(4, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(4, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(4, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(4, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(4, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(4, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(4, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(4, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(4, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(4, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(4, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(1, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(1, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(2, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(2, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(3, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(3, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(4, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(4, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(1, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[4], y[5], y[6], y[7], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(1, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(2, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[8], y[9], y[10], y[11], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(2, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(3, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[12], y[13], y[14], y[15], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(3, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(4, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[16], y[17], y[18], y[19], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(4, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(5, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(5, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(5, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(5, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(5, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(5, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(5, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(5, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(5, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(5, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(5, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(5, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(6, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(6, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(6, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(6, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(6, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(6, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(6, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(6, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(6, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(6, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(6, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(6, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(7, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(7, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(7, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(7, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(7, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(7, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(7, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(7, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(7, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(7, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(7, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(7, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[68], y[69], y[70], y[71]))
            {
                output.second.emplace_back(8, 17);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[72], y[73], y[74], y[75]))
            {
                output.second.emplace_back(8, 18);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[76], y[77], y[78], y[79]))
            {
                output.second.emplace_back(8, 19);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[80], y[81], y[82], y[83]))
            {
                output.second.emplace_back(8, 20);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[84], y[85], y[86], y[87]))
            {
                output.second.emplace_back(8, 21);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[88], y[89], y[90], y[91]))
            {
                output.second.emplace_back(8, 22);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[92], y[93], y[94], y[95]))
            {
                output.second.emplace_back(8, 23);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[96], y[97], y[98], y[99]))
            {
                output.second.emplace_back(8, 24);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[100], y[101], y[102], y[103]))
            {
                output.second.emplace_back(8, 25);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[104], y[105], y[106], y[107]))
            {
                output.second.emplace_back(8, 26);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[108], y[109], y[110], y[111]))
            {
                output.second.emplace_back(8, 27);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[112], y[113], y[114], y[115]))
            {
                output.second.emplace_back(8, 28);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(5, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(5, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(5, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(5, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(5, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(6, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(6, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(6, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(6, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(6, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(7, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(7, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(7, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(7, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(7, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(8, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(8, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(8, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(8, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(8, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(5, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(5, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(5, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(5, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(5, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(5, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(5, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(5, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(5, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(5, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(5, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(5, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(5, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(5, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(5, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(5, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(5, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(5, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(6, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(6, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(6, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(6, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(6, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(6, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(6, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(6, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(6, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(6, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(6, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(6, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(6, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(6, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(6, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(6, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(6, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(6, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(7, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(7, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(7, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(7, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(7, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(7, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(7, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(7, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(7, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(7, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(7, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(7, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(7, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(7, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(7, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(7, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(7, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(7, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(8, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(8, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(8, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(8, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(8, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(8, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(8, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(8, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(8, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(8, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(8, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(8, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(8, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(8, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(8, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(8, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(8, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(8, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(5, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(5, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(6, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(6, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(7, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(7, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(8, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(8, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(5, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[20], y[21], y[22], y[23], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(5, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(6, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[24], y[25], y[26], y[27], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(6, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(7, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[28], y[29], y[30], y[31], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(7, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(8, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[32], y[33], y[34], y[35], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(8, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(17, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(17, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(17, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(17, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(17, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(18, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(18, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(18, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(18, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(18, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(19, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(19, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(19, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(19, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(19, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(20, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(20, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(20, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(20, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(20, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(21, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(21, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(21, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(21, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(21, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(22, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(22, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(22, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(22, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(22, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(23, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(23, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(23, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(23, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(23, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(24, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(24, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(24, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(24, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(24, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(25, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(25, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(25, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(25, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(25, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(26, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(26, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(26, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(26, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(26, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(27, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(27, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(27, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(27, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(27, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[128], y[129], y[130], y[131]))
            {
                output.second.emplace_back(28, 32);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[132], y[133], y[134], y[135]))
            {
                output.second.emplace_back(28, 33);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[136], y[137], y[138], y[139]))
            {
                output.second.emplace_back(28, 34);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[140], y[141], y[142], y[143]))
            {
                output.second.emplace_back(28, 35);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[144], y[145], y[146], y[147]))
            {
                output.second.emplace_back(28, 36);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(17, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(17, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(17, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(17, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(17, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(17, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(17, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(17, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(17, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(17, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(17, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(17, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(17, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(17, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(17, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(17, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(17, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(17, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(18, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(18, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(18, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(18, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(18, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(18, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(18, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(18, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(18, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(18, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(18, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(18, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(18, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(18, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(18, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(18, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(18, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(18, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(19, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(19, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(19, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(19, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(19, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(19, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(19, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(19, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(19, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(19, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(19, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(19, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(19, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(19, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(19, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(19, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(19, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(19, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(20, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(20, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(20, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(20, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(20, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(20, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(20, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(20, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(20, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(20, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(20, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(20, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(20, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(20, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(20, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(20, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(20, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(20, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(21, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(21, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(21, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(21, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(21, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(21, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(21, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(21, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(21, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(21, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(21, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(21, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(21, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(21, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(21, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(21, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(21, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(21, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(22, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(22, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(22, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(22, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(22, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(22, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(22, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(22, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(22, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(22, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(22, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(22, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(22, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(22, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(22, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(22, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(22, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(22, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(23, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(23, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(23, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(23, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(23, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(23, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(23, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(23, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(23, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(23, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(23, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(23, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(23, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(23, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(23, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(23, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(23, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(23, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(24, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(24, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(24, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(24, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(24, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(24, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(24, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(24, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(24, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(24, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(24, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(24, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(24, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(24, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(24, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(24, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(24, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(24, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(25, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(25, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(25, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(25, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(25, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(25, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(25, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(25, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(25, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(25, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(25, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(25, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(25, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(25, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(25, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(25, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(25, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(25, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(26, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(26, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(26, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(26, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(26, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(26, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(26, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(26, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(26, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(26, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(26, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(26, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(26, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(26, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(26, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(26, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(26, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(26, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(27, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(27, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(27, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(27, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(27, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(27, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(27, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(27, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(27, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(27, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(27, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(27, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(27, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(27, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(27, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(27, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(27, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(27, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[148], y[149], y[150], y[151]))
            {
                output.second.emplace_back(28, 37);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[152], y[153], y[154], y[155]))
            {
                output.second.emplace_back(28, 38);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[156], y[157], y[158], y[159]))
            {
                output.second.emplace_back(28, 39);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[160], y[161], y[162], y[163]))
            {
                output.second.emplace_back(28, 40);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[164], y[165], y[166], y[167]))
            {
                output.second.emplace_back(28, 41);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[168], y[169], y[170], y[171]))
            {
                output.second.emplace_back(28, 42);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[172], y[173], y[174], y[175]))
            {
                output.second.emplace_back(28, 43);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[176], y[177], y[178], y[179]))
            {
                output.second.emplace_back(28, 44);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[180], y[181], y[182], y[183]))
            {
                output.second.emplace_back(28, 45);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[184], y[185], y[186], y[187]))
            {
                output.second.emplace_back(28, 46);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[188], y[189], y[190], y[191]))
            {
                output.second.emplace_back(28, 47);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[192], y[193], y[194], y[195]))
            {
                output.second.emplace_back(28, 48);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[196], y[197], y[198], y[199]))
            {
                output.second.emplace_back(28, 49);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[200], y[201], y[202], y[203]))
            {
                output.second.emplace_back(28, 50);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[204], y[205], y[206], y[207]))
            {
                output.second.emplace_back(28, 51);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[208], y[209], y[210], y[211]))
            {
                output.second.emplace_back(28, 52);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[212], y[213], y[214], y[215]))
            {
                output.second.emplace_back(28, 53);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[216], y[217], y[218], y[219]))
            {
                output.second.emplace_back(28, 54);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(17, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(17, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(18, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(18, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(19, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(19, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(20, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(20, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(21, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(21, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(22, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(22, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(23, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(23, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(24, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(24, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(25, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(25, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(26, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(26, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(27, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(27, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[220], y[221], y[222], y[223]))
            {
                output.second.emplace_back(28, 55);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[224], y[225], y[226], y[227]))
            {
                output.second.emplace_back(28, 56);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(17, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[68], y[69], y[70], y[71], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(17, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(18, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[72], y[73], y[74], y[75], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(18, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(19, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[76], y[77], y[78], y[79], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(19, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(20, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[80], y[81], y[82], y[83], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(20, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(21, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[84], y[85], y[86], y[87], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(21, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(22, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[88], y[89], y[90], y[91], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(22, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(23, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[92], y[93], y[94], y[95], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(23, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(24, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[96], y[97], y[98], y[99], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(24, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(25, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[100], y[101], y[102], y[103], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(25, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(26, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[104], y[105], y[106], y[107], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(26, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(27, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[108], y[109], y[110], y[111], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(27, 58);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[228], y[229], y[230], y[231]))
            {
                output.second.emplace_back(28, 57);
            }

            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[112], y[113], y[114], y[115], y[232], y[233], y[234], y[235]))
            {
                output.second.emplace_back(28, 58);
            }

            return output;
        }

        template <std::size_t rake>
        inline static bool fkcc(
            const vamp::collision::Environment<FloatVector<rake>> &environment,
            const ConfigurationBlock<rake> &x) noexcept
        {
            std::array<FloatVector<rake, 1>, 57> v;
            std::array<FloatVector<rake, 1>, 280> y;

            v[0] = sin(x[0]);
            v[1] = -v[0];
            y[4] = -0.08 * v[1];
            v[2] = cos(x[0]);
            y[5] = -0.08 * v[2];
            y[8] = -0.03 * v[1];
            y[9] = -0.03 * v[2];
            y[20] = 0.03 * v[1];
            y[21] = 0.03 * v[2];
            y[24] = 0.08 * v[1];
            y[25] = 0.08 * v[2];
            v[3] = sin(x[1]);
            v[4] = -v[3];
            v[5] = cos(x[1]);
            v[6] = 4.89663865010925e-12 * v[5];
            v[7] = v[2] * v[4] + v[1] * v[6];
            y[28] = -0.12 * v[7];
            v[6] = v[0] * v[4] + v[2] * v[6];
            y[29] = -0.12 * v[6];
            v[4] = -1. * v[5];
            y[30] = 0.333 + -0.12 * v[4];
            y[32] = -0.17 * v[7];
            y[33] = -0.17 * v[6];
            y[34] = 0.333 + -0.17 * v[4];
            v[8] = -1. * v[7] + 4.89663865010925e-12 * v[1];
            v[9] = -0.316 * v[7];
            y[36] = -0.1 * v[8] + v[9];
            v[10] = -1. * v[6] + 4.89663865010925e-12 * v[2];
            v[11] = -0.316 * v[6];
            y[37] = -0.1 * v[10] + v[11];
            v[12] = 2.39770700697438e-23 + -1. * v[4];
            v[13] = 0.333 + -0.316 * v[4];
            y[38] = -0.1 * v[12] + v[13];
            y[40] = -0.06 * v[8] + v[9];
            y[41] = -0.06 * v[10] + v[11];
            y[42] = -0.06 * v[12] + v[13];
            v[14] = 4.89663865010925e-12 * v[3];
            v[15] = v[2] * v[5] + v[1] * v[14];
            v[16] = cos(x[2]);
            v[17] = sin(x[2]);
            v[18] = 4.89663865010925e-12 * v[17];
            v[19] = v[15] * v[16] + v[7] * v[18] + v[1] * v[17];
            v[20] = -v[17];
            v[21] = 4.89663865010925e-12 * v[16];
            v[15] = v[15] * v[20] + v[7] * v[21] + v[1] * v[16];
            y[44] = 0.08 * v[19] + 0.06 * v[15] + v[9];
            v[14] = v[0] * v[5] + v[2] * v[14];
            v[5] = v[14] * v[16] + v[6] * v[18] + v[2] * v[17];
            v[14] = v[14] * v[20] + v[6] * v[21] + v[2] * v[16];
            y[45] = 0.08 * v[5] + 0.06 * v[14] + v[11];
            v[3] = -1. * v[3];
            v[18] = v[3] * v[16] + v[4] * v[18] + 4.89663865010925e-12 * v[17];
            v[3] = v[3] * v[20] + v[4] * v[21] + 4.89663865010925e-12 * v[16];
            y[46] = 0.08 * v[18] + 0.06 * v[3] + v[13];
            y[48] = 0.08 * v[19] + 0.02 * v[15] + v[9];
            y[49] = 0.08 * v[5] + 0.02 * v[14] + v[11];
            y[50] = 0.08 * v[18] + 0.02 * v[3] + v[13];
            v[21] = cos(x[3]);
            v[20] = sin(x[3]);
            v[16] = 4.89663865010925e-12 * v[20];
            v[17] = v[19] * v[21] + v[15] * v[16] + v[8] * v[20];
            v[0] = -v[20];
            v[22] = 4.89663865010925e-12 * v[21];
            v[23] = v[19] * v[0] + v[15] * v[22] + v[8] * v[21];
            v[24] = v[9] + 0.0825 * v[19];
            y[52] = -0.08 * v[17] + 0.095 * v[23] + v[24];
            v[25] = v[5] * v[21] + v[14] * v[16] + v[10] * v[20];
            v[26] = v[5] * v[0] + v[14] * v[22] + v[10] * v[21];
            v[27] = v[11] + 0.0825 * v[5];
            y[53] = -0.08 * v[25] + 0.095 * v[26] + v[27];
            v[16] = v[18] * v[21] + v[3] * v[16] + v[12] * v[20];
            v[22] = v[18] * v[0] + v[3] * v[22] + v[12] * v[21];
            v[0] = v[13] + 0.0825 * v[18];
            y[54] = -0.08 * v[16] + 0.095 * v[22] + v[0];
            v[21] = -1. * v[15] + 4.89663865010925e-12 * v[8];
            y[56] = 0.02 * v[21] + v[24];
            v[20] = -1. * v[14] + 4.89663865010925e-12 * v[10];
            y[57] = 0.02 * v[20] + v[27];
            v[28] = -1. * v[3] + 4.89663865010925e-12 * v[12];
            y[58] = 0.02 * v[28] + v[0];
            y[60] = 0.06 * v[21] + v[24];
            y[61] = 0.06 * v[20] + v[27];
            y[62] = 0.06 * v[28] + v[0];
            y[64] = -0.08 * v[17] + 0.06 * v[23] + v[24];
            y[65] = -0.08 * v[25] + 0.06 * v[26] + v[27];
            y[66] = -0.08 * v[16] + 0.06 * v[22] + v[0];
            v[29] = sin(x[4]);
            v[30] = -v[29];
            v[31] = cos(x[4]);
            v[32] = 4.89663865010925e-12 * v[31];
            v[33] = -1. * v[31];
            v[34] = v[17] * v[30] + v[23] * v[32] + v[21] * v[33];
            y[116] = v[24] + -0.0825 * v[17] + 0.384 * v[23];
            y[68] = 0.055 * v[34] + y[116];
            v[35] = v[25] * v[30] + v[26] * v[32] + v[20] * v[33];
            y[117] = v[27] + -0.0825 * v[25] + 0.384 * v[26];
            y[69] = 0.055 * v[35] + y[117];
            v[33] = v[16] * v[30] + v[22] * v[32] + v[28] * v[33];
            y[118] = v[0] + -0.0825 * v[16] + 0.384 * v[22];
            y[70] = 0.055 * v[33] + y[118];
            y[72] = 0.075 * v[34] + y[116];
            y[73] = 0.075 * v[35] + y[117];
            y[74] = 0.075 * v[33] + y[118];
            v[32] = v[23] + 4.89663865010925e-12 * v[21];
            y[76] = -0.22 * v[32] + y[116];
            v[30] = v[26] + 4.89663865010925e-12 * v[20];
            y[77] = -0.22 * v[30] + y[117];
            v[36] = v[22] + 4.89663865010925e-12 * v[28];
            y[78] = -0.22 * v[36] + y[118];
            y[80] = 0.05 * v[34] + -0.18 * v[32] + y[116];
            y[81] = 0.05 * v[35] + -0.18 * v[30] + y[117];
            y[82] = 0.05 * v[33] + -0.18 * v[36] + y[118];
            v[37] = 4.89663865010925e-12 * v[29];
            v[29] = -1. * v[29];
            v[38] = v[17] * v[31] + v[23] * v[37] + v[21] * v[29];
            y[84] = 0.01 * v[38] + 0.08 * v[34] + -0.14 * v[32] + y[116];
            v[39] = v[25] * v[31] + v[26] * v[37] + v[20] * v[29];
            y[85] = 0.01 * v[39] + 0.08 * v[35] + -0.14 * v[30] + y[117];
            v[29] = v[16] * v[31] + v[22] * v[37] + v[28] * v[29];
            y[86] = 0.01 * v[29] + 0.08 * v[33] + -0.14 * v[36] + y[118];
            y[88] = 0.01 * v[38] + 0.085 * v[34] + -0.11 * v[32] + y[116];
            y[89] = 0.01 * v[39] + 0.085 * v[35] + -0.11 * v[30] + y[117];
            y[90] = 0.01 * v[29] + 0.085 * v[33] + -0.11 * v[36] + y[118];
            y[92] = 0.01 * v[38] + 0.09 * v[34] + -0.08 * v[32] + y[116];
            y[93] = 0.01 * v[39] + 0.09 * v[35] + -0.08 * v[30] + y[117];
            y[94] = 0.01 * v[29] + 0.09 * v[33] + -0.08 * v[36] + y[118];
            y[96] = 0.01 * v[38] + 0.095 * v[34] + -0.05 * v[32] + y[116];
            y[97] = 0.01 * v[39] + 0.095 * v[35] + -0.05 * v[30] + y[117];
            y[98] = 0.01 * v[29] + 0.095 * v[33] + -0.05 * v[36] + y[118];
            y[100] = -0.01 * v[38] + 0.08 * v[34] + -0.14 * v[32] + y[116];
            y[101] = -0.01 * v[39] + 0.08 * v[35] + -0.14 * v[30] + y[117];
            y[102] = -0.01 * v[29] + 0.08 * v[33] + -0.14 * v[36] + y[118];
            y[104] = -0.01 * v[38] + 0.085 * v[34] + -0.11 * v[32] + y[116];
            y[105] = -0.01 * v[39] + 0.085 * v[35] + -0.11 * v[30] + y[117];
            y[106] = -0.01 * v[29] + 0.085 * v[33] + -0.11 * v[36] + y[118];
            y[108] = -0.01 * v[38] + 0.09 * v[34] + -0.08 * v[32] + y[116];
            y[109] = -0.01 * v[39] + 0.09 * v[35] + -0.08 * v[30] + y[117];
            y[110] = -0.01 * v[29] + 0.09 * v[33] + -0.08 * v[36] + y[118];
            y[112] = -0.01 * v[38] + 0.095 * v[34] + -0.05 * v[32] + y[116];
            y[113] = -0.01 * v[39] + 0.095 * v[35] + -0.05 * v[30] + y[117];
            y[114] = -0.01 * v[29] + 0.095 * v[33] + -0.05 * v[36] + y[118];
            v[37] = cos(x[5]);
            v[31] = sin(x[5]);
            v[40] = 4.89663865010925e-12 * v[31];
            v[41] = v[38] * v[37] + v[34] * v[40] + v[32] * v[31];
            v[42] = -v[31];
            v[43] = 4.89663865010925e-12 * v[37];
            v[38] = v[38] * v[42] + v[34] * v[43] + v[32] * v[37];
            y[120] = 0.08 * v[41] + -0.01 * v[38] + y[116];
            v[44] = v[39] * v[37] + v[35] * v[40] + v[30] * v[31];
            v[39] = v[39] * v[42] + v[35] * v[43] + v[30] * v[37];
            y[121] = 0.08 * v[44] + -0.01 * v[39] + y[117];
            v[40] = v[29] * v[37] + v[33] * v[40] + v[36] * v[31];
            v[43] = v[29] * v[42] + v[33] * v[43] + v[36] * v[37];
            y[122] = 0.08 * v[40] + -0.01 * v[43] + y[118];
            y[124] = 0.08 * v[41] + 0.035 * v[38] + y[116];
            y[125] = 0.08 * v[44] + 0.035 * v[39] + y[117];
            y[126] = 0.08 * v[40] + 0.035 * v[43] + y[118];
            v[42] = -1. * v[34] + 4.89663865010925e-12 * v[32];
            v[37] = -1. * v[38] + 4.89663865010925e-12 * v[42];
            v[29] = y[116] + 0.088 * v[41];
            y[128] = 0.07 * v[37] + v[29];
            v[31] = -1. * v[35] + 4.89663865010925e-12 * v[30];
            v[45] = -1. * v[39] + 4.89663865010925e-12 * v[31];
            v[46] = y[117] + 0.088 * v[44];
            y[129] = 0.07 * v[45] + v[46];
            v[47] = -1. * v[33] + 4.89663865010925e-12 * v[36];
            v[48] = -1. * v[43] + 4.89663865010925e-12 * v[47];
            v[49] = y[118] + 0.088 * v[40];
            y[130] = 0.07 * v[48] + v[49];
            v[50] = cos(x[6]);
            v[51] = sin(x[6]);
            v[52] = 4.89663865010925e-12 * v[51];
            v[53] = v[41] * v[50] + v[38] * v[52] + v[42] * v[51];
            v[54] = -v[51];
            v[55] = 4.89663865010925e-12 * v[50];
            v[42] = v[41] * v[54] + v[38] * v[55] + v[42] * v[50];
            y[132] = 0.02 * v[53] + 0.04 * v[42] + 0.08 * v[37] + v[29];
            v[56] = v[44] * v[50] + v[39] * v[52] + v[31] * v[51];
            v[31] = v[44] * v[54] + v[39] * v[55] + v[31] * v[50];
            y[133] = 0.02 * v[56] + 0.04 * v[31] + 0.08 * v[45] + v[46];
            v[52] = v[40] * v[50] + v[43] * v[52] + v[47] * v[51];
            v[55] = v[40] * v[54] + v[43] * v[55] + v[47] * v[50];
            y[134] = 0.02 * v[52] + 0.04 * v[55] + 0.08 * v[48] + v[49];
            y[136] = 0.04 * v[53] + 0.02 * v[42] + 0.08 * v[37] + v[29];
            y[137] = 0.04 * v[56] + 0.02 * v[31] + 0.08 * v[45] + v[46];
            y[138] = 0.04 * v[52] + 0.02 * v[55] + 0.08 * v[48] + v[49];
            y[140] = 0.04 * v[53] + 0.06 * v[42] + 0.085 * v[37] + v[29];
            y[141] = 0.04 * v[56] + 0.06 * v[31] + 0.085 * v[45] + v[46];
            y[142] = 0.04 * v[52] + 0.06 * v[55] + 0.085 * v[48] + v[49];
            y[144] = 0.06 * v[53] + 0.04 * v[42] + 0.085 * v[37] + v[29];
            y[145] = 0.06 * v[56] + 0.04 * v[31] + 0.085 * v[45] + v[46];
            y[146] = 0.06 * v[52] + 0.04 * v[55] + 0.085 * v[48] + v[49];
            y[148] = -0.0530330085889673 * v[53] + -0.0530330085890148 * v[42] + 0.117 * v[37] + v[29];
            y[149] = -0.0530330085889673 * v[56] + -0.0530330085890148 * v[31] + 0.117 * v[45] + v[46];
            y[150] = -0.0530330085889673 * v[52] + -0.0530330085890148 * v[55] + 0.117 * v[48] + v[49];
            y[152] = -0.0318198051533804 * v[53] + -0.0318198051534089 * v[42] + 0.117 * v[37] + v[29];
            y[153] = -0.0318198051533804 * v[56] + -0.0318198051534089 * v[31] + 0.117 * v[45] + v[46];
            y[154] = -0.0318198051533804 * v[52] + -0.0318198051534089 * v[55] + 0.117 * v[48] + v[49];
            y[156] = -0.0106066017177935 * v[53] + -0.010606601717803 * v[42] + 0.117 * v[37] + v[29];
            y[157] = -0.0106066017177935 * v[56] + -0.010606601717803 * v[31] + 0.117 * v[45] + v[46];
            y[158] = -0.0106066017177935 * v[52] + -0.010606601717803 * v[55] + 0.117 * v[48] + v[49];
            y[160] = 0.0106066017177935 * v[53] + 0.010606601717803 * v[42] + 0.117 * v[37] + v[29];
            y[161] = 0.0106066017177935 * v[56] + 0.010606601717803 * v[31] + 0.117 * v[45] + v[46];
            y[162] = 0.0106066017177935 * v[52] + 0.010606601717803 * v[55] + 0.117 * v[48] + v[49];
            y[164] = 0.0318198051533804 * v[53] + 0.0318198051534089 * v[42] + 0.117 * v[37] + v[29];
            y[165] = 0.0318198051533804 * v[56] + 0.0318198051534089 * v[31] + 0.117 * v[45] + v[46];
            y[166] = 0.0318198051533804 * v[52] + 0.0318198051534089 * v[55] + 0.117 * v[48] + v[49];
            y[168] = 0.0530330085889673 * v[53] + 0.0530330085890148 * v[42] + 0.117 * v[37] + v[29];
            y[169] = 0.0530330085889673 * v[56] + 0.0530330085890148 * v[31] + 0.117 * v[45] + v[46];
            y[170] = 0.0530330085889673 * v[52] + 0.0530330085890148 * v[55] + 0.117 * v[48] + v[49];
            y[172] = -0.0530330085889673 * v[53] + -0.0530330085890148 * v[42] + 0.137 * v[37] + v[29];
            y[173] = -0.0530330085889673 * v[56] + -0.0530330085890148 * v[31] + 0.137 * v[45] + v[46];
            y[174] = -0.0530330085889673 * v[52] + -0.0530330085890148 * v[55] + 0.137 * v[48] + v[49];
            y[176] = -0.0318198051533804 * v[53] + -0.0318198051534089 * v[42] + 0.137 * v[37] + v[29];
            y[177] = -0.0318198051533804 * v[56] + -0.0318198051534089 * v[31] + 0.137 * v[45] + v[46];
            y[178] = -0.0318198051533804 * v[52] + -0.0318198051534089 * v[55] + 0.137 * v[48] + v[49];
            y[180] = -0.0106066017177935 * v[53] + -0.010606601717803 * v[42] + 0.137 * v[37] + v[29];
            y[181] = -0.0106066017177935 * v[56] + -0.010606601717803 * v[31] + 0.137 * v[45] + v[46];
            y[182] = -0.0106066017177935 * v[52] + -0.010606601717803 * v[55] + 0.137 * v[48] + v[49];
            y[184] = 0.0106066017177935 * v[53] + 0.010606601717803 * v[42] + 0.137 * v[37] + v[29];
            y[185] = 0.0106066017177935 * v[56] + 0.010606601717803 * v[31] + 0.137 * v[45] + v[46];
            y[186] = 0.0106066017177935 * v[52] + 0.010606601717803 * v[55] + 0.137 * v[48] + v[49];
            y[188] = 0.0318198051533804 * v[53] + 0.0318198051534089 * v[42] + 0.137 * v[37] + v[29];
            y[189] = 0.0318198051533804 * v[56] + 0.0318198051534089 * v[31] + 0.137 * v[45] + v[46];
            y[190] = 0.0318198051533804 * v[52] + 0.0318198051534089 * v[55] + 0.137 * v[48] + v[49];
            y[192] = 0.0530330085889673 * v[53] + 0.0530330085890148 * v[42] + 0.137 * v[37] + v[29];
            y[193] = 0.0530330085889673 * v[56] + 0.0530330085890148 * v[31] + 0.137 * v[45] + v[46];
            y[194] = 0.0530330085889673 * v[52] + 0.0530330085890148 * v[55] + 0.137 * v[48] + v[49];
            y[196] = -0.0530330085889673 * v[53] + -0.0530330085890148 * v[42] + 0.157 * v[37] + v[29];
            y[197] = -0.0530330085889673 * v[56] + -0.0530330085890148 * v[31] + 0.157 * v[45] + v[46];
            y[198] = -0.0530330085889673 * v[52] + -0.0530330085890148 * v[55] + 0.157 * v[48] + v[49];
            y[200] = -0.0318198051533804 * v[53] + -0.0318198051534089 * v[42] + 0.157 * v[37] + v[29];
            y[201] = -0.0318198051533804 * v[56] + -0.0318198051534089 * v[31] + 0.157 * v[45] + v[46];
            y[202] = -0.0318198051533804 * v[52] + -0.0318198051534089 * v[55] + 0.157 * v[48] + v[49];
            y[204] = -0.0106066017177935 * v[53] + -0.010606601717803 * v[42] + 0.157 * v[37] + v[29];
            y[205] = -0.0106066017177935 * v[56] + -0.010606601717803 * v[31] + 0.157 * v[45] + v[46];
            y[206] = -0.0106066017177935 * v[52] + -0.010606601717803 * v[55] + 0.157 * v[48] + v[49];
            y[208] = 0.0106066017177935 * v[53] + 0.010606601717803 * v[42] + 0.157 * v[37] + v[29];
            y[209] = 0.0106066017177935 * v[56] + 0.010606601717803 * v[31] + 0.157 * v[45] + v[46];
            y[210] = 0.0106066017177935 * v[52] + 0.010606601717803 * v[55] + 0.157 * v[48] + v[49];
            y[212] = 0.0318198051533804 * v[53] + 0.0318198051534089 * v[42] + 0.157 * v[37] + v[29];
            y[213] = 0.0318198051533804 * v[56] + 0.0318198051534089 * v[31] + 0.157 * v[45] + v[46];
            y[214] = 0.0318198051533804 * v[52] + 0.0318198051534089 * v[55] + 0.157 * v[48] + v[49];
            y[216] = 0.0530330085889673 * v[53] + 0.0530330085890148 * v[42] + 0.157 * v[37] + v[29];
            y[217] = 0.0530330085889673 * v[56] + 0.0530330085890148 * v[31] + 0.157 * v[45] + v[46];
            y[218] = 0.0530330085889673 * v[52] + 0.0530330085890148 * v[55] + 0.157 * v[48] + v[49];
            y[220] = 0.0565685424948984 * v[53] + 0.0565685424949492 * v[42] + 0.1874 * v[37] + v[29];
            y[221] = 0.0565685424948984 * v[56] + 0.0565685424949492 * v[31] + 0.1874 * v[45] + v[46];
            y[222] = 0.0565685424948984 * v[52] + 0.0565685424949492 * v[55] + 0.1874 * v[48] + v[49];
            y[224] = 0.0516187950265948 * v[53] + 0.0516187950266411 * v[42] + 0.2094 * v[37] + v[29];
            y[225] = 0.0516187950265948 * v[56] + 0.0516187950266411 * v[31] + 0.2094 * v[45] + v[46];
            y[226] = 0.0516187950265948 * v[52] + 0.0516187950266411 * v[55] + 0.2094 * v[48] + v[49];
            y[228] = -0.0565685424948984 * v[53] + -0.0565685424949492 * v[42] + 0.1874 * v[37] + v[29];
            y[229] = -0.0565685424948984 * v[56] + -0.0565685424949492 * v[31] + 0.1874 * v[45] + v[46];
            y[230] = -0.0565685424948984 * v[52] + -0.0565685424949492 * v[55] + 0.1874 * v[48] + v[49];
            y[232] = -0.0516187950265948 * v[53] + -0.0516187950266411 * v[42] + 0.2094 * v[37] + v[29];
            y[233] = -0.0516187950265948 * v[56] + -0.0516187950266411 * v[31] + 0.2094 * v[45] + v[46];
            y[234] = -0.0516187950265948 * v[52] + -0.0516187950266411 * v[55] + 0.2094 * v[48] + v[49];
            y[240] = -0.0399999991059303 * v[1];
            y[241] = -0.0399999991059303 * v[2];
            y[244] = -0.0850000008940697 * v[7] + 0.0399999991059303 * v[1];
            y[245] = -0.0850000008940697 * v[6] + 0.0399999991059303 * v[2];
            y[246] = 0.333 + 1.95865541626434e-13 + -0.0850000008940697 * v[4];
            y[248] =
                0.0385857857763767 * v[19] + 0.0289393402636051 * v[15] + -0.0517677664756775 * v[8] + v[9];
            y[249] =
                0.0385857857763767 * v[5] + 0.0289393402636051 * v[14] + -0.0517677664756775 * v[10] + v[11];
            y[250] =
                0.0385857857763767 * v[18] + 0.0289393402636051 * v[3] + -0.0517677664756775 * v[12] + v[13];
            y[252] =
                -0.041449997574091 * v[17] + 0.0492218732833862 * v[23] + 0.0289125014096498 * v[21] + v[24];
            y[253] =
                -0.041449997574091 * v[25] + 0.0492218732833862 * v[26] + 0.0289125014096498 * v[20] + v[27];
            y[254] =
                -0.041449997574091 * v[16] + 0.0492218732833862 * v[22] + 0.0289125014096498 * v[28] + v[0];
            y[256] = 0.0375000014901161 * v[34] + -0.109999999403954 * v[32] + y[116];
            y[257] = 0.0375000014901161 * v[35] + -0.109999999403954 * v[30] + y[117];
            y[258] = 0.0375000014901161 * v[33] + -0.109999999403954 * v[36] + y[118];
            y[260] = 0.0424309633672237 * v[41] + 0.0144476993009448 * v[38] + y[116];
            y[261] = 0.0424309633672237 * v[44] + 0.0144476993009448 * v[39] + y[117];
            y[262] = 0.0424309633672237 * v[40] + 0.0144476993009448 * v[43] + y[118];
            y[264] =
                0.0153005504980683 * v[53] + 0.0153005504980683 * v[42] + 0.0745901614427567 * v[37] + v[29];
            y[265] =
                0.0153005504980683 * v[56] + 0.0153005504980683 * v[31] + 0.0745901614427567 * v[45] + v[46];
            y[266] =
                0.0153005504980683 * v[52] + 0.0153005504980683 * v[55] + 0.0745901614427567 * v[48] + v[49];
            y[268] = -5.20417042793042e-18 * v[53] + -2.16840434497101e-18 * v[42] +
                     0.129201397299767 * v[37] + v[29];
            y[269] = -5.20417042793042e-18 * v[56] + -2.16840434497101e-18 * v[31] +
                     0.129201397299767 * v[45] + v[46];
            y[270] = -5.20417042793042e-18 * v[52] + -2.16840434497101e-18 * v[55] +
                     0.129201397299767 * v[48] + v[49];
            y[272] =
                0.0540936700999737 * v[53] + 0.0540936700999737 * v[42] + 0.198400005698204 * v[37] + v[29];
            y[273] =
                0.0540936700999737 * v[56] + 0.0540936700999737 * v[31] + 0.198400005698204 * v[45] + v[46];
            y[274] =
                0.0540936700999737 * v[52] + 0.0540936700999737 * v[55] + 0.198400005698204 * v[48] + v[49];
            y[276] =
                -0.0540936700999737 * v[53] + -0.0540936700999737 * v[42] + 0.198400005698204 * v[37] + v[29];
            y[277] =
                -0.0540936700999737 * v[56] + -0.0540936700999737 * v[31] + 0.198400005698204 * v[45] + v[46];
            y[278] =
                -0.0540936700999737 * v[52] + -0.0540936700999737 * v[55] + 0.198400005698204 * v[48] + v[49];
            // dependent variables without operations
            y[0] = 0.;
            y[1] = 0.;
            y[2] = 0.05;
            y[3] = 0.0799999982118607;
            y[6] = 0.333;
            y[7] = 0.0599999986588955;
            y[10] = 0.333;
            y[11] = 0.0599999986588955;
            y[12] = 0.;
            y[13] = 0.;
            y[14] = 0.213;
            y[15] = 0.0599999986588955;
            y[16] = 0.;
            y[17] = 0.;
            y[18] = 0.163;
            y[19] = 0.0599999986588955;
            y[22] = 0.333000000000147;
            y[23] = 0.0599999986588955;
            y[26] = 0.333000000000392;
            y[27] = 0.0599999986588955;
            y[31] = 0.0599999986588955;
            y[35] = 0.0599999986588955;
            y[39] = 0.0599999986588955;
            y[43] = 0.0500000007450581;
            y[47] = 0.0549999997019768;
            y[51] = 0.0549999997019768;
            y[55] = 0.0599999986588955;
            y[59] = 0.0549999997019768;
            y[63] = 0.0549999997019768;
            y[67] = 0.0549999997019768;
            y[71] = 0.0599999986588955;
            y[75] = 0.0599999986588955;
            y[79] = 0.0599999986588955;
            y[83] = 0.0500000007450581;
            y[87] = 0.025000000372529;
            y[91] = 0.025000000372529;
            y[95] = 0.025000000372529;
            y[99] = 0.025000000372529;
            y[103] = 0.025000000372529;
            y[107] = 0.025000000372529;
            y[111] = 0.025000000372529;
            y[115] = 0.025000000372529;
            y[119] = 0.0500000007450581;
            y[123] = 0.0500000007450581;
            y[127] = 0.0520000010728836;
            y[131] = 0.0500000007450581;
            y[135] = 0.025000000372529;
            y[139] = 0.025000000372529;
            y[143] = 0.0199999995529652;
            y[147] = 0.0199999995529652;
            y[151] = 0.0280000008642673;
            y[155] = 0.0280000008642673;
            y[159] = 0.0280000008642673;
            y[163] = 0.0280000008642673;
            y[167] = 0.0280000008642673;
            y[171] = 0.0280000008642673;
            y[175] = 0.0260000005364418;
            y[179] = 0.0260000005364418;
            y[183] = 0.0260000005364418;
            y[187] = 0.0260000005364418;
            y[191] = 0.0260000005364418;
            y[195] = 0.0260000005364418;
            y[199] = 0.0240000002086163;
            y[203] = 0.0240000002086163;
            y[207] = 0.0240000002086163;
            y[211] = 0.0240000002086163;
            y[215] = 0.0240000002086163;
            y[219] = 0.0240000002086163;
            y[223] = 0.0120000001043081;
            y[227] = 0.0120000001043081;
            y[231] = 0.0120000001043081;
            y[235] = 0.0120000001043081;
            y[236] = 0.;
            y[237] = 0.;
            y[238] = 0.0500000007450581;
            y[239] = 0.0799999982118607;
            y[242] = 0.24799999910593;
            y[243] = 0.153941467404366;
            y[247] = 0.153941467404366;
            y[251] = 0.128210678696632;
            y[255] = 0.126465573906898;
            y[259] = 0.176216393709183;
            y[263] = 0.0948232412338257;
            y[267] = 0.0721197500824928;
            y[271] = 0.103986009955406;
            y[275] = 0.0235433969646692;
            y[279] = 0.0235433969646692;

            //
            // environment vs. robot collisions
            //

            // panda_rightfinger
            if (sphere_environment_in_collision(environment, y[276], y[277], y[278], y[279]))
            {
                if (sphere_environment_in_collision(environment, y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_leftfinger
            if (sphere_environment_in_collision(environment, y[272], y[273], y[274], y[275]))
            {
                if (sphere_environment_in_collision(environment, y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_hand
            if (sphere_environment_in_collision(environment, y[268], y[269], y[270], y[271]))
            {
                if (sphere_environment_in_collision(environment, y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link7
            if (sphere_environment_in_collision(environment, y[264], y[265], y[266], y[267]))
            {
                if (sphere_environment_in_collision(environment, y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link6
            if (sphere_environment_in_collision(environment, y[260], y[261], y[262], y[263]))
            {
                if (sphere_environment_in_collision(environment, y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link5
            if (sphere_environment_in_collision(environment, y[256], y[257], y[258], y[259]))
            {
                if (sphere_environment_in_collision(environment, y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link4
            if (sphere_environment_in_collision(environment, y[252], y[253], y[254], y[255]))
            {
                if (sphere_environment_in_collision(environment, y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[60], y[61], y[62], y[63]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[64], y[65], y[66], y[67]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link3
            if (sphere_environment_in_collision(environment, y[248], y[249], y[250], y[251]))
            {
                if (sphere_environment_in_collision(environment, y[36], y[37], y[38], y[39]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[48], y[49], y[50], y[51]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link2
            if (sphere_environment_in_collision(environment, y[244], y[245], y[246], y[247]))
            {
                if (sphere_environment_in_collision(environment, y[20], y[21], y[22], y[23]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[24], y[25], y[26], y[27]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[32], y[33], y[34], y[35]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link1
            if (sphere_environment_in_collision(environment, y[240], y[241], y[242], y[243]))
            {
                if (sphere_environment_in_collision(environment, y[4], y[5], y[6], y[7]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[8], y[9], y[10], y[11]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[12], y[13], y[14], y[15]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[16], y[17], y[18], y[19]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link0
            if (sphere_environment_in_collision(environment, y[236], y[237], y[238], y[239]))
            {
                if (sphere_environment_in_collision(environment, y[0], y[1], y[2], y[3]))
                {
                    return false;
                }
            }

            //
            // robot self-collisions
            //

            // panda_link0 vs. panda_link5
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[256], y[257], y[258], y[259]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // panda_link0 vs. panda_link6
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[260], y[261], y[262], y[263]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            // panda_link0 vs. panda_link7
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[264], y[265], y[266], y[267]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // panda_link0 vs. panda_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[268], y[269], y[270], y[271]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // panda_link0 vs. panda_leftfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[272], y[273], y[274], y[275]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // panda_link0 vs. panda_rightfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[276], y[277], y[278], y[279]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_link5
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[256], y[257], y[258], y[259]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_link6
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[260], y[261], y[262], y[263]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_link7
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[264], y[265], y[266], y[267]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[268], y[269], y[270], y[271]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_leftfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[272], y[273], y[274], y[275]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_rightfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[276], y[277], y[278], y[279]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // panda_link2 vs. panda_link5
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[256], y[257], y[258], y[259]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // panda_link2 vs. panda_link7
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[264], y[265], y[266], y[267]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // panda_link2 vs. panda_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[268], y[269], y[270], y[271]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // panda_link2 vs. panda_leftfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[272], y[273], y[274], y[275]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // panda_link2 vs. panda_rightfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[276], y[277], y[278], y[279]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // panda_link5 vs. panda_link7
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[264], y[265], y[266], y[267]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // panda_link5 vs. panda_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[268], y[269], y[270], y[271]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // panda_link5 vs. panda_leftfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[272], y[273], y[274], y[275]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // panda_link5 vs. panda_rightfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[276], y[277], y[278], y[279]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            return true;
        }

        template <std::size_t rake>
        inline static bool fkcc_attach(
            const vamp::collision::Environment<FloatVector<rake>> &environment,
            const ConfigurationBlock<rake> &x) noexcept
        {
            std::array<FloatVector<rake, 1>, 54> v;
            std::array<FloatVector<rake, 1>, 292> y;

            v[0] = sin(x[0]);
            v[1] = -v[0];
            y[4] = -0.08 * v[1];
            v[2] = cos(x[0]);
            y[5] = -0.08 * v[2];
            y[8] = -0.03 * v[1];
            y[9] = -0.03 * v[2];
            y[20] = 0.03 * v[1];
            y[21] = 0.03 * v[2];
            y[24] = 0.08 * v[1];
            y[25] = 0.08 * v[2];
            v[3] = sin(x[1]);
            v[4] = -v[3];
            v[5] = cos(x[1]);
            v[6] = 4.89663865010925e-12 * v[5];
            v[7] = v[2] * v[4] + v[1] * v[6];
            y[28] = -0.12 * v[7];
            v[6] = v[0] * v[4] + v[2] * v[6];
            y[29] = -0.12 * v[6];
            v[4] = -1. * v[5];
            y[30] = 0.333 + -0.12 * v[4];
            y[32] = -0.17 * v[7];
            y[33] = -0.17 * v[6];
            y[34] = 0.333 + -0.17 * v[4];
            v[8] = -1. * v[7] + 4.89663865010925e-12 * v[1];
            v[9] = -0.316 * v[7];
            y[36] = -0.1 * v[8] + v[9];
            v[10] = -1. * v[6] + 4.89663865010925e-12 * v[2];
            v[11] = -0.316 * v[6];
            y[37] = -0.1 * v[10] + v[11];
            v[12] = 2.39770700697438e-23 + -1. * v[4];
            v[13] = 0.333 + -0.316 * v[4];
            y[38] = -0.1 * v[12] + v[13];
            y[40] = -0.06 * v[8] + v[9];
            y[41] = -0.06 * v[10] + v[11];
            y[42] = -0.06 * v[12] + v[13];
            v[14] = 4.89663865010925e-12 * v[3];
            v[15] = v[2] * v[5] + v[1] * v[14];
            v[16] = cos(x[2]);
            v[17] = sin(x[2]);
            v[18] = 4.89663865010925e-12 * v[17];
            v[19] = v[15] * v[16] + v[7] * v[18] + v[1] * v[17];
            v[20] = -v[17];
            v[21] = 4.89663865010925e-12 * v[16];
            v[15] = v[15] * v[20] + v[7] * v[21] + v[1] * v[16];
            y[44] = 0.08 * v[19] + 0.06 * v[15] + v[9];
            v[14] = v[0] * v[5] + v[2] * v[14];
            v[5] = v[14] * v[16] + v[6] * v[18] + v[2] * v[17];
            v[14] = v[14] * v[20] + v[6] * v[21] + v[2] * v[16];
            y[45] = 0.08 * v[5] + 0.06 * v[14] + v[11];
            v[3] = -1. * v[3];
            v[18] = v[3] * v[16] + v[4] * v[18] + 4.89663865010925e-12 * v[17];
            v[3] = v[3] * v[20] + v[4] * v[21] + 4.89663865010925e-12 * v[16];
            y[46] = 0.08 * v[18] + 0.06 * v[3] + v[13];
            y[48] = 0.08 * v[19] + 0.02 * v[15] + v[9];
            y[49] = 0.08 * v[5] + 0.02 * v[14] + v[11];
            y[50] = 0.08 * v[18] + 0.02 * v[3] + v[13];
            v[21] = cos(x[3]);
            v[20] = sin(x[3]);
            v[16] = 4.89663865010925e-12 * v[20];
            v[17] = v[19] * v[21] + v[15] * v[16] + v[8] * v[20];
            v[0] = -v[20];
            v[22] = 4.89663865010925e-12 * v[21];
            v[23] = v[19] * v[0] + v[15] * v[22] + v[8] * v[21];
            v[24] = v[9] + 0.0825 * v[19];
            y[52] = -0.08 * v[17] + 0.095 * v[23] + v[24];
            v[25] = v[5] * v[21] + v[14] * v[16] + v[10] * v[20];
            v[26] = v[5] * v[0] + v[14] * v[22] + v[10] * v[21];
            v[27] = v[11] + 0.0825 * v[5];
            y[53] = -0.08 * v[25] + 0.095 * v[26] + v[27];
            v[16] = v[18] * v[21] + v[3] * v[16] + v[12] * v[20];
            v[22] = v[18] * v[0] + v[3] * v[22] + v[12] * v[21];
            v[0] = v[13] + 0.0825 * v[18];
            y[54] = -0.08 * v[16] + 0.095 * v[22] + v[0];
            v[21] = -1. * v[15] + 4.89663865010925e-12 * v[8];
            y[56] = 0.02 * v[21] + v[24];
            v[20] = -1. * v[14] + 4.89663865010925e-12 * v[10];
            y[57] = 0.02 * v[20] + v[27];
            v[28] = -1. * v[3] + 4.89663865010925e-12 * v[12];
            y[58] = 0.02 * v[28] + v[0];
            y[60] = 0.06 * v[21] + v[24];
            y[61] = 0.06 * v[20] + v[27];
            y[62] = 0.06 * v[28] + v[0];
            y[64] = -0.08 * v[17] + 0.06 * v[23] + v[24];
            y[65] = -0.08 * v[25] + 0.06 * v[26] + v[27];
            y[66] = -0.08 * v[16] + 0.06 * v[22] + v[0];
            v[29] = sin(x[4]);
            v[30] = -v[29];
            v[31] = cos(x[4]);
            v[32] = 4.89663865010925e-12 * v[31];
            v[33] = -1. * v[31];
            v[34] = v[17] * v[30] + v[23] * v[32] + v[21] * v[33];
            y[116] = v[24] + -0.0825 * v[17] + 0.384 * v[23];
            y[68] = 0.055 * v[34] + y[116];
            v[35] = v[25] * v[30] + v[26] * v[32] + v[20] * v[33];
            y[117] = v[27] + -0.0825 * v[25] + 0.384 * v[26];
            y[69] = 0.055 * v[35] + y[117];
            v[33] = v[16] * v[30] + v[22] * v[32] + v[28] * v[33];
            y[118] = v[0] + -0.0825 * v[16] + 0.384 * v[22];
            y[70] = 0.055 * v[33] + y[118];
            y[72] = 0.075 * v[34] + y[116];
            y[73] = 0.075 * v[35] + y[117];
            y[74] = 0.075 * v[33] + y[118];
            v[32] = v[23] + 4.89663865010925e-12 * v[21];
            y[76] = -0.22 * v[32] + y[116];
            v[30] = v[26] + 4.89663865010925e-12 * v[20];
            y[77] = -0.22 * v[30] + y[117];
            v[36] = v[22] + 4.89663865010925e-12 * v[28];
            y[78] = -0.22 * v[36] + y[118];
            y[80] = 0.05 * v[34] + -0.18 * v[32] + y[116];
            y[81] = 0.05 * v[35] + -0.18 * v[30] + y[117];
            y[82] = 0.05 * v[33] + -0.18 * v[36] + y[118];
            v[37] = 4.89663865010925e-12 * v[29];
            v[29] = -1. * v[29];
            v[38] = v[17] * v[31] + v[23] * v[37] + v[21] * v[29];
            y[84] = 0.01 * v[38] + 0.08 * v[34] + -0.14 * v[32] + y[116];
            v[39] = v[25] * v[31] + v[26] * v[37] + v[20] * v[29];
            y[85] = 0.01 * v[39] + 0.08 * v[35] + -0.14 * v[30] + y[117];
            v[29] = v[16] * v[31] + v[22] * v[37] + v[28] * v[29];
            y[86] = 0.01 * v[29] + 0.08 * v[33] + -0.14 * v[36] + y[118];
            y[88] = 0.01 * v[38] + 0.085 * v[34] + -0.11 * v[32] + y[116];
            y[89] = 0.01 * v[39] + 0.085 * v[35] + -0.11 * v[30] + y[117];
            y[90] = 0.01 * v[29] + 0.085 * v[33] + -0.11 * v[36] + y[118];
            y[92] = 0.01 * v[38] + 0.09 * v[34] + -0.08 * v[32] + y[116];
            y[93] = 0.01 * v[39] + 0.09 * v[35] + -0.08 * v[30] + y[117];
            y[94] = 0.01 * v[29] + 0.09 * v[33] + -0.08 * v[36] + y[118];
            y[96] = 0.01 * v[38] + 0.095 * v[34] + -0.05 * v[32] + y[116];
            y[97] = 0.01 * v[39] + 0.095 * v[35] + -0.05 * v[30] + y[117];
            y[98] = 0.01 * v[29] + 0.095 * v[33] + -0.05 * v[36] + y[118];
            y[100] = -0.01 * v[38] + 0.08 * v[34] + -0.14 * v[32] + y[116];
            y[101] = -0.01 * v[39] + 0.08 * v[35] + -0.14 * v[30] + y[117];
            y[102] = -0.01 * v[29] + 0.08 * v[33] + -0.14 * v[36] + y[118];
            y[104] = -0.01 * v[38] + 0.085 * v[34] + -0.11 * v[32] + y[116];
            y[105] = -0.01 * v[39] + 0.085 * v[35] + -0.11 * v[30] + y[117];
            y[106] = -0.01 * v[29] + 0.085 * v[33] + -0.11 * v[36] + y[118];
            y[108] = -0.01 * v[38] + 0.09 * v[34] + -0.08 * v[32] + y[116];
            y[109] = -0.01 * v[39] + 0.09 * v[35] + -0.08 * v[30] + y[117];
            y[110] = -0.01 * v[29] + 0.09 * v[33] + -0.08 * v[36] + y[118];
            y[112] = -0.01 * v[38] + 0.095 * v[34] + -0.05 * v[32] + y[116];
            y[113] = -0.01 * v[39] + 0.095 * v[35] + -0.05 * v[30] + y[117];
            y[114] = -0.01 * v[29] + 0.095 * v[33] + -0.05 * v[36] + y[118];
            v[37] = cos(x[5]);
            v[31] = sin(x[5]);
            v[40] = 4.89663865010925e-12 * v[31];
            v[41] = v[38] * v[37] + v[34] * v[40] + v[32] * v[31];
            v[42] = -v[31];
            v[43] = 4.89663865010925e-12 * v[37];
            v[38] = v[38] * v[42] + v[34] * v[43] + v[32] * v[37];
            y[120] = 0.08 * v[41] + -0.01 * v[38] + y[116];
            v[44] = v[39] * v[37] + v[35] * v[40] + v[30] * v[31];
            v[39] = v[39] * v[42] + v[35] * v[43] + v[30] * v[37];
            y[121] = 0.08 * v[44] + -0.01 * v[39] + y[117];
            v[40] = v[29] * v[37] + v[33] * v[40] + v[36] * v[31];
            v[43] = v[29] * v[42] + v[33] * v[43] + v[36] * v[37];
            y[122] = 0.08 * v[40] + -0.01 * v[43] + y[118];
            y[124] = 0.08 * v[41] + 0.035 * v[38] + y[116];
            y[125] = 0.08 * v[44] + 0.035 * v[39] + y[117];
            y[126] = 0.08 * v[40] + 0.035 * v[43] + y[118];
            v[42] = -1. * v[34] + 4.89663865010925e-12 * v[32];
            y[289] = -1. * v[38] + 4.89663865010925e-12 * v[42];
            v[37] = y[116] + 0.088 * v[41];
            y[128] = 0.07 * y[289] + v[37];
            v[29] = -1. * v[35] + 4.89663865010925e-12 * v[30];
            y[290] = -1. * v[39] + 4.89663865010925e-12 * v[29];
            v[31] = y[117] + 0.088 * v[44];
            y[129] = 0.07 * y[290] + v[31];
            v[45] = -1. * v[33] + 4.89663865010925e-12 * v[36];
            y[291] = -1. * v[43] + 4.89663865010925e-12 * v[45];
            v[46] = y[118] + 0.088 * v[40];
            y[130] = 0.07 * y[291] + v[46];
            v[47] = cos(x[6]);
            v[48] = sin(x[6]);
            v[49] = 4.89663865010925e-12 * v[48];
            v[50] = v[41] * v[47] + v[38] * v[49] + v[42] * v[48];
            v[51] = -v[48];
            v[52] = 4.89663865010925e-12 * v[47];
            v[42] = v[41] * v[51] + v[38] * v[52] + v[42] * v[47];
            y[132] = 0.02 * v[50] + 0.04 * v[42] + 0.08 * y[289] + v[37];
            v[53] = v[44] * v[47] + v[39] * v[49] + v[29] * v[48];
            v[29] = v[44] * v[51] + v[39] * v[52] + v[29] * v[47];
            y[133] = 0.02 * v[53] + 0.04 * v[29] + 0.08 * y[290] + v[31];
            v[49] = v[40] * v[47] + v[43] * v[49] + v[45] * v[48];
            v[52] = v[40] * v[51] + v[43] * v[52] + v[45] * v[47];
            y[134] = 0.02 * v[49] + 0.04 * v[52] + 0.08 * y[291] + v[46];
            y[136] = 0.04 * v[50] + 0.02 * v[42] + 0.08 * y[289] + v[37];
            y[137] = 0.04 * v[53] + 0.02 * v[29] + 0.08 * y[290] + v[31];
            y[138] = 0.04 * v[49] + 0.02 * v[52] + 0.08 * y[291] + v[46];
            y[140] = 0.04 * v[50] + 0.06 * v[42] + 0.085 * y[289] + v[37];
            y[141] = 0.04 * v[53] + 0.06 * v[29] + 0.085 * y[290] + v[31];
            y[142] = 0.04 * v[49] + 0.06 * v[52] + 0.085 * y[291] + v[46];
            y[144] = 0.06 * v[50] + 0.04 * v[42] + 0.085 * y[289] + v[37];
            y[145] = 0.06 * v[53] + 0.04 * v[29] + 0.085 * y[290] + v[31];
            y[146] = 0.06 * v[49] + 0.04 * v[52] + 0.085 * y[291] + v[46];
            y[148] = -0.0530330085889673 * v[50] + -0.0530330085890148 * v[42] + 0.117 * y[289] + v[37];
            y[149] = -0.0530330085889673 * v[53] + -0.0530330085890148 * v[29] + 0.117 * y[290] + v[31];
            y[150] = -0.0530330085889673 * v[49] + -0.0530330085890148 * v[52] + 0.117 * y[291] + v[46];
            y[152] = -0.0318198051533804 * v[50] + -0.0318198051534089 * v[42] + 0.117 * y[289] + v[37];
            y[153] = -0.0318198051533804 * v[53] + -0.0318198051534089 * v[29] + 0.117 * y[290] + v[31];
            y[154] = -0.0318198051533804 * v[49] + -0.0318198051534089 * v[52] + 0.117 * y[291] + v[46];
            y[156] = -0.0106066017177935 * v[50] + -0.010606601717803 * v[42] + 0.117 * y[289] + v[37];
            y[157] = -0.0106066017177935 * v[53] + -0.010606601717803 * v[29] + 0.117 * y[290] + v[31];
            y[158] = -0.0106066017177935 * v[49] + -0.010606601717803 * v[52] + 0.117 * y[291] + v[46];
            y[160] = 0.0106066017177935 * v[50] + 0.010606601717803 * v[42] + 0.117 * y[289] + v[37];
            y[161] = 0.0106066017177935 * v[53] + 0.010606601717803 * v[29] + 0.117 * y[290] + v[31];
            y[162] = 0.0106066017177935 * v[49] + 0.010606601717803 * v[52] + 0.117 * y[291] + v[46];
            y[164] = 0.0318198051533804 * v[50] + 0.0318198051534089 * v[42] + 0.117 * y[289] + v[37];
            y[165] = 0.0318198051533804 * v[53] + 0.0318198051534089 * v[29] + 0.117 * y[290] + v[31];
            y[166] = 0.0318198051533804 * v[49] + 0.0318198051534089 * v[52] + 0.117 * y[291] + v[46];
            y[168] = 0.0530330085889673 * v[50] + 0.0530330085890148 * v[42] + 0.117 * y[289] + v[37];
            y[169] = 0.0530330085889673 * v[53] + 0.0530330085890148 * v[29] + 0.117 * y[290] + v[31];
            y[170] = 0.0530330085889673 * v[49] + 0.0530330085890148 * v[52] + 0.117 * y[291] + v[46];
            y[172] = -0.0530330085889673 * v[50] + -0.0530330085890148 * v[42] + 0.137 * y[289] + v[37];
            y[173] = -0.0530330085889673 * v[53] + -0.0530330085890148 * v[29] + 0.137 * y[290] + v[31];
            y[174] = -0.0530330085889673 * v[49] + -0.0530330085890148 * v[52] + 0.137 * y[291] + v[46];
            y[176] = -0.0318198051533804 * v[50] + -0.0318198051534089 * v[42] + 0.137 * y[289] + v[37];
            y[177] = -0.0318198051533804 * v[53] + -0.0318198051534089 * v[29] + 0.137 * y[290] + v[31];
            y[178] = -0.0318198051533804 * v[49] + -0.0318198051534089 * v[52] + 0.137 * y[291] + v[46];
            y[180] = -0.0106066017177935 * v[50] + -0.010606601717803 * v[42] + 0.137 * y[289] + v[37];
            y[181] = -0.0106066017177935 * v[53] + -0.010606601717803 * v[29] + 0.137 * y[290] + v[31];
            y[182] = -0.0106066017177935 * v[49] + -0.010606601717803 * v[52] + 0.137 * y[291] + v[46];
            y[184] = 0.0106066017177935 * v[50] + 0.010606601717803 * v[42] + 0.137 * y[289] + v[37];
            y[185] = 0.0106066017177935 * v[53] + 0.010606601717803 * v[29] + 0.137 * y[290] + v[31];
            y[186] = 0.0106066017177935 * v[49] + 0.010606601717803 * v[52] + 0.137 * y[291] + v[46];
            y[188] = 0.0318198051533804 * v[50] + 0.0318198051534089 * v[42] + 0.137 * y[289] + v[37];
            y[189] = 0.0318198051533804 * v[53] + 0.0318198051534089 * v[29] + 0.137 * y[290] + v[31];
            y[190] = 0.0318198051533804 * v[49] + 0.0318198051534089 * v[52] + 0.137 * y[291] + v[46];
            y[192] = 0.0530330085889673 * v[50] + 0.0530330085890148 * v[42] + 0.137 * y[289] + v[37];
            y[193] = 0.0530330085889673 * v[53] + 0.0530330085890148 * v[29] + 0.137 * y[290] + v[31];
            y[194] = 0.0530330085889673 * v[49] + 0.0530330085890148 * v[52] + 0.137 * y[291] + v[46];
            y[196] = -0.0530330085889673 * v[50] + -0.0530330085890148 * v[42] + 0.157 * y[289] + v[37];
            y[197] = -0.0530330085889673 * v[53] + -0.0530330085890148 * v[29] + 0.157 * y[290] + v[31];
            y[198] = -0.0530330085889673 * v[49] + -0.0530330085890148 * v[52] + 0.157 * y[291] + v[46];
            y[200] = -0.0318198051533804 * v[50] + -0.0318198051534089 * v[42] + 0.157 * y[289] + v[37];
            y[201] = -0.0318198051533804 * v[53] + -0.0318198051534089 * v[29] + 0.157 * y[290] + v[31];
            y[202] = -0.0318198051533804 * v[49] + -0.0318198051534089 * v[52] + 0.157 * y[291] + v[46];
            y[204] = -0.0106066017177935 * v[50] + -0.010606601717803 * v[42] + 0.157 * y[289] + v[37];
            y[205] = -0.0106066017177935 * v[53] + -0.010606601717803 * v[29] + 0.157 * y[290] + v[31];
            y[206] = -0.0106066017177935 * v[49] + -0.010606601717803 * v[52] + 0.157 * y[291] + v[46];
            y[208] = 0.0106066017177935 * v[50] + 0.010606601717803 * v[42] + 0.157 * y[289] + v[37];
            y[209] = 0.0106066017177935 * v[53] + 0.010606601717803 * v[29] + 0.157 * y[290] + v[31];
            y[210] = 0.0106066017177935 * v[49] + 0.010606601717803 * v[52] + 0.157 * y[291] + v[46];
            y[212] = 0.0318198051533804 * v[50] + 0.0318198051534089 * v[42] + 0.157 * y[289] + v[37];
            y[213] = 0.0318198051533804 * v[53] + 0.0318198051534089 * v[29] + 0.157 * y[290] + v[31];
            y[214] = 0.0318198051533804 * v[49] + 0.0318198051534089 * v[52] + 0.157 * y[291] + v[46];
            y[216] = 0.0530330085889673 * v[50] + 0.0530330085890148 * v[42] + 0.157 * y[289] + v[37];
            y[217] = 0.0530330085889673 * v[53] + 0.0530330085890148 * v[29] + 0.157 * y[290] + v[31];
            y[218] = 0.0530330085889673 * v[49] + 0.0530330085890148 * v[52] + 0.157 * y[291] + v[46];
            y[220] = 0.0565685424948984 * v[50] + 0.0565685424949492 * v[42] + 0.1874 * y[289] + v[37];
            y[221] = 0.0565685424948984 * v[53] + 0.0565685424949492 * v[29] + 0.1874 * y[290] + v[31];
            y[222] = 0.0565685424948984 * v[49] + 0.0565685424949492 * v[52] + 0.1874 * y[291] + v[46];
            y[224] = 0.0516187950265948 * v[50] + 0.0516187950266411 * v[42] + 0.2094 * y[289] + v[37];
            y[225] = 0.0516187950265948 * v[53] + 0.0516187950266411 * v[29] + 0.2094 * y[290] + v[31];
            y[226] = 0.0516187950265948 * v[49] + 0.0516187950266411 * v[52] + 0.2094 * y[291] + v[46];
            y[228] = -0.0565685424948984 * v[50] + -0.0565685424949492 * v[42] + 0.1874 * y[289] + v[37];
            y[229] = -0.0565685424948984 * v[53] + -0.0565685424949492 * v[29] + 0.1874 * y[290] + v[31];
            y[230] = -0.0565685424948984 * v[49] + -0.0565685424949492 * v[52] + 0.1874 * y[291] + v[46];
            y[232] = -0.0516187950265948 * v[50] + -0.0516187950266411 * v[42] + 0.2094 * y[289] + v[37];
            y[233] = -0.0516187950265948 * v[53] + -0.0516187950266411 * v[29] + 0.2094 * y[290] + v[31];
            y[234] = -0.0516187950265948 * v[49] + -0.0516187950266411 * v[52] + 0.2094 * y[291] + v[46];
            y[240] = -0.0399999991059303 * v[1];
            y[241] = -0.0399999991059303 * v[2];
            y[244] = -0.0850000008940697 * v[7] + 0.0399999991059303 * v[1];
            y[245] = -0.0850000008940697 * v[6] + 0.0399999991059303 * v[2];
            y[246] = 0.333 + 1.95865541626434e-13 + -0.0850000008940697 * v[4];
            y[248] =
                0.0385857857763767 * v[19] + 0.0289393402636051 * v[15] + -0.0517677664756775 * v[8] + v[9];
            y[249] =
                0.0385857857763767 * v[5] + 0.0289393402636051 * v[14] + -0.0517677664756775 * v[10] + v[11];
            y[250] =
                0.0385857857763767 * v[18] + 0.0289393402636051 * v[3] + -0.0517677664756775 * v[12] + v[13];
            y[252] =
                -0.041449997574091 * v[17] + 0.0492218732833862 * v[23] + 0.0289125014096498 * v[21] + v[24];
            y[253] =
                -0.041449997574091 * v[25] + 0.0492218732833862 * v[26] + 0.0289125014096498 * v[20] + v[27];
            y[254] =
                -0.041449997574091 * v[16] + 0.0492218732833862 * v[22] + 0.0289125014096498 * v[28] + v[0];
            y[256] = 0.0375000014901161 * v[34] + -0.109999999403954 * v[32] + y[116];
            y[257] = 0.0375000014901161 * v[35] + -0.109999999403954 * v[30] + y[117];
            y[258] = 0.0375000014901161 * v[33] + -0.109999999403954 * v[36] + y[118];
            y[260] = 0.0424309633672237 * v[41] + 0.0144476993009448 * v[38] + y[116];
            y[261] = 0.0424309633672237 * v[44] + 0.0144476993009448 * v[39] + y[117];
            y[262] = 0.0424309633672237 * v[40] + 0.0144476993009448 * v[43] + y[118];
            y[264] =
                0.0153005504980683 * v[50] + 0.0153005504980683 * v[42] + 0.0745901614427567 * y[289] + v[37];
            y[265] =
                0.0153005504980683 * v[53] + 0.0153005504980683 * v[29] + 0.0745901614427567 * y[290] + v[31];
            y[266] =
                0.0153005504980683 * v[49] + 0.0153005504980683 * v[52] + 0.0745901614427567 * y[291] + v[46];
            y[268] = -5.20417042793042e-18 * v[50] + -2.16840434497101e-18 * v[42] +
                     0.129201397299767 * y[289] + v[37];
            y[269] = -5.20417042793042e-18 * v[53] + -2.16840434497101e-18 * v[29] +
                     0.129201397299767 * y[290] + v[31];
            y[270] = -5.20417042793042e-18 * v[49] + -2.16840434497101e-18 * v[52] +
                     0.129201397299767 * y[291] + v[46];
            y[272] =
                0.0540936700999737 * v[50] + 0.0540936700999737 * v[42] + 0.198400005698204 * y[289] + v[37];
            y[273] =
                0.0540936700999737 * v[53] + 0.0540936700999737 * v[29] + 0.198400005698204 * y[290] + v[31];
            y[274] =
                0.0540936700999737 * v[49] + 0.0540936700999737 * v[52] + 0.198400005698204 * y[291] + v[46];
            y[276] = -0.0540936700999737 * v[50] + -0.0540936700999737 * v[42] + 0.198400005698204 * y[289] +
                     v[37];
            y[277] = -0.0540936700999737 * v[53] + -0.0540936700999737 * v[29] + 0.198400005698204 * y[290] +
                     v[31];
            y[278] = -0.0540936700999737 * v[49] + -0.0540936700999737 * v[52] + 0.198400005698204 * y[291] +
                     v[46];
            y[280] = v[37] + 0.212 * y[289];
            y[281] = v[31] + 0.212 * y[290];
            y[282] = v[46] + 0.212 * y[291];
            y[283] = 0.707106781186865 * v[50] + -0.70710678118623 * v[42];
            y[284] = 0.707106781186865 * v[53] + -0.70710678118623 * v[29];
            y[285] = 0.707106781186865 * v[49] + -0.70710678118623 * v[52];
            y[286] = 0.70710678118623 * v[50] + 0.707106781186865 * v[42];
            y[287] = 0.70710678118623 * v[53] + 0.707106781186865 * v[29];
            y[288] = 0.70710678118623 * v[49] + 0.707106781186865 * v[52];
            // dependent variables without operations
            y[0] = 0.;
            y[1] = 0.;
            y[2] = 0.05;
            y[3] = 0.0799999982118607;
            y[6] = 0.333;
            y[7] = 0.0599999986588955;
            y[10] = 0.333;
            y[11] = 0.0599999986588955;
            y[12] = 0.;
            y[13] = 0.;
            y[14] = 0.213;
            y[15] = 0.0599999986588955;
            y[16] = 0.;
            y[17] = 0.;
            y[18] = 0.163;
            y[19] = 0.0599999986588955;
            y[22] = 0.333000000000147;
            y[23] = 0.0599999986588955;
            y[26] = 0.333000000000392;
            y[27] = 0.0599999986588955;
            y[31] = 0.0599999986588955;
            y[35] = 0.0599999986588955;
            y[39] = 0.0599999986588955;
            y[43] = 0.0500000007450581;
            y[47] = 0.0549999997019768;
            y[51] = 0.0549999997019768;
            y[55] = 0.0599999986588955;
            y[59] = 0.0549999997019768;
            y[63] = 0.0549999997019768;
            y[67] = 0.0549999997019768;
            y[71] = 0.0599999986588955;
            y[75] = 0.0599999986588955;
            y[79] = 0.0599999986588955;
            y[83] = 0.0500000007450581;
            y[87] = 0.025000000372529;
            y[91] = 0.025000000372529;
            y[95] = 0.025000000372529;
            y[99] = 0.025000000372529;
            y[103] = 0.025000000372529;
            y[107] = 0.025000000372529;
            y[111] = 0.025000000372529;
            y[115] = 0.025000000372529;
            y[119] = 0.0500000007450581;
            y[123] = 0.0500000007450581;
            y[127] = 0.0520000010728836;
            y[131] = 0.0500000007450581;
            y[135] = 0.025000000372529;
            y[139] = 0.025000000372529;
            y[143] = 0.0199999995529652;
            y[147] = 0.0199999995529652;
            y[151] = 0.0280000008642673;
            y[155] = 0.0280000008642673;
            y[159] = 0.0280000008642673;
            y[163] = 0.0280000008642673;
            y[167] = 0.0280000008642673;
            y[171] = 0.0280000008642673;
            y[175] = 0.0260000005364418;
            y[179] = 0.0260000005364418;
            y[183] = 0.0260000005364418;
            y[187] = 0.0260000005364418;
            y[191] = 0.0260000005364418;
            y[195] = 0.0260000005364418;
            y[199] = 0.0240000002086163;
            y[203] = 0.0240000002086163;
            y[207] = 0.0240000002086163;
            y[211] = 0.0240000002086163;
            y[215] = 0.0240000002086163;
            y[219] = 0.0240000002086163;
            y[223] = 0.0120000001043081;
            y[227] = 0.0120000001043081;
            y[231] = 0.0120000001043081;
            y[235] = 0.0120000001043081;
            y[236] = 0.;
            y[237] = 0.;
            y[238] = 0.0500000007450581;
            y[239] = 0.0799999982118607;
            y[242] = 0.24799999910593;
            y[243] = 0.153941467404366;
            y[247] = 0.153941467404366;
            y[251] = 0.128210678696632;
            y[255] = 0.126465573906898;
            y[259] = 0.176216393709183;
            y[263] = 0.0948232412338257;
            y[267] = 0.0721197500824928;
            y[271] = 0.103986009955406;
            y[275] = 0.0235433969646692;
            y[279] = 0.0235433969646692;

            //
            // environment vs. robot collisions
            //

            // panda_rightfinger
            if (sphere_environment_in_collision(environment, y[276], y[277], y[278], y[279]))
            {
                if (sphere_environment_in_collision(environment, y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_leftfinger
            if (sphere_environment_in_collision(environment, y[272], y[273], y[274], y[275]))
            {
                if (sphere_environment_in_collision(environment, y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_hand
            if (sphere_environment_in_collision(environment, y[268], y[269], y[270], y[271]))
            {
                if (sphere_environment_in_collision(environment, y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link7
            if (sphere_environment_in_collision(environment, y[264], y[265], y[266], y[267]))
            {
                if (sphere_environment_in_collision(environment, y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link6
            if (sphere_environment_in_collision(environment, y[260], y[261], y[262], y[263]))
            {
                if (sphere_environment_in_collision(environment, y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link5
            if (sphere_environment_in_collision(environment, y[256], y[257], y[258], y[259]))
            {
                if (sphere_environment_in_collision(environment, y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link4
            if (sphere_environment_in_collision(environment, y[252], y[253], y[254], y[255]))
            {
                if (sphere_environment_in_collision(environment, y[52], y[53], y[54], y[55]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[56], y[57], y[58], y[59]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[60], y[61], y[62], y[63]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[64], y[65], y[66], y[67]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link3
            if (sphere_environment_in_collision(environment, y[248], y[249], y[250], y[251]))
            {
                if (sphere_environment_in_collision(environment, y[36], y[37], y[38], y[39]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[40], y[41], y[42], y[43]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[44], y[45], y[46], y[47]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[48], y[49], y[50], y[51]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link2
            if (sphere_environment_in_collision(environment, y[244], y[245], y[246], y[247]))
            {
                if (sphere_environment_in_collision(environment, y[20], y[21], y[22], y[23]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[24], y[25], y[26], y[27]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[32], y[33], y[34], y[35]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link1
            if (sphere_environment_in_collision(environment, y[240], y[241], y[242], y[243]))
            {
                if (sphere_environment_in_collision(environment, y[4], y[5], y[6], y[7]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[8], y[9], y[10], y[11]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[12], y[13], y[14], y[15]))
                {
                    return false;
                }

                if (sphere_environment_in_collision(environment, y[16], y[17], y[18], y[19]))
                {
                    return false;
                }
            }

            //
            // environment vs. robot collisions
            //

            // panda_link0
            if (sphere_environment_in_collision(environment, y[236], y[237], y[238], y[239]))
            {
                if (sphere_environment_in_collision(environment, y[0], y[1], y[2], y[3]))
                {
                    return false;
                }
            }

            //
            // robot self-collisions
            //

            // panda_link0 vs. panda_link5
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[256], y[257], y[258], y[259]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // panda_link0 vs. panda_link6
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[260], y[261], y[262], y[263]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            // panda_link0 vs. panda_link7
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[264], y[265], y[266], y[267]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // panda_link0 vs. panda_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[268], y[269], y[270], y[271]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // panda_link0 vs. panda_leftfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[272], y[273], y[274], y[275]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // panda_link0 vs. panda_rightfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[236], y[237], y[238], y[239], y[276], y[277], y[278], y[279]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[0], y[1], y[2], y[3], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_link5
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[256], y[257], y[258], y[259]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_link6
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[260], y[261], y[262], y[263]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[116], y[117], y[118], y[119]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[120], y[121], y[122], y[123]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[124], y[125], y[126], y[127]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_link7
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[264], y[265], y[266], y[267]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[268], y[269], y[270], y[271]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_leftfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[272], y[273], y[274], y[275]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // panda_link1 vs. panda_rightfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[240], y[241], y[242], y[243], y[276], y[277], y[278], y[279]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[4], y[5], y[6], y[7], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[8], y[9], y[10], y[11], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[12], y[13], y[14], y[15], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[16], y[17], y[18], y[19], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // panda_link2 vs. panda_link5
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[256], y[257], y[258], y[259]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            // panda_link2 vs. panda_link7
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[264], y[265], y[266], y[267]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // panda_link2 vs. panda_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[268], y[269], y[270], y[271]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // panda_link2 vs. panda_leftfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[272], y[273], y[274], y[275]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // panda_link2 vs. panda_rightfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[244], y[245], y[246], y[247], y[276], y[277], y[278], y[279]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[20], y[21], y[22], y[23], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[24], y[25], y[26], y[27], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[28], y[29], y[30], y[31], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[32], y[33], y[34], y[35], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // panda_link5 vs. panda_link7
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[264], y[265], y[266], y[267]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[128], y[129], y[130], y[131]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[132], y[133], y[134], y[135]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[136], y[137], y[138], y[139]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[140], y[141], y[142], y[143]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[144], y[145], y[146], y[147]))
                {
                    return false;
                }
            }

            // panda_link5 vs. panda_hand
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[268], y[269], y[270], y[271]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[148], y[149], y[150], y[151]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[152], y[153], y[154], y[155]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[156], y[157], y[158], y[159]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[160], y[161], y[162], y[163]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[164], y[165], y[166], y[167]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[168], y[169], y[170], y[171]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[172], y[173], y[174], y[175]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[176], y[177], y[178], y[179]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[180], y[181], y[182], y[183]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[184], y[185], y[186], y[187]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[188], y[189], y[190], y[191]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[192], y[193], y[194], y[195]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[196], y[197], y[198], y[199]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[200], y[201], y[202], y[203]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[204], y[205], y[206], y[207]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[208], y[209], y[210], y[211]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[212], y[213], y[214], y[215]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[216], y[217], y[218], y[219]))
                {
                    return false;
                }
            }

            // panda_link5 vs. panda_leftfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[272], y[273], y[274], y[275]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[220], y[221], y[222], y[223]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[224], y[225], y[226], y[227]))
                {
                    return false;
                }
            }

            // panda_link5 vs. panda_rightfinger
            if (sphere_sphere_self_collision<decltype(x[0])>(
                    y[256], y[257], y[258], y[259], y[276], y[277], y[278], y[279]))
            {
                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[68], y[69], y[70], y[71], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[72], y[73], y[74], y[75], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[76], y[77], y[78], y[79], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[80], y[81], y[82], y[83], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[84], y[85], y[86], y[87], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[88], y[89], y[90], y[91], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[92], y[93], y[94], y[95], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[96], y[97], y[98], y[99], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[100], y[101], y[102], y[103], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[104], y[105], y[106], y[107], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[108], y[109], y[110], y[111], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[228], y[229], y[230], y[231]))
                {
                    return false;
                }

                if (sphere_sphere_self_collision<decltype(x[0])>(
                        y[112], y[113], y[114], y[115], y[232], y[233], y[234], y[235]))
                {
                    return false;
                }
            }

            // attaching at panda_grasptarget
            set_attachment_pose(environment, to_isometry(&y[280]));

            //
            // attachment vs. environment collisions
            //
            if (attachment_environment_collision(environment))
            {
                return false;
            }

            //
            // attachment vs. robot collisions
            //

            // Attachment vs. panda_link0
            if (attachment_sphere_collision<decltype(x[0])>(environment, y[236], y[237], y[238], y[239]))
            {
                if (attachment_sphere_collision<decltype(x[0])>(environment, y[0], y[1], y[2], y[3]))
                {
                    return false;
                }
            }

            // Attachment vs. panda_link1
            if (attachment_sphere_collision<decltype(x[0])>(environment, y[240], y[241], y[242], y[243]))
            {
                if (attachment_sphere_collision<decltype(x[0])>(environment, y[4], y[5], y[6], y[7]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[8], y[9], y[10], y[11]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[12], y[13], y[14], y[15]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[16], y[17], y[18], y[19]))
                {
                    return false;
                }
            }

            // Attachment vs. panda_link2
            if (attachment_sphere_collision<decltype(x[0])>(environment, y[244], y[245], y[246], y[247]))
            {
                if (attachment_sphere_collision<decltype(x[0])>(environment, y[20], y[21], y[22], y[23]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[24], y[25], y[26], y[27]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[28], y[29], y[30], y[31]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[32], y[33], y[34], y[35]))
                {
                    return false;
                }
            }

            // Attachment vs. panda_link5
            if (attachment_sphere_collision<decltype(x[0])>(environment, y[256], y[257], y[258], y[259]))
            {
                if (attachment_sphere_collision<decltype(x[0])>(environment, y[68], y[69], y[70], y[71]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[72], y[73], y[74], y[75]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[76], y[77], y[78], y[79]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[80], y[81], y[82], y[83]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[84], y[85], y[86], y[87]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[88], y[89], y[90], y[91]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[92], y[93], y[94], y[95]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[96], y[97], y[98], y[99]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[100], y[101], y[102], y[103]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[104], y[105], y[106], y[107]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[108], y[109], y[110], y[111]))
                {
                    return false;
                }

                if (attachment_sphere_collision<decltype(x[0])>(environment, y[112], y[113], y[114], y[115]))
                {
                    return false;
                }
            }

            return true;
        }

        inline static auto eefk(const std::array<float, 7> &x) noexcept -> Eigen::Isometry3f
        {
            std::array<float, 36> v;
            std::array<float, 12> y;

            v[0] = cos(x[0]);
            v[1] = sin(x[1]);
            v[2] = -v[1];
            v[3] = sin(x[0]);
            v[4] = -v[3];
            v[5] = cos(x[1]);
            v[6] = 4.89663865010925e-12 * v[5];
            v[7] = v[0] * v[2] + v[4] * v[6];
            v[8] = 4.89663865010925e-12 * v[1];
            v[9] = v[0] * v[5] + v[4] * v[8];
            v[10] = cos(x[2]);
            v[11] = sin(x[2]);
            v[12] = 4.89663865010925e-12 * v[11];
            v[13] = v[9] * v[10] + v[7] * v[12] + v[4] * v[11];
            v[14] = cos(x[3]);
            v[15] = -v[11];
            v[16] = 4.89663865010925e-12 * v[10];
            v[9] = v[9] * v[15] + v[7] * v[16] + v[4] * v[10];
            v[17] = sin(x[3]);
            v[18] = 4.89663865010925e-12 * v[17];
            v[4] = -1. * v[7] + 4.89663865010925e-12 * v[4];
            v[19] = v[13] * v[14] + v[9] * v[18] + v[4] * v[17];
            v[20] = -v[17];
            v[21] = 4.89663865010925e-12 * v[14];
            v[22] = v[13] * v[20] + v[9] * v[21] + v[4] * v[14];
            v[23] = cos(x[4]);
            v[24] = sin(x[4]);
            v[25] = 4.89663865010925e-12 * v[24];
            v[4] = -1. * v[9] + 4.89663865010925e-12 * v[4];
            v[9] = -1. * v[24];
            v[26] = v[19] * v[23] + v[22] * v[25] + v[4] * v[9];
            v[27] = cos(x[5]);
            v[24] = -v[24];
            v[28] = 4.89663865010925e-12 * v[23];
            v[29] = -1. * v[23];
            v[30] = v[19] * v[24] + v[22] * v[28] + v[4] * v[29];
            v[31] = sin(x[5]);
            v[32] = 4.89663865010925e-12 * v[31];
            v[4] = v[22] + 4.89663865010925e-12 * v[4];
            v[33] = v[26] * v[27] + v[30] * v[32] + v[4] * v[31];
            v[34] = -v[31];
            v[35] = 4.89663865010925e-12 * v[27];
            v[26] = v[26] * v[34] + v[30] * v[35] + v[4] * v[27];
            v[4] = -1. * v[30] + 4.89663865010925e-12 * v[4];
            y[9] = -1. * v[26] + 4.89663865010925e-12 * v[4];
            y[0] = -0.316 * v[7] + 0.0825 * v[13] + -0.0825 * v[19] + 0.384 * v[22] + 0.088 * v[33] +
                   0.212 * y[9];
            v[6] = v[3] * v[2] + v[0] * v[6];
            v[8] = v[3] * v[5] + v[0] * v[8];
            v[3] = v[8] * v[10] + v[6] * v[12] + v[0] * v[11];
            v[8] = v[8] * v[15] + v[6] * v[16] + v[0] * v[10];
            v[0] = -1. * v[6] + 4.89663865010925e-12 * v[0];
            v[2] = v[3] * v[14] + v[8] * v[18] + v[0] * v[17];
            v[22] = v[3] * v[20] + v[8] * v[21] + v[0] * v[14];
            v[0] = -1. * v[8] + 4.89663865010925e-12 * v[0];
            v[8] = v[2] * v[23] + v[22] * v[25] + v[0] * v[9];
            v[19] = v[2] * v[24] + v[22] * v[28] + v[0] * v[29];
            v[0] = v[22] + 4.89663865010925e-12 * v[0];
            v[13] = v[8] * v[27] + v[19] * v[32] + v[0] * v[31];
            v[8] = v[8] * v[34] + v[19] * v[35] + v[0] * v[27];
            v[0] = -1. * v[19] + 4.89663865010925e-12 * v[0];
            y[10] = -1. * v[8] + 4.89663865010925e-12 * v[0];
            y[1] = -0.316 * v[6] + 0.0825 * v[3] + -0.0825 * v[2] + 0.384 * v[22] + 0.088 * v[13] +
                   0.212 * y[10];
            v[5] = -1. * v[5];
            v[1] = -1. * v[1];
            v[12] = v[1] * v[10] + v[5] * v[12] + 4.89663865010925e-12 * v[11];
            v[1] = v[1] * v[15] + v[5] * v[16] + 4.89663865010925e-12 * v[10];
            v[16] = 2.39770700697438e-23 + -1. * v[5];
            v[18] = v[12] * v[14] + v[1] * v[18] + v[16] * v[17];
            v[21] = v[12] * v[20] + v[1] * v[21] + v[16] * v[14];
            v[16] = -1. * v[1] + 4.89663865010925e-12 * v[16];
            v[9] = v[18] * v[23] + v[21] * v[25] + v[16] * v[9];
            v[29] = v[18] * v[24] + v[21] * v[28] + v[16] * v[29];
            v[16] = v[21] + 4.89663865010925e-12 * v[16];
            v[32] = v[9] * v[27] + v[29] * v[32] + v[16] * v[31];
            v[9] = v[9] * v[34] + v[29] * v[35] + v[16] * v[27];
            v[16] = -1. * v[29] + 4.89663865010925e-12 * v[16];
            y[11] = -1. * v[9] + 4.89663865010925e-12 * v[16];
            y[2] = 0.333 + -0.316 * v[5] + 0.0825 * v[12] + -0.0825 * v[18] + 0.384 * v[21] + 0.088 * v[32] +
                   0.212 * y[11];
            v[21] = cos(x[6]);
            v[18] = sin(x[6]);
            v[12] = 4.89663865010925e-12 * v[18];
            v[5] = v[33] * v[21] + v[26] * v[12] + v[4] * v[18];
            v[29] = -v[18];
            v[35] = 4.89663865010925e-12 * v[21];
            v[4] = v[33] * v[29] + v[26] * v[35] + v[4] * v[21];
            y[3] = 0.707106781186865 * v[5] + -0.70710678118623 * v[4];
            v[26] = v[13] * v[21] + v[8] * v[12] + v[0] * v[18];
            v[0] = v[13] * v[29] + v[8] * v[35] + v[0] * v[21];
            y[4] = 0.707106781186865 * v[26] + -0.70710678118623 * v[0];
            v[12] = v[32] * v[21] + v[9] * v[12] + v[16] * v[18];
            v[35] = v[32] * v[29] + v[9] * v[35] + v[16] * v[21];
            y[5] = 0.707106781186865 * v[12] + -0.70710678118623 * v[35];
            y[6] = 0.70710678118623 * v[5] + 0.707106781186865 * v[4];
            y[7] = 0.70710678118623 * v[26] + 0.707106781186865 * v[0];
            y[8] = 0.70710678118623 * v[12] + 0.707106781186865 * v[35];

            return to_isometry(y.data());
        }
    };
}  // namespace vamp::robots

// NOLINTEND(*-magic-numbers)
