#!/usr/bin/env roseus

(require :unittest "lib/llib/unittest.l")
(ros::roseus "read_parameter") ;; some test depends on this namespace, do not change test-name "read_parameter" of test-parameter.test

(init-unit-test)

(deftest test-get-param ()
  (assert (ros::has-param "~test") "[~test] (ros::has-param) failed")
  (let ((ret (ros::get-param "~test")))
    (ros::ros-info "~test ~A" ret)
    (assert (string= "test_private" (setq as ret)) "[~test] test_private != ~A" as))

  (assert (ros::has-param "test")  "[test] (ros::has-param) failed")
  (let ((ret (ros::get-param "test")))
    (ros::ros-info "test ~A" ret)
    (assert (string= "test_ns" (setq as ret)) "[test] test_ns != ~A" as))

  (assert (ros::has-param "/test") "[/test] (ros::has-param) failed")
  (let ((ret (ros::get-param "/test")))
    (ros::ros-info "/test ~A" ret)
    (assert (string= "test_global" (setq as ret)) "[/test] test_global != ~A" as))

  (assert (ros::has-param "/test_dictionary") "[/test_dictionary] (ros::has-param) failed")
  (let ((ret (ros::get-param "/test_dictionary")))
    (ros::ros-info "/test_dictionary ~A" ret)
    (let ((val (assoc "str_value" ret :test #'string=)))
      (assert val "str_value not found")
      (assert (string= (setq as (cdr val)) "00000") "\"00000\" != ~S" as))
    (let ((val (assoc "int_value" ret :test #'string=)))
      (assert val "int_value not found")
      (assert (= (setq as (cdr val)) 10000) "10000 != ~A" as))
    (let ((val (assoc "dbl_value" ret :test #'string=)))
      (assert val "dbl_value not found")
      (assert (= (setq as (cdr val)) 20000.0) "20000.0 != ~A" as))
    (let ((val (assoc "bool_value" ret :test #'string=)))
      (assert val "bool_value not found")
      (assert (eq (setq as (cdr val)) nil) "nil != ~A" as))
    (let ((val (assoc "list" ret :test #'string=)))
      (assert val "list not found")
      (let ((lst (cdr val)))
        (assert (listp (setq as lst)) "~A is not list" as)
        (assert (string= (setq as (elt lst 0)) "0") "\"0\" != ~S" as)
        (assert (= (setq as (elt lst 1)) 1) "1 != ~A" as)
        (assert (= (setq as (elt lst 2)) 2.0) "2.0 != ~A" as)
        (assert (eq (setq as (elt lst 3)) t) "t != ~A" as)
        ;; check list
        ;; check dictionary
        ))
    (let ((val (assoc "dictionary" ret :test #'string=)))
      (assert val "dictionary not found")
      (let ((lst (cdr val)))
        (assert (listp (setq as lst)) "~A is not list" as)
        (let ((v (assoc "key_str" lst :test #'string=)))
          (let ((lst (cdr v)))
            (assert (string= (setq as lst) "0000") "\"0000\" != ~S" as)))
        (let ((v (assoc "key_int" lst :test #'string=)))
          (let ((lst (cdr v)))
            (assert (= (setq as lst) 1000) "1000 != ~A" as)))
        (let ((v (assoc "key_dbl" lst :test #'string=)))
          (let ((lst (cdr v)))
            (assert (= (setq as lst) 2000.0) "2000.0 != ~A" as)))
        (let ((v (assoc "key_bool" lst :test #'string=)))
          (let ((lst (cdr v)))
            (assert (eq (setq as lst) t) "t != ~A" as)))
        (let ((v (assoc "key_list" lst :test #'string=)))
          (assert (listp (setq as (cdr v))) "~A is not list" as))
        ))
    )
  )

(deftest test-set-param ()
  (assert (null (ros::has-param "~test2")) "[~test2] (ros::has-param) failed")
  (let (ret)
    (ros::set-param "~test2" "test_private2")
    (setq ret (ros::get-param "~test2"))
    (ros::ros-info "~test2 ~A" ret)
    (assert (string= "test_private2" (setq as ret)) "[~test2] test_private2 != ~A" as))

  (assert (null (ros::has-param "test2"))  "[test2] (ros::has-param) failed")
  (let (ret)
    (ros::set-param "test2" "test_ns2")
    (setq ret (ros::get-param "test2"))
    (ros::ros-info "test2 ~A" ret)
    (assert (string= "test_ns2" (setq as ret)) "[test2] test_ns2 != ~A" as))

  (assert (null (ros::has-param "/test2")) "[/test2] (ros::has-param) failed")
  (let (ret)
    (ros::set-param "/test2" "test_global2")
    (setq ret (ros::get-param "/test2"))
    (ros::ros-info "/test2 ~A" ret)
    (assert (string= "test_global2" (setq as ret)) "[/test2] test_global2 != ~A" as))

  (assert (null (ros::has-param "/test_dictionary2")) "[/test_dictionary2] (ros::has-param) failed")
  (let (ret (param '((bool_value . nil) (dbl_value . 20000) (dictionary (key_bool . t) (key_dbl . 2000.0) (key_int . 1000) (key_list "0" 1 2.0 t) (key_str . "0000")) (int_value . 10000) (list "0" 1 2.0 t (nest0 100 200.0 t) ((nest_bool . t) (nest_dbl . 20.0) (nest_int . 10) (nest_str . nest))) (list2 ((nest_bool . t) (nest_dbl . 20.0) (nest_int . 10) (nest_str . nest))) (str_value . "00000"))))
    (ros::ros-info "/test_dictionary2 param ~A" param)
    (ros::set-param "/test_dictionary2" param)
    (setq ret (ros::get-param "/test_dictionary2"))
    (ros::ros-info "/test_dictionary2 ~A" ret)
    (let ((val (assoc "str_value" ret :test #'string=)))
      (assert val "str_value not found")
      (assert (string= (setq as (cdr val)) "00000") "\"00000\" != ~S" as))
    (let ((val (assoc "int_value" ret :test #'string=)))
      (assert val "int_value not found")
      (ros::ros-info "~A" val)
      (ros::ros-info "~A" (car val))
      (ros::ros-info "~A" (cdr val))
      (ros::ros-info "~A" (stringp (cdr val)))
      (ros::ros-info "~A" (numberp (cdr val)))
      (assert (= (setq as (cdr val)) 10000) "10000 != ~A" as))
    (let ((val (assoc "dbl_value" ret :test #'string=)))
      (assert val "dbl_value not found")
      (assert (= (setq as (cdr val)) 20000.0) "20000.0 != ~A" as))
    (let ((val (assoc "bool_value" ret :test #'string=)))
      (assert val "bool_value not found")
      (assert (eq (setq as (cdr val)) nil) "nil != ~A" as))
    (let ((val (assoc "list" ret :test #'string=)))
      (assert val "list not found")
      (let ((lst (cdr val)))
        (assert (listp (setq as lst)) "~A is not list" as)
        (assert (string= (setq as (elt lst 0)) "0") "\"0\" != ~S" as)
        (assert (= (setq as (elt lst 1)) 1) "1 != ~A" as)
        (assert (= (setq as (elt lst 2)) 2.0) "2.0 != ~A" as)
        (assert (eq (setq as (elt lst 3)) t) "t != ~A" as)
        ;; check list
        ;; check dictionary
        ))
    (let ((l1 (cdr (assoc "list" ret :test #'string=)))
          (l2 (cdr (assoc "list2" ret :test #'string=))))
      (assert (equal (nth 5 l1) (nth 0 l2)) "list objects differ"))
    (let ((val (assoc "dictionary" ret :test #'string=)))
      (assert val "dictionary not found")
      (let ((lst (cdr val)))
        (assert (listp (setq as lst)) "~A is not list" as)
        (let ((v (assoc "key_str" lst :test #'string=)))
          (let ((lst (cdr v)))
            (assert (string= (setq as lst) "0000") "\"0000\" != ~S" as)))
        (let ((v (assoc "key_int" lst :test #'string=)))
          (let ((lst (cdr v)))
            (assert (= (setq as lst) 1000) "1000 != ~A" as)))
        (let ((v (assoc "key_dbl" lst :test #'string=)))
          (let ((lst (cdr v)))
            (assert (= (setq as lst) 2000.0) "2000.0 != ~A" as)))
        (let ((v (assoc "key_bool" lst :test #'string=)))
          (let ((lst (cdr v)))
            (assert (eq (setq as lst) t) "t != ~A" as)))
        (let ((v (assoc "key_list" lst :test #'string=)))
          (assert (listp (setq as (cdr v))) "~A is not list" as))
        ))
    )
  )

(deftest test-set-param-recursive ()
  (assert (ros::has-param "/test_dictionary") "[/test_dictionary] (ros::has-param) failed")
  (assert (null (ros::has-param "/test_recursive")) "[/test_recursive] (ros::has-param) failed")
  (let ((test-dict (ros::get-param "/test_dictionary"))
        test-rec)
    (ros::ros-info "/test_dictionary ~S" test-dict)
    (ros::set-param "/test_recursive" test-dict)
    (setq test-rec (ros::get-param "/test_recursive"))
    (ros::ros-info "/test_recursive ~S" test-rec)
    (assert (equal test-dict test-rec) "unable to perform ros::set-param recursively")
    (assert (ros::delete-param "/test_recursive")
            "(ros::delete-param \"test_recursive\") returns nil")))

(deftest test-param-default-value ()
  ;; Read parameter 1000 times with default value
  (dotimes (i 1000)
    (assert (equal "foo" (ros::get-param "bar" "foo")))
    )
  )

(deftest test-delete-param ()
  (dolist (prefix '("" "/" "~"))
    (let ((key (format nil "~Adel_param_test" prefix)))
      (assert (not (ros::has-param key)) "(ros::has-param \"~A\") returns value" key)
      (assert (ros::set-param key "test_param_value") "(ros::set-param \"~A\") returns nil" key)
      (assert (ros::has-param key) "(ros::has-param \"~A\") returns nil" key)
      (assert (ros::delete-param key) "(ros::delete-param \"~A\") returns nil" key)
      (assert (not (ros::has-param key)) "(ros::has-param \"~A\") returns value after delete-param" key)))
  (assert (not (ros::has-param "param_not_exists")) "(ros::has-param param_not_exists) returns value")
  (assert (not (ros::delete-param "param_not_exists")) "(ros::delete-param param_not_exists) returns value even if param does not exists"))

(deftest test-param-cache ()
  (assert (ros::has-param "test")  "[cached] (ros::has-param test) failed")
  (assert (string= (ros::get-param "test") (ros::get-param-cached "test")) "(ros::get-param-cached test) has different value from (ros::get-param test): ~A" (ros::get-param-cached "test"))
  (assert (< (bench2 (dotimes (i 100) (ros::get-param-cached "test")))
             (bench2 (dotimes (i 100) (ros::get-param "test"))))
          "(ros::get-param-cached test) has less performance than (ros::get-param test)"))

;; this should be call after test-set-param
(deftest test-list-param ()
  (let ((ret (ros::list-param)) diff)
    (assert ret "list-param returns nil")
    (setq diff (set-difference (cdr ret)
                               '("/test_dictionary/str_value" "/test_dictionary/dictionary/key_bool" "/test_dictionary/dictionary/key_dbl" "/test_dictionary/dictionary/key_str" "/test_dictionary/dictionary/key_int" "/test_dictionary/dictionary/key_list" "/test_dictionary/list" "/test_dictionary/list2" "/test_dictionary/bool_value" "/test_dictionary/int_value" "/test_dictionary/dbl_value" "/rosversion" "/run_id" "/param/test" "/param/read_parameter/test" "/test" "/rosdistro"
                                 "/test2" "/param/read_parameter/test2" "/param/test2" "/test_dictionary2/str_value" "/test_dictionary2/dictionary/key_bool" "/test_dictionary2/dictionary/key_dbl" "/test_dictionary2/dictionary/key_str" "/test_dictionary2/dictionary/key_int" "/test_dictionary2/dictionary/key_list" "/test_dictionary2/bool_value" "/test_dictionary2/list" "/test_dictionary2/list2" "/test_dictionary2/int_value" "/test_dictionary2/dbl_value")
                               :test #'equal))
    (setq diff (remove-if #'(lambda (x) (substringp "/roslaunch/uris/host_" x)) diff)) ;; from noetic roslaunch set params https://github.com/ros/ros_comm/blob/07fb5469c71e10e4a05fa3a631897d9adece61c5/tools/roslaunch/src/roslaunch/launch.py#L448-L449
    (assert (not diff) "list-param ~A failed" diff)
    ))

(deftest test-search-param ()
  (let (ret)
    ;; search param only look for upper namespace
    (setq ret (ros::search-param "test"))
    (assert (string= ret "/param/read_parameter/test") "search-param returns ~A" ret)
    (setq ret (ros::search-param "rosdistro"))
    (assert (string= ret "/rosdistro")  "search-param returns ~A" ret)
    ))

(run-all-tests)

(exit)
